#pragma warn -waus

#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <windows.h>
#include <stdio.h>

#include "ccdlib.h"
#include "giotto.h"

#ifdef _PROGRESS_
extern long	progresscounter;
#endif

// miscellaneous functions

extern FILE * debug;
extern char debuglog[80];
extern void Debug( char * );

/***************************************************************************
*                                                                          *
*   CCDLIB_CopyRegion:  Routine zum Kopieren von Ausschnitten              *
*                                                                          *
*   Parameter:  * destbuffer  : Zielbild                                   *
*               * sourcebuffer: Quellbuffer                                *
*               * pictureparam: Systemparameter des zu bearbeitenden       *
*                               Bildes                                     *
*               * copyrect    : Auschnittrechteck                          *
*                                                                          *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung      *
*                                                                          *
*               CCDLIB_Successful:            Alles OK !                   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen     *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse          *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.    *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen     *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher            *
*                                                                          *
*   Version:    1.0 (18.05.1996)                                           *
*   Written by: Georg Dittie, 1996                                         *
*                                                                          *
***************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CopyRegion(
  short * destbuffer,
  short * sourcebuffer,
  CCDLIB_PictureparamStruct * picpar,
  CCDLIB_RectangleStruct * copyrect )
{
	long xs, xd, yss, yds, slinestep, dlinestep,
		  xstart, xstop, ystart, ystop;

	if( CCDLIB_CheckRectangleValues( copyrect, picpar ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( sourcebuffer == NULL || destbuffer == NULL )
		return( CCDLIB_NoPointerError );

	xstart = copyrect->xpos;
	ystart = copyrect->ypos;

	xstop = copyrect->xpos + copyrect->xsize;
	ystop = copyrect->ypos + copyrect->ysize;

	ystart *= picpar->linelength;
	ystop  *= picpar->linelength;

	if( picpar->colorchannel != CCDLIB_RGBChannelColor )
	{
		slinestep = picpar->linelength;
		dlinestep = copyrect->xsize;

		yds = 0;
		for( yss = ystart;
			  yss < ystop;
			  yss += slinestep )
		{
			xd = 0;
			for(xs = xstart;
				 xs < xstop;
				 xs ++ )
			{
				destbuffer[ yds + xd ] = sourcebuffer[ yss + xs ];
				xd++;
			}
			yds += dlinestep;
		}
	}
	else
	{
		xstart *= 3;
		ystart *= 3;
      xstop  *= 3;
		ystop  *= 3;
		slinestep = picpar->linelength*3;
		dlinestep = copyrect->xsize*3;

		yds = 0;
		for( yss = ystart;
			  yss < ystop;
			  yss += slinestep )
		{
			xd = 0;
			for(xs = xstart;
				 xs < xstop;
				 xs ++ )
			{
				destbuffer[ yds + xd ] = sourcebuffer[ yss + xs ];
				xd++;
			}
			yds += dlinestep;
		}
	}
	return( CCDLIB_Successful );
}

/***************************************************************************
*                                                                          *
*   CCDLIB_InitRectangle :  Routine zum Setzen von Rechteckwerten          *
*                                                                          *
*   Parameter:  * picturrect: Zu fuellende Struktur                        *
*               pos,size    : Fuellwerte                                   *
*                                                                          *
*   Version:    1.0 (21.09.1997)                                           *
*   Written by: Georg Dittie, 1997                                         *
*                                                                          *
***************************************************************************/

void CCDLIB_InitRectangle(
	CCDLIB_RectangleStruct * picturerect,
	long xpos,
	long ypos,
	long xsize,
	long ysize )
{
	picturerect->xpos  = xpos;
	picturerect->ypos  = ypos;
	picturerect->xsize = xsize;
	picturerect->ysize = ysize;
}

/**************************************************************************
*                                                                         *
*   CCDLIB_FlipMirror: Routine zum Spiegeln oder rechtwinkligen Drehen.   *
*                                                                         *
*   Parameter:  flipmethod:  Arbeitsanweisung                             *
*               *sourcepicparam:  Parameter der Quelle                    *
*               *destpicparam  :  Parameter des Ziels                     *
*               * sourcepic    :  32-Bit-Pointer auf die Bildquelle       *
*               * destpic      :  32-Bit-Pointer auf das Bildziel         *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   ACHTUNG: Source und Ziel duerfen nicht identisch sein !!!             *
*                                                                         *
*   Version:    1.1, (11.04.1999)                         32 Bit          *
*   Written by: Georg Dittie, 1997 - 1999                                 *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_FlipMirror
(
	enum CCDLIB_FlipMethodEnum flipmethod,
	CCDLIB_PictureparamStruct * sourcepicparam,
	CCDLIB_PictureparamStruct * destpicparam,
	short * sourcepic,
	short * destpic )
{
	long xs, xd, ys, yd, xsstop, ysstop,
		  xdstep, ydstep, ydstart, xdstart;

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( sourcepicparam == NULL || destpicparam == NULL )
		return( CCDLIB_NoPointerError );

	if( sourcepic == destpic || sourcepicparam == destpicparam )
		return( CCDLIB_WrongParameterError );

	*destpicparam = *sourcepicparam;

	if( sourcepicparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		xsstop = sourcepicparam->linelength;
		ysstop = sourcepicparam->columnheight*sourcepicparam->linelength;

		switch( flipmethod )
		{
		case CCDLIB_FlipTopdownConst:
			ydstep  = sourcepicparam->linelength;
			ydstart = ysstop-sourcepicparam->linelength;
			for( ys=0, yd=ydstart; ys<ysstop; ys+=ydstep, yd-=ydstep )
				for( xs=0; xs<xsstop; xs++ )
					destpic[ xs+yd ] = sourcepic[ xs+ys ];
			break;
		case CCDLIB_FlipMirrorConst:
			ydstep  = sourcepicparam->linelength;
			xdstart = sourcepicparam->linelength-1;
			for( ys=0; ys<ysstop; ys+=ydstep )
				for( xs=0, xd=xdstart; xs<xsstop; xs++, xd-- )
					destpic[ xd+ys ] = sourcepic[ xs+ys ];
			break;
		case CCDLIB_FlipErectConst:
			ydstep  = sourcepicparam->linelength;
			xdstart = sourcepicparam->linelength-1;
			ydstart = ysstop-sourcepicparam->linelength;
			for( ys=0, yd=ydstart; ys<ysstop; ys+=ydstep, yd-=ydstep )
				for( xs=0, xd=xdstart; xs<xsstop; xs++, xd-- )
					destpic[ xd+yd ] = sourcepic[ xs+ys ];
			break;
		case CCDLIB_FlipRightturnConst:
			xdstep  = sourcepicparam->columnheight;
			ydstep  = sourcepicparam->linelength;
			ydstart = ysstop-sourcepicparam->columnheight;
			for( ys=0, xd=0; ys<ysstop; ys+=ydstep, xd++ )
				for( xs=0, yd=ydstart; xs<xsstop; xs++, yd-=xdstep )
					destpic[ xd+yd ] = sourcepic[ xs+ys ];
			destpicparam->linelength = sourcepicparam->columnheight;
			destpicparam->columnheight = sourcepicparam->linelength;
			break;
		case CCDLIB_FlipLeftturnConst:
			xdstep  = sourcepicparam->columnheight;
			ydstep  = sourcepicparam->linelength;
			xdstart = sourcepicparam->columnheight-1;
			for( ys=0, xd=xdstart; ys<ysstop; ys+=ydstep, xd-- )
				for( xs=0, yd=0; xs<xsstop; xs++, yd+=xdstep )
					destpic[ xd+yd ] = sourcepic[ xs+ys ];
			destpicparam->linelength = sourcepicparam->columnheight;
			destpicparam->columnheight = sourcepicparam->linelength;
			break;
		default:
			return( CCDLIB_WrongParameterError );
		}
	}
	else
	{
		xsstop = sourcepicparam->linelength*3;
		ysstop = sourcepicparam->columnheight*sourcepicparam->linelength*3;

		switch( flipmethod )
		{
		case CCDLIB_FlipTopdownConst:
			ydstep  = xsstop;
			ydstart = ysstop-xsstop;
			for( ys=0, yd=ydstart; ys<ysstop; ys+=ydstep, yd-=ydstep )
				for( xs=0; xs<xsstop; xs+=3 )
				{
					destpic[ xs+yd ]   = sourcepic[ xs+ys ];
					destpic[ xs+yd+1 ] = sourcepic[ xs+ys+1 ];
					destpic[ xs+yd+2 ] = sourcepic[ xs+ys+2 ];
				}
			break;
		case CCDLIB_FlipMirrorConst:
			ydstep  = xsstop;
			xdstart = xsstop-3;
			for( ys=0; ys<ysstop; ys+=ydstep )
				for( xs=0, xd=xdstart; xs<xsstop; xs+=3, xd-=3 )
				{
					destpic[ xd+ys ]   = sourcepic[ xs+ys ];
					destpic[ xd+ys+1 ] = sourcepic[ xs+ys+1 ];
					destpic[ xd+ys+2 ] = sourcepic[ xs+ys+2 ];
				}
			break;
		case CCDLIB_FlipErectConst:
			ydstep  = xsstop;
			xdstart = xsstop-3;
			ydstart = ysstop-xsstop;
			for( ys=0, yd=ydstart; ys<ysstop; ys+=ydstep, yd-=ydstep )
				for( xs=0, xd=xdstart; xs<xsstop; xs+=3, xd-=3 )
				{
					destpic[ xd+yd ]   = sourcepic[ xs+ys ];
					destpic[ xd+yd+1 ] = sourcepic[ xs+ys+1 ];
					destpic[ xd+yd+2 ] = sourcepic[ xs+ys+2 ];
				}
			break;
		case CCDLIB_FlipRightturnConst:
			xdstep  = (sourcepicparam->columnheight)*3;
			ydstep  = (sourcepicparam->linelength)*3;
			ydstart = ysstop-sourcepicparam->columnheight*3;
			for( ys=0, xd=0; ys<ysstop; ys+=ydstep, xd+=3 )
				for( xs=0, yd=ydstart; xs<xsstop; xs+=3, yd-=xdstep )
				{
					destpic[ xd+yd ]   = sourcepic[ xs+ys ];
					destpic[ xd+yd+1 ] = sourcepic[ xs+ys+1 ];
					destpic[ xd+yd+2 ] = sourcepic[ xs+ys+2 ];
				}
			destpicparam->linelength = sourcepicparam->columnheight;
			destpicparam->columnheight = sourcepicparam->linelength;
			break;
		case CCDLIB_FlipLeftturnConst:
			xdstep  = (sourcepicparam->columnheight)*3;
			ydstep  = (sourcepicparam->linelength)*3;
			xdstart = (sourcepicparam->columnheight-1)*3;
			for( ys=0, xd=xdstart; ys<ysstop; ys+=ydstep, xd-=3 )
				for( xs=0, yd=0; xs<xsstop; xs+=3, yd+=xdstep )
				{
					destpic[ xd+yd ]   = sourcepic[ xs+ys ];
					destpic[ xd+yd+1 ] = sourcepic[ xs+ys+1 ];
					destpic[ xd+yd+2 ] = sourcepic[ xs+ys+2 ];
				}
			destpicparam->linelength = sourcepicparam->columnheight;
			destpicparam->columnheight = sourcepicparam->linelength;
			break;
		default:
			return( CCDLIB_WrongParameterError );
		}
	}

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_AddPictures : Routine zum anteilmaessigen Aufaddieren von    *
*                        Bildern. Als Resultat entsteht die Summe aller *
*                        bisher aufaddierten Bilder. Hier kann in ein   *
*                        konstantes oder mitwachsendes Bildfeld addiert *
*                        werden.                                        *
*																								*
*	 ACHTUNG: Addpuffer, Addcount und Picparam werden bei jedem Schritt  *
*            geaendert, wenn die Additionsflaeche mitwaechst ! Die Be-  *
*            zugskoordinate bezieht sich immer auf den aktuellen Addi-  *
*            tionspuffer !                                              *
*   																							* 
*   Hinweis: Pixel mit negativen Grauwerten werden nicht mitaddiert !   *
*                                                                       *
*   Parameter:  * sourcepicrect: Arbeitsbereiches im Quellbild          *
*               * destpicpos  :  Position der oberen/linken Zielecke    *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picpuffer   :  32-Bit-Pointer auf das aktuelle Bild   *
*               **addpuffer   :	32-Bit-Pointer auf die Grauwertsumme   *
*               **addcount    :	32-Bit-Ptr auf Anzahl der add. Pixel   *
*               piccount      :  Anzahl der bisher aufaddierten Bilder  *
*               constarea     :  FALSE = mitwachsendes Bildfeld         *
*                                TRUE: Anfangsbildfeld bleibt konstant  *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    5.0  2002)                                  *
*   Written by: Georg Dittie, 1996-2002                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_AddPictures
(
	CCDLIB_RectangleStruct * sourcepicrect,
   CCDLIB_PositionStruct  * destpicpos,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_PictureparamStruct * addparam,
	long ** addpuffer,
	long ** addcount,
	long piccount,
	BOOL constarea,
   enum CCDLIB_AddImagetypeEnum imagetype
)
{
	long greyval, greyflag, xs, xd, xc, ys, yd, yc,
		  xsstart, ysstart, xstop, ystop,
		  xdstart, ydstart, ysstep, ydstep,
		  istop, cstop, oldlinelength, oldcolumnheight,
        leftlimit, toplimit, rightlimit, bottomlimit, dynamikm1;
	long *newaddpuffer, *newaddcount;

	if( piccount < 0 )
		return( CCDLIB_WrongParameterError );

	if( picture == NULL )
		return( CCDLIB_NoPointerError );

   dynamikm1 = pictureparam->dynamik-1;

	if( piccount == 0 )
	{
		cstop = pictureparam->linelength * pictureparam->columnheight;
		if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
			istop = cstop * 3L;
		else
			istop = cstop;

      if( *addpuffer != NULL || *addcount != NULL )
   		return( CCDLIB_WrongParameterError );

		*addpuffer = (long *)malloc( istop*sizeof(long) );
		*addcount  = (long *)malloc( cstop*sizeof(long) );
		if( *addpuffer == NULL || *addcount == NULL )
		{
			if( *addpuffer != NULL )
			{
				free( *addpuffer );
				*addpuffer = NULL;
			}
			if( *addcount != NULL )
			{
				free( *addcount );
				*addcount = NULL;
			}
			return( CCDLIB_OutOfMemoryError );
		}
		memset( *addpuffer, 0, istop*sizeof(long) );
		memset( *addcount, 0, cstop*sizeof(long) );

		*addparam = *pictureparam;
	}
/*
	else if( piccount != 0 && constarea == FALSE )
	{
		oldlinelength   = addparam->linelength;
		oldcolumnheight = addparam->columnheight;

		if( destpicpos->xpos < 0 )
		{
			addparam->linelength -= destpicpos->xpos;
			xdstart = -destpicpos->xpos;
		}
		else if( destpicpos->xpos+pictureparam->linelength > addparam->linelength )
		{
			addparam->linelength = pictureparam->linelength+destpicpos->xpos;
			xdstart = 0;
		}
		else
			xdstart = 0;
		xstop = xdstart + oldlinelength;

		if( destpicpos->ypos < 0 )
		{
			addparam->columnheight -= destpicpos->ypos;
			ydstart = -destpicpos->ypos;
		}
		else if( destpicpos->ypos+pictureparam->columnheight > addparam->columnheight )
		{
			addparam->columnheight = pictureparam->columnheight+destpicpos->ypos;
			ydstart = 0;
		}
		else
			ydstart = 0;
		ystop = ydstart + oldcolumnheight;

		cstop = addparam->linelength * addparam->columnheight;
		if( addparam->colorchannel == CCDLIB_RGBChannelColor )
			istop = cstop * 3L;
		else
			istop = cstop;

		newaddpuffer = (long *)malloc( istop*sizeof(long) );
		newaddcount  = (long *)malloc( cstop*sizeof(long) );
		if( newaddpuffer == NULL || newaddcount == NULL )
		{
			if( newaddpuffer != NULL )	free( newaddpuffer );
			if( newaddcount != NULL ) free( newaddcount );
			return( CCDLIB_OutOfMemoryError );
		}
		memset( newaddpuffer, 0, istop*sizeof(long) );
		memset( newaddcount, 0, cstop*sizeof(long) );

		ydstart *= addparam->linelength;
		ystop   *= addparam->linelength;

		for( yd = ydstart, ys = 0;
			  yd < ystop;
			  yd += addparam->linelength, ys += oldlinelength )
			for( xd = xdstart, xs = 0;
				  xd < xstop;
				  xd++, xs ++ )
				newaddcount[ yd+xd ] = (*addcount)[ ys+xs ];

		if( addparam->colorchannel == CCDLIB_RGBChannelColor )
		{
			ydstart *= 3; xdstart *= 3; ystop *=3; xstop *= 3;
		}
		for( yd = ydstart, ys = 0;
			  yd < ystop;
			  yd += addparam->linelength, ys += oldlinelength )
			for( xd = xdstart, xs = 0;
				  xd < xstop;
				  xd++, xs ++ )
				newaddpuffer[ yd+xd ] = (*addpuffer)[ ys+xs ];

		free( *addpuffer );
		free( *addcount );
		*addpuffer = newaddpuffer;
		*addcount  = newaddcount;
	}
*/
	xsstart = -destpicpos->xpos;
	ysstart = -destpicpos->ypos*pictureparam->linelength;
	xstop   = addparam->linelength;
	ystop   = addparam->columnheight*addparam->linelength;
   leftlimit   = sourcepicrect->xpos;
   rightlimit  = sourcepicrect->xpos+sourcepicrect->xsize;
   toplimit    = (sourcepicrect->ypos)*pictureparam->linelength;
   bottomlimit = (sourcepicrect->ypos+sourcepicrect->ysize)*pictureparam->linelength;

   switch( imagetype )
   {
   case CCDLIB_AddAverage:
	   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	   {
	      xsstart *= 3;
	      ysstart *= 3;
	      xstop   *= 3;
	      ystop   *= 3;
         leftlimit   *= 3;
         rightlimit  *= 3;
         toplimit    *= 3;
         bottomlimit *= 3;
		   ydstep = addparam->linelength*3;
		   ysstep = pictureparam->linelength*3;

		   for( yd = 0, ys = ysstart, yc = 0;
			     yd < ystop;
			     yd += ydstep, ys += ysstep, yc += addparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart, xc = 0;
				     xd < xstop;
				     xd += 3, xs += 3, xc++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyflag = 0;
				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd ] += greyval;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 1 ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd + 1 ] += greyval;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 2 ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd + 2 ] += greyval;
					   greyflag++;
               }
				   if( greyflag > 0 )
					   (*addcount)[ yc + xc ]++;
			   }
		   }
	   }
	   else
	   {
		   for( yd = 0, ys = ysstart;
			     yd < ystop;
			     yd += addparam->linelength, ys += pictureparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart;
				     xd < xstop;
				     xd ++, xs ++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 )
				   {
					   (*addpuffer)[ yd + xd ] += greyval;
					   (*addcount)[ yd + xd ]++;
				   }
			   }
		   }
      }
      break;

   case CCDLIB_AddAccumulate:
      if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	   {
	      xsstart *= 3;
	      ysstart *= 3;
	      xstop   *= 3;
	      ystop   *= 3;
         leftlimit   *= 3;
         rightlimit  *= 3;
         toplimit    *= 3;
         bottomlimit *= 3;
		   ydstep = addparam->linelength*3;
		   ysstep = pictureparam->linelength*3;

		   for( yd = 0, ys = ysstart, yc = 0;
			     yd < ystop;
			     yd += ydstep, ys += ysstep, yc += addparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart, xc = 0;
				     xd < xstop;
				     xd += 3, xs += 3, xc++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyflag = 0;
				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 && (*addpuffer)[ yd + xd ] < greyval )
               {
					   (*addpuffer)[ yd + xd ] = greyval;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 1 ];
				   if( greyval > 0 && (*addpuffer)[ yd + xd + 1 ] < greyval )
               {
					   (*addpuffer)[ yd + xd + 1 ] = greyval;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 2 ];
				   if( greyval > 0 && (*addpuffer)[ yd + xd + 2 ] < greyval )
               {
					   (*addpuffer)[ yd + xd + 2 ] = greyval;
					   greyflag++;
               }
				   if( greyflag > 0 )
					   (*addcount)[ yc + xc ]=1;
			   }
		   }
	   }
	   else
	   {
		   for( yd = 0, ys = ysstart;
			     yd < ystop;
			     yd += addparam->linelength, ys += pictureparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart;
				     xd < xstop;
				     xd ++, xs ++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 && (*addpuffer)[ yd + xd ] < greyval )
               {
					   (*addpuffer)[ yd + xd ] = greyval;
					   (*addcount)[ yd + xd ]=1;
				   }
			   }
		   }
      }
      break;

   case CCDLIB_AddExposure:
      if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	   {
	      xsstart *= 3;
	      ysstart *= 3;
	      xstop   *= 3;
	      ystop   *= 3;
         leftlimit   *= 3;
         rightlimit  *= 3;
         toplimit    *= 3;
         bottomlimit *= 3;
		   ydstep = addparam->linelength*3;
		   ysstep = pictureparam->linelength*3;

		   for( yd = 0, ys = ysstart, yc = 0;
			     yd < ystop;
			     yd += ydstep, ys += ysstep, yc += addparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart, xc = 0;
				     xd < xstop;
				     xd += 3, xs += 3, xc++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyflag = 0;
				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd ] += greyval;
                  if( (*addpuffer)[ yd + xd ] >= dynamikm1 )
                     (*addpuffer)[ yd + xd ] = dynamikm1;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 1 ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd + 1 ] += greyval;
                  if( (*addpuffer)[ yd + xd + 1 ] >= dynamikm1 )
                     (*addpuffer)[ yd + xd +1 ] = dynamikm1;
					   greyflag++;
               }
				   greyval = (long)picture[ ys + xs + 2 ];
				   if( greyval > 0 )
               {
					   (*addpuffer)[ yd + xd + 2 ] += greyval;
                  if( (*addpuffer)[ yd + xd + 2 ] >= dynamikm1 )
                     (*addpuffer)[ yd + xd + 2 ] = dynamikm1;
					   greyflag++;
               }
				   if( greyflag > 0 )
					   (*addcount)[ yc + xc ]=1;
			   }
		   }
	   }
	   else
	   {
		   for( yd = 0, ys = ysstart;
			     yd < ystop;
			     yd += addparam->linelength, ys += pictureparam->linelength )
		   {
            if( ys < toplimit || ys >= bottomlimit )
               continue;

			   for( xd = 0, xs = xsstart;
				     xd < xstop;
				     xd ++, xs ++ )
			   {
               if( xs < leftlimit || xs >= rightlimit )
                  continue;

				   greyval = (long)picture[ ys + xs ];
				   if( greyval > 0 )
				   {
					   (*addpuffer)[ yd + xd ] += greyval;
                  if( (*addpuffer)[ yd + xd ] >= dynamikm1 )
                     (*addpuffer)[ yd + xd ] = dynamikm1;
					   (*addcount)[ yd + xd ]=1;
				   }
			   }
		   }
      }
      break;
	}

	return( CCDLIB_Successful );
}


/************************************************************************
*                                                                       *
*	 CCDLIB_MoveAddfieldToPicture : Routine zum normierten Uebertragen   *
*               von auffaddierten Bildern. Als Resultat entsteht der    *
*               arithmetische Mittelwert aller aufaddierten Bilder.     *
*																								*
*   Parameter:  * addpicparam :  Systemparameter Additionspuffer        *
*               * destpicparam:  Systemparameter Zielpuffer             *
*               * picpuffer   :  32-Bit-Pointer auf das aktuelle Bild   *
*               * addpuffer   :	32-Bit-Pointer auf die Summe           *
*               * addcount    :  Anzahl der bisher aufaddierten Bilder  *
*               movemethod    :  Original/Ausschnitt/Schrumpfen         *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    3.0, (08.07.2001)                                       *
*   Written by: Georg Dittie, 1994-2001                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_MoveAddfieldToPicture
(
	CCDLIB_PictureparamStruct * addpicparam,
	CCDLIB_PictureparamStruct * destpicparam,
	short * destpic,
	long * addpuffer,
	long * addcount,
   enum CCDLIB_MoveMethodEnum movemethod,
   CCDLIB_RectangleStruct *cutrect
)
{
	long x, xc, ys, ycs, xd, yd, xm, ym, ymstop,
        xastop, yastop, yastep, ydstep, ycstep, count,
        greyval, redval, greenval, blueval, valcnt;


	if( destpic == NULL || addpuffer == NULL || addcount == NULL )
		return( CCDLIB_NoPointerError );

	if( addpicparam->colorchannel == CCDLIB_RGBChannelColor )
	{
      switch( movemethod )
      {
      case CCDLIB_MoveUnchangedConst:
      	xastop = addpicparam->linelength * 3;
	      yastop = addpicparam->columnheight * addpicparam->linelength * 3;
         ycstep = addpicparam->linelength;

		   for(ys = 0, ycs = 0;
			    ys < yastop;
			    ys += xastop, ycs += ycstep )
		   {
			   for(x = 0, xc = 0;
				    x < xastop;
				    x += 3, xc ++ )
			   {
				   count = addcount[ ycs+xc ];
				   if( count > 0 )
				   {
					   destpic[ ys+x   ] = (short)( addpuffer[ ys+x   ] / count );
					   destpic[ ys+x+1 ] = (short)( addpuffer[ ys+x+1 ] / count );
					   destpic[ ys+x+2 ] = (short)( addpuffer[ ys+x+2 ] / count );
				   }
				   else
				   {
					   destpic[ ys+x   ] = destpicparam->bblankvalue.word;
					   destpic[ ys+x+1 ] = destpicparam->gblankvalue.word;
					   destpic[ ys+x+2 ] = destpicparam->blankvalue.word;
				   }
			   }
		   }
         break;
      case CCDLIB_MoveCutoffConst:
         if( cutrect == NULL )
            return( CCDLIB_WrongRectangleValuesError );

      	xastop = (cutrect->xpos+cutrect->xsize) * 3;
	      yastop = (cutrect->ypos+cutrect->ysize) * addpicparam->linelength * 3;
         yastep = addpicparam->linelength * 3;
	      ydstep = destpicparam->linelength * 3;
         ycstep = addpicparam->linelength;

		   for(ys = cutrect->ypos*yastep, yd = 0, ycs = cutrect->ypos*ycstep;
			    ys < yastop;
			    ys += yastep, yd += ydstep, ycs += ycstep )
		   {
			   for(x = cutrect->xpos*3, xd = 0, xc = cutrect->xpos;
				    x < xastop;
				    x += 3, xd += 3, xc ++ )
			   {
				   count = addcount[ ycs+xc ];
				   if( count > 0 )
				   {
					   destpic[ yd+xd   ] = (short)( addpuffer[ ys+x   ] / count );
					   destpic[ yd+xd+1 ] = (short)( addpuffer[ ys+x+1 ] / count );
					   destpic[ yd+xd+2 ] = (short)( addpuffer[ ys+x+2 ] / count );
				   }
				   else
				   {
					   destpic[ yd+xd   ] = destpicparam->bblankvalue.word;
					   destpic[ yd+xd+1 ] = destpicparam->gblankvalue.word;
					   destpic[ yd+xd+2 ] = destpicparam->blankvalue.word;
				   }
			   }
		   }
         break;
      case CCDLIB_MoveShrink2Const:
      	xastop = addpicparam->linelength * 3;
	      yastop = addpicparam->columnheight * addpicparam->linelength * 3;
         yastep = addpicparam->linelength * 3;
         ycstep = addpicparam->linelength;
         ydstep = destpicparam->linelength * 3;

		   for(ys = 0, yd = 0, ycs = 0;
			    ys < yastop;
			    ys += yastep*2, yd += ydstep, ycs += ycstep*2 )
		   {
			   for(x = 0, xd = 0, xc = 0;
				    x < xastop;
				    x += 6, xd += 3, xc += 2 )
			   {
               blueval  = 0;
               greenval = 0;
               redval   = 0;
               valcnt   = 0;

				   count  = addcount[ ycs+xc ];
				   if( count > 0 )
               {
					   blueval  += (addpuffer[ ys+x   ] / count);
					   greenval += (addpuffer[ ys+x+1 ] / count);
					   redval   += (addpuffer[ ys+x+2 ] / count);
                  valcnt++;
               }
				   count  = addcount[ ycs+xc+1 ];
				   if( count > 0 )
               {
					   blueval  += (addpuffer[ ys+x+3 ] / count);
					   greenval += (addpuffer[ ys+x+4 ] / count);
					   redval   += (addpuffer[ ys+x+5 ] / count);
                  valcnt++;
               }
				   count  = addcount[ ycs+ycstep+xc ];
				   if( count > 0 )
               {
					   blueval  += (addpuffer[ ys+yastep+x   ] / count);
					   greenval += (addpuffer[ ys+yastep+x+1 ] / count);
					   redval   += (addpuffer[ ys+yastep+x+2 ] / count);
                  valcnt++;
               }
				   count  = addcount[ ycs+ycstep+xc+1 ];
				   if( count > 0 )
               {
					   blueval  += (addpuffer[ ys+yastep+x+3 ] / count);
					   greenval += (addpuffer[ ys+yastep+x+4 ] / count);
					   redval   += (addpuffer[ ys+yastep+x+5 ] / count);
                  valcnt++;
               }

               if( valcnt > 0 )
               {
					   destpic[ yd+xd   ] = (short)(blueval/valcnt);
					   destpic[ yd+xd+1 ] = (short)(greenval/valcnt);
					   destpic[ yd+xd+2 ] = (short)(redval/valcnt);
               }
               else
               {
					   destpic[ yd+xd   ] = destpicparam->bblankvalue.word;
					   destpic[ yd+xd+1 ] = destpicparam->gblankvalue.word;
					   destpic[ yd+xd+2 ] = destpicparam->blankvalue.word;
               }
			   }
		   }
         break;
      case CCDLIB_MoveShrink4Const:
      	xastop = addpicparam->linelength * 3;
	      yastop = addpicparam->columnheight * addpicparam->linelength * 3;
         yastep = addpicparam->linelength * 3;
         ydstep = destpicparam->linelength * 3;

		   for(ys = 0, yd = 0;
			    ys < yastop;
			    ys += yastep*4, yd += ydstep )
		   {
			   for(x = 0, xd = 0;
				    x < xastop;
				    x += 12, xd += 3 )
			   {
               blueval  = 0;
               greenval = 0;
               redval   = 0;
               valcnt   = 0;
               ymstop   = ys+4*yastep;

               for( ym=ys; ym<ymstop; ym+=yastep )
               {
                  for( xm=x; xm<x+12; xm+=3 )
                  {
				         count = addcount[ (ym+xm)/3 ];
                     if( count > 0 )
                     {
					         blueval  += (addpuffer[ ym+xm   ] / count);
					         greenval += (addpuffer[ ym+xm+1 ] / count);
					         redval   += (addpuffer[ ym+xm+2 ] / count);
                        valcnt++;
                     }
                  }
               }

               if( valcnt > 0 )
               {
					   destpic[ yd+xd   ] = (short)(blueval/valcnt);
					   destpic[ yd+xd+1 ] = (short)(greenval/valcnt);
					   destpic[ yd+xd+2 ] = (short)(redval/valcnt);
               }
               else
               {
					   destpic[ yd+xd   ] = destpicparam->bblankvalue.word;
					   destpic[ yd+xd+1 ] = destpicparam->gblankvalue.word;
					   destpic[ yd+xd+2 ] = destpicparam->blankvalue.word;
               }
			   }
		   }
         break;
      }
	}
	else
	{
      switch( movemethod )
      {
      case CCDLIB_MoveUnchangedConst:
      	xastop = addpicparam->linelength;
	      yastop = addpicparam->columnheight * addpicparam->linelength;

   		for(ys = 0;
	   		 ys < yastop;
		   	 ys += xastop )
		   {
			   for(x = 0;
				    x < xastop;
				    x++ )
			   {
				   if( addcount[ ys+x ] > 0 )
					   destpic[ ys+x ] = (short)( addpuffer[ ys+x ] / addcount[ ys+x ] );
				   else
					   destpic[ ys+x ] = destpicparam->blankvalue.word;
			   }
		   }
      case CCDLIB_MoveCutoffConst:
         if( cutrect == NULL )
            return( CCDLIB_WrongRectangleValuesError );

      	xastop = cutrect->xpos+cutrect->xsize;
	      yastop = (cutrect->ypos+cutrect->ysize) * addpicparam->linelength;
         yastep = addpicparam->linelength;
	      ydstep = destpicparam->linelength;

		   for(ys = cutrect->ypos*yastep, yd = 0;
			    ys < yastop;
			    ys += yastep, yd += ydstep )
		   {
			   for(x = cutrect->xpos, xd = 0;
				    x < xastop;
				    x ++, xd ++ )
			   {
				   count = addcount[ ys+x ];
				   if( count > 0 )
					   destpic[ yd+xd ] = (short)( addpuffer[ ys+x ] / count );
				   else
					   destpic[ yd+xd ] = destpicparam->blankvalue.word;
			   }
		   }
         break;
      case CCDLIB_MoveShrink2Const:
      	xastop = addpicparam->linelength;
	      yastop = addpicparam->columnheight * addpicparam->linelength;
         yastep = addpicparam->linelength;
         ydstep = destpicparam->linelength;

		   for(ys = 0, yd = 0;
			    ys < yastop;
			    ys += yastep*2, yd += ydstep )
		   {
			   for(x = 0, xd = 0;
				    x < xastop;
				    x += 2, xd++ )
			   {
               greyval  = 0;
               valcnt   = 0;

				   count = addcount[ ys+x ];
				   if( count > 0 )
               {
					   greyval += (addpuffer[ ys+x ] / count);
                  valcnt++;
               }
				   count = addcount[ ys+x+1 ];
				   if( count > 0 )
               {
					   greyval += (addpuffer[ ys+x+1 ] / count);
                  valcnt++;
               }
				   count = addcount[ ys+yastep+x ];
				   if( count > 0 )
               {
					   greyval += (addpuffer[ ys+yastep+x ] / count);
                  valcnt++;
               }
				   count = addcount[ ys+yastep+x+1 ];
				   if( count > 0 )
               {
					   greyval += (addpuffer[ ys+yastep+x+1 ] / count);
                  valcnt++;
               }

               if( valcnt > 0 )
					   destpic[ yd+xd ] = (short)(greyval/valcnt);
               else
					   destpic[ yd+xd ] = destpicparam->blankvalue.word;
			   }
		   }
         break;
      case CCDLIB_MoveShrink4Const:
      	xastop = addpicparam->linelength;
	      yastop = addpicparam->columnheight * addpicparam->linelength;
         yastep = addpicparam->linelength;
         ydstep = destpicparam->linelength;

		   for(ys = 0, yd = 0;
			    ys < yastop;
			    ys += yastep*4, yd += ydstep )
		   {
			   for(x = 0, xd = 0;
				    x < xastop;
				    x += 4, xd++ )
			   {
               greyval  = 0;
               valcnt   = 0;
               ymstop   = ys+4*yastep;

               for( ym=ys; ym<ymstop; ym+=yastep )
               {
                  for( xm=x; xm<x+4; xm++ )
                  {
				         count = addcount[ ym+xm ];
                     if( count > 0 )
                     {
					         greyval += (addpuffer[ ym+xm ] / count);
                        valcnt++;
                     }
                  }
               }

               if( valcnt > 0 )
					   destpic[ yd+xd ] = (short)(greyval/valcnt);
               else
					   destpic[ yd+xd ] = destpicparam->blankvalue.word;
			   }
		   }
         break;
      }

	}
	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_QuickEnlarge: Geschwindigkeitsoptimiertes Vergroessern eines   *
*               Puffers um das zwei- und vierfache                        *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *sourcepic   : 32-Bit-Pointer auf die Quellbilddaten      *
*               *destpic     : 32-Bit-Pointer auf die Zielbilddaten       *
*               enlargefac   : Vergroesserungsfaktor                      *
*                                                                         *
*               ACHTUNG: sourcepic darf nicht destpic sein !              *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (7. 7. 2001 )                                         *
*   Written by: Georg Dittie, 2001                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_QuickEnlarge
(
	CCDLIB_PictureparamStruct * picparam,
	short * sourcepic,
	short * destpic,
	long enlargefac
)
{
	long xs, xs1, xd, yd1, xf, yf, yss, yss1, yd, yds, ys, ys1,
        xdest0, xdest1, xsrc0, xsrc1,
        ysstep, ysstop, ydstop, ydstep,
		  xc, xm0, xm1, xm2, xm3;

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( sourcepic == destpic )
		return( CCDLIB_WrongParameterError );

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
      switch( enlargefac )
      {
      case 2:
         ysstop = picparam->linelength*(picparam->columnheight-1);
         ysstep = picparam->linelength;
         ydstep = ysstep*4;

         for( ys=0, yd=0, ys1=ysstep, yd1=ysstep*2;
              ys<ysstop;
              ys+=ysstep, yd+=ydstep, ys1+=ysstep, yd1+=ydstep )
         {
            for( xs=0, xdest0=yd, xdest1=yd1, xsrc0=ys, xsrc1=ys1;
                 xs<ysstep;
                 xs++, xdest0+=2, xdest1+=2, xsrc0++, xsrc1++ )
            {
               destpic[xdest0] = sourcepic[xsrc0];
               destpic[xdest0+1] = (short)((sourcepic[xsrc0]+sourcepic[xsrc0+1])>>1);
               destpic[xdest1] = (short)((sourcepic[xsrc0]+sourcepic[xsrc1])>>1);
               destpic[xdest1+1] = (short)(
                  (sourcepic[xsrc0]+sourcepic[xsrc0+1]+sourcepic[xsrc1]+sourcepic[xsrc1+1])>>2);
            }
         }
         break;
      case 4:
 	      ydstep = picparam->linelength*4;
 	      ydstop = picparam->linelength*picparam->columnheight*16;
         ysstep = picparam->linelength;
         for( yd=0, yds=0; yd<ydstop; yd++, yds+=ydstep )
         {
            yss  = (yd>>2)*ysstep;
            yss1 = yss+ysstep;
            yf   = yd&0x03;
            for( xd=0; xd<ydstep; xd++ )
            {
               xs  = xd>>2;
               xs1 = xs+1;
               xf  = xd&0x03;

 					xm0 = (long)sourcepic[ yss +xs  ];
					xm1 = (long)sourcepic[ yss +xs1 ];
					xm2 = (long)sourcepic[ yss1+xs  ];
					xm3 = (long)sourcepic[ yss1+xs1 ];

               destpic[ yds+xd ] = (short)(
						( (xm0<<4) + (xm1-xm0)*(xf<<2) + (((xm2-xm0)<<2) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 4 );
            }
         }
         break;
      default:
         return( CCDLIB_WrongParameterError );
      }
   }
   else
   {
      switch( enlargefac )
      {
      case 2:
         ysstop = picparam->linelength*(picparam->columnheight-1)*3;
         ysstep = picparam->linelength*3;
         ydstep = ysstep*4;

         for( ys=0, yd=0, ys1=ysstep, yd1=ysstep*2;
              ys<ysstop;
              ys+=ysstep, yd+=ydstep, ys1+=ysstep, yd1+=ydstep )
         {
            for( xs=0, xdest0=yd, xdest1=yd1, xsrc0=ys, xsrc1=ys1;
                 xs<ysstep;
                 xs+=3, xdest0+=6, xdest1+=6, xsrc0+=3, xsrc1+=3 )
            {
               destpic[xdest0  ] = sourcepic[xsrc0  ];
               destpic[xdest0+1] = sourcepic[xsrc0+1];
               destpic[xdest0+2] = sourcepic[xsrc0+2];

               destpic[xdest0+3] = (short)((sourcepic[xsrc0  ]+sourcepic[xsrc0+3])>>1);
               destpic[xdest0+4] = (short)((sourcepic[xsrc0+1]+sourcepic[xsrc0+4])>>1);
               destpic[xdest0+5] = (short)((sourcepic[xsrc0+2]+sourcepic[xsrc0+5])>>1);

               destpic[xdest1  ] = (short)((sourcepic[xsrc0  ]+sourcepic[xsrc1  ])>>1);
               destpic[xdest1+1] = (short)((sourcepic[xsrc0+1]+sourcepic[xsrc1+1])>>1);
               destpic[xdest1+2] = (short)((sourcepic[xsrc0+2]+sourcepic[xsrc1+2])>>1);

               destpic[xdest1+3] = (short)(
                  (sourcepic[xsrc0  ]+sourcepic[xsrc0+4]+sourcepic[xsrc1  ]+sourcepic[xsrc1+4])>>2);
               destpic[xdest1+4] = (short)(
                  (sourcepic[xsrc0+1]+sourcepic[xsrc0+5]+sourcepic[xsrc1+1]+sourcepic[xsrc1+5])>>2);
               destpic[xdest1+5] = (short)(
                  (sourcepic[xsrc0+2]+sourcepic[xsrc0+6]+sourcepic[xsrc1+2]+sourcepic[xsrc1+6])>>2);
            }
         }
         break;

      case 4:
/*
         ysstop = picparam->linelength*(picparam->columnheight-1)*3;
         ysstep = picparam->linelength*3;
         ydstep = ysstep*4;

         for( ys=0, yd=0, ys1=ysstep, yd1=ysstep*2;
              ys<ysstop;
              ys+=ysstep, yd+=ydstep, ys1+=ysstep, yd1+=ydstep )
         {
            for( xs=0, xdest0=yd, xdest1=yd1, xsrc0=ys, xsrc1=ys1;
                 xs<ysstep;
                 xs+=3, xdest0+=6, xdest1+=6, xsrc0+=3, xsrc1+=3 )
            {
               destpic[xdest0  ] = sourcepic[xsrc0  ];
               destpic[xdest0+1] = sourcepic[xsrc0+1];
               destpic[xdest0+2] = sourcepic[xsrc0+2];

               destpic[xdest0+3] = (short)((sourcepic[xsrc0  ]*3+sourcepic[xsrc0+3])>>2);
               destpic[xdest0+4] = (short)((sourcepic[xsrc0+1]*3+sourcepic[xsrc0+4])>>2);
               destpic[xdest0+5] = (short)((sourcepic[xsrc0+2]*3+sourcepic[xsrc0+5])>>2);

               destpic[xdest0+6] = (short)((sourcepic[xsrc0  ]+sourcepic[xsrc0+3])>>1);
               destpic[xdest0+7] = (short)((sourcepic[xsrc0+1]+sourcepic[xsrc0+4])>>1);
               destpic[xdest0+8] = (short)((sourcepic[xsrc0+2]+sourcepic[xsrc0+5])>>1);

               destpic[xdest0+9]  = (short)((sourcepic[xsrc0  ]+sourcepic[xsrc0+3]*3)>>2);
               destpic[xdest0+10] = (short)((sourcepic[xsrc0+1]+sourcepic[xsrc0+4]*3)>>2);
               destpic[xdest0+11] = (short)((sourcepic[xsrc0+2]+sourcepic[xsrc0+5]*3)>>2);

               destpic[xdest1  ] = (short)((sourcepic[xsrc0  ]*3+sourcepic[xsrc1  ])>>2);
               destpic[xdest1+1] = (short)((sourcepic[xsrc0+1]*3+sourcepic[xsrc1+1])>>2);
               destpic[xdest1+2] = (short)((sourcepic[xsrc0+2]*3+sourcepic[xsrc1+2])>>2);

               destpic[xdest1+3] = (short)(
                  (sourcepic[xsrc0  ]*3+sourcepic[xsrc0+4]+sourcepic[xsrc1  ]*3+sourcepic[xsrc1+4])>>4);
               destpic[xdest1+4] = (short)(
                  (sourcepic[xsrc0+1]*3+sourcepic[xsrc0+5]+sourcepic[xsrc1+1]*3+sourcepic[xsrc1+5])>>4);
               destpic[xdest1+5] = (short)(
                  (sourcepic[xsrc0+2]*3+sourcepic[xsrc0+6]+sourcepic[xsrc1+2]*3+sourcepic[xsrc1+6])>>4);

               destpic[xdest1+6] = (short)(
                  (sourcepic[xsrc0  ]*2+sourcepic[xsrc0+4]+sourcepic[xsrc1  ]*2+sourcepic[xsrc1+4])>>4);
               destpic[xdest1+7] = (short)(
                  (sourcepic[xsrc0+1]*2+sourcepic[xsrc0+5]+sourcepic[xsrc1+1]*2+sourcepic[xsrc1+5])>>4);
               destpic[xdest1+8] = (short)(
                  (sourcepic[xsrc0+2]*2+sourcepic[xsrc0+6]+sourcepic[xsrc1+2]*2+sourcepic[xsrc1+6])>>4);

               destpic[xdest1+9]  = (short)(
                  (sourcepic[xsrc0  ]+sourcepic[xsrc0+4]+sourcepic[xsrc1  ]+sourcepic[xsrc1+4])>>2);
               destpic[xdest1+10] = (short)(
                  (sourcepic[xsrc0+1]+sourcepic[xsrc0+5]+sourcepic[xsrc1+1]+sourcepic[xsrc1+5])>>2);
               destpic[xdest1+11] = (short)(
                  (sourcepic[xsrc0+2]+sourcepic[xsrc0+6]+sourcepic[xsrc1+2]+sourcepic[xsrc1+6])>>2);
            }
         }
*/
 	      ydstep = picparam->linelength*12;
 	      ydstop = picparam->columnheight*4;
         ysstep = picparam->linelength*3;
         for( yd=0, yds=0; yd<ydstop; yd++, yds+=ydstep )
         {
            yss  = (yd>>2)*ysstep;
            yss1 = yss+ysstep;
            yf   = yd&0x03;
            for( xd=0, xc=0; xd<ydstep; xc++, xd+=3 )
            {
               xs  = (xc>>2)*3;
               xs1 = xs+3;
               xf  = xc&0x03;

 					xm0 = (long)sourcepic[ yss +xs  ];
					xm1 = (long)sourcepic[ yss +xs1 ];
					xm2 = (long)sourcepic[ yss1+xs  ];
					xm3 = (long)sourcepic[ yss1+xs1 ];

               destpic[ yds+xd ] = (short)(
						( (xm0<<2) + (xm1-xm0)*(xf<<1) + (((xm2-xm0)<<1) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 2 );

 					xm0 = (long)sourcepic[ yss +xs +1 ];
					xm1 = (long)sourcepic[ yss +xs1+1 ];
					xm2 = (long)sourcepic[ yss1+xs +1 ];
					xm3 = (long)sourcepic[ yss1+xs1+1 ];

               destpic[ yds+xd+1 ] = (short)(
						( (xm0<<2) + (xm1-xm0)*(xf<<1) + (((xm2-xm0)<<1) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 2 );

 					xm0 = (long)sourcepic[ yss +xs +2 ];
					xm1 = (long)sourcepic[ yss +xs1+2 ];
					xm2 = (long)sourcepic[ yss1+xs +2 ];
					xm3 = (long)sourcepic[ yss1+xs1+2 ];

               destpic[ yds+xd+2 ] = (short)(
						( (xm0<<2) + (xm1-xm0)*(xf<<1) + (((xm2-xm0)<<1) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 2 );
            }
         }
         break;
      default:
         return( CCDLIB_WrongParameterError );
      }
   }

	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_GetPictureCorr : Routine zur Bestimmung der Kreuzkorrelation   *
*                           von zwei gegeneinander verschobenen Bildern   *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf das Bild                *
*               *reference   : 32-Bit-Pointer auf das Vergleichsbild      *
*               *picshift    : Verschiebung der Bilder gegeneinander      *
*               piclowpass   : Zusaetzlicher Tiefpass fuer Bild           *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*               correlation:     : ermittelte Kreuzkorrelation            *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !		              *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0, (23.08.2000, Sommerfrische)                32 Bit    *
*   Written by: Georg Dittie, 2000                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetPictureCorr
(
   double *correlation,
	short * picture,
	short * reference,
	CCDLIB_PictureparamStruct * pictureparam,
   BOOL piclowpass,
	CCDLIB_PositionStruct *picshift,
   CCDLIB_RectangleStruct *evalrect
)
{
	long buflength, x1, ys1, x2, ys2, ysstep, n,
        x1start, x1stop, x2start, x2stop,
        ys1start, ys2start, ys1stop, ys2stop;
   double val1, val2, mean1, mean2, meansqr1, meansqr2, meansqr12;

   short *tempbuf1, *tempbuf2;

   CCDLIB_PictureparamStruct workpicparam;
   CCDLIB_RectangleStruct picturerect;

   if( picture == NULL || reference == NULL )
		return( CCDLIB_NoPointerError );

   if( abs( picshift->xpos ) > (pictureparam->linelength/2) ||
       abs( picshift->ypos ) > (pictureparam->columnheight/2) )
   {
      *correlation = 0.;
      return( CCDLIB_Successful );
   }

   buflength =
      pictureparam->linelength*pictureparam->columnheight*sizeof(short);
   tempbuf1 = (short *)malloc( buflength );
   tempbuf2 = (short *)malloc( buflength );
   if( tempbuf1 == NULL || tempbuf2 == NULL )
   {
      if( tempbuf1 != NULL ) free( tempbuf1 );
      if( tempbuf2 != NULL ) free( tempbuf2 );
      return( CCDLIB_OutOfMemoryError );
   }

	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
   {
		CCDLIB_ExtractLuminance( picture, pictureparam, tempbuf1 );
      CCDLIB_ExtractLuminance( reference, pictureparam, tempbuf2 );
   }
   else
   {
      memcpy( tempbuf1, picture, buflength );
      memcpy( tempbuf2, reference, buflength );
   }

   workpicparam = *pictureparam;
   workpicparam.colorchannel = CCDLIB_BlackWhiteColor;

   if( piclowpass == TRUE )
   {
      if( evalrect == NULL )
         CCDLIB_InitRectangle(
            &picturerect, 0, 0,
            workpicparam.linelength, workpicparam.columnheight );
      else
         picturerect = *evalrect;

 	   CCDLIB_NoisefilterFunction(
		   &picturerect, &workpicparam, tempbuf1, tempbuf1,
         1., CCDLIB_Noise5x5MittelConst, TRUE );
   }

   if( evalrect == NULL )
   {
      if( picshift->xpos >= 0 )
      {
	      x1start = CCDLIB_MINIMALRIM;
         x2start = CCDLIB_MINIMALRIM+picshift->xpos;
         x1stop  = pictureparam->linelength-CCDLIB_MINIMALRIM-picshift->xpos;
      }
      else
      {
	      x1start = CCDLIB_MINIMALRIM-picshift->xpos;
         x2start = CCDLIB_MINIMALRIM;
         x1stop  = pictureparam->linelength-CCDLIB_MINIMALRIM;
      }

      if( picshift->ypos >= 0 )
      {
	      ys1start = CCDLIB_MINIMALRIM;
         ys2start = CCDLIB_MINIMALRIM+picshift->ypos;
         ys1stop  = pictureparam->columnheight-CCDLIB_MINIMALRIM-picshift->ypos;
      }
      else
      {
	      ys1start = CCDLIB_MINIMALRIM-picshift->ypos;
         ys2start = CCDLIB_MINIMALRIM;
         ys1stop  = pictureparam->columnheight-CCDLIB_MINIMALRIM;
      }
   }
   else
   {
      x1start = evalrect->xpos;
      x2start = evalrect->xpos+picshift->xpos;
      x1stop  = evalrect->xpos+evalrect->xsize;

      ys1start = evalrect->ypos;
      ys2start = evalrect->ypos+picshift->ypos;
      ys1stop  = evalrect->ypos+evalrect->ysize;
   }

   ys1start *= pictureparam->linelength;
   ys2start *= pictureparam->linelength;
   ys1stop  *= pictureparam->linelength;

   x2stop  = pictureparam->linelength;
   ys2stop = pictureparam->linelength*pictureparam->columnheight;

   ysstep = pictureparam->linelength;

   mean1 = 0.; mean2 = 0.; n = 0L;
   for( ys1 = ys1start, ys2 = ys2start;
        ys1 < ys1stop;
        ys1 += ysstep, ys2 += ysstep )
   {
      if( ys2 < 0 || ys2 >= ys2stop )
         continue;

      for( x1 = x1start, x2 = x2start;
           x1 < x1stop;
           x1 ++, x2++ )
      {
         if( x2 < 0 || x2 >= x2stop )
            continue;

         val1 = (double)tempbuf1[ys1+x1];
         val2 = (double)tempbuf2[ys2+x2];
         mean1 += val1;
         mean2 += val2;
         n++;
      }
   }
   if( n > 0 )
   {
      mean1 /= (double)n;
      mean2 /= (double)n;
   }
   else
   {
      free( tempbuf1 );
      free( tempbuf2 );
      *correlation = 0.;
      return( CCDLIB_Successful );
   }

   meansqr1  = 0.;
   meansqr2  = 0.;
   meansqr12 = 0.;
   for( ys1 = ys1start, ys2 = ys2start;
        ys1 < ys1stop;
        ys1 += ysstep, ys2 += ysstep )
   {
      if( ys2 < 0 || ys2 >= ys2stop )
         continue;
      for( x1 = x1start, x2 = x2start;
           x1 < x1stop;
           x1++, x2++ )
      {
         if( x2 < 0 || x2 >= x2stop )
            continue;
         val1 = (double)tempbuf1[ys1+x1] - mean1;
         val2 = (double)tempbuf2[ys2+x2] - mean2;
         meansqr1  += val1*val1;
         meansqr2  += val2*val2;
         meansqr12 += val1*val2;
      }
   }
   meansqr1  /= (double)n;
   meansqr2  /= (double)n;
   meansqr12 /= (double)n;

   if( meansqr1*meansqr2 > 0. )
      *correlation = meansqr12 / sqrt( meansqr1*meansqr2 );
   else
      *correlation = 0.;

   free( tempbuf1 );
   free( tempbuf2 );

	return( CCDLIB_Successful );
}

/*** Functions for partial image separation ***/

/************************************************************************
*                                                                       *
*	 CCDLIB_ExtractOddEven : Routine zur Trennung vom Bild mit geraden   *
*   und ungraden Zeilen aus einem Interlace-Bild, wobei die Leerzeilen  *
*   mit linearer Interpolation gefuellt werden.                         *
*                                                                       *
*   Parameter   * oddpic :       Bild der ungeraden Zeilen              *
*               * evenpic :      Bild der geraden Zeilen                *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf das aktuelle Bild   *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Ergebnis muss in den vorher initialisierten Buffern stehen      *
*                                                                       *
*   Version 1.0 vom Pfingstsonntag 1999       (c) '99 by DITTELSOFT     *
*   programmiert im Dachgarten zur 18. Planetentagung in Violau         *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_ExtractOddEven
(
	short *oddpic,
	short *evenpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct * pictureparam
)
{
	long x, ys, ym, yp,
		  xstop, ystep, ystop, ydoff, ysoff;

	if( sourcepic == NULL || evenpic == NULL || oddpic == NULL )
		return( CCDLIB_NoPointerError );

	xstop = pictureparam->linelength;
	ystep = pictureparam->linelength*2;
	ystop = pictureparam->linelength*pictureparam->columnheight;
	ysoff = ystop - ystep;
	ydoff = ystop - pictureparam->linelength;

	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	{
		xstop *= 3;
		ystep *= 3;
		ystop *= 3;
		ysoff *= 3;
		ydoff *= 3;
	}

	for( ys=0; ys<ystop; ys+=ystep )
		for( x=0; x<xstop; x++ )
			oddpic[ ys+x ] = sourcepic[ ys+x ];

	for( ys=xstop; ys<ystop; ys+=ystep )
		for( x=0; x<xstop; x++ )
			evenpic[ ys+x ] = sourcepic[ ys+x ];

	for( ys=xstop, ym=0, yp=2*xstop;
		  yp<ystop;
		  ys+=ystep, ym+=ystep, yp+=ystep )
		for( x=0; x<xstop; x++ )
			oddpic[ ys+x ] = (short)((sourcepic[ ym+x ]+sourcepic[ yp+x ])/2);

	for( ys=2*xstop, ym=xstop, yp=3*xstop;
		  yp<ystop;
		  ys+=ystep, ym+=ystep, yp+=ystep )
		for( x=0; x<xstop; x++ )
			evenpic[ ys+x ] = (short)((sourcepic[ ym+x ]+sourcepic[ yp+x ])/2);

	for( x=0; x<xstop; x++ )
		evenpic[ x ] = sourcepic[ xstop+x ];

	if( (pictureparam->columnheight%2) == 0 )
		for( x=0; x<xstop; x++ )
			oddpic[ ydoff+x ] = sourcepic[ ysoff+x ];

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_ExtractInterlace : Routine zur Separierung von Teilbildern   *
*   aus einem Interlace-Bild, wobei die Leerzeilen mit linearer         *
*   Interpolation gefuellt werden.                                      *
*                                                                       *
*   Parameter   * extractpic :   extrahiertes Bild                      *
*               * picture     :  32-Bit-Pointer auf das aktuelle Bild   *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Ergebnis muss in den vorher initialisierten Buffern stehen      *
*                                                                       *
*   Version 1.1 vom 13. 06. 2003       (c) 2000 - 2003 by DITTELSOFT    *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_ExtractInterlace
(
	short *extractpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct * pictureparam,
   enum CCDLIB_FrametypeEnum frametype
)
{
	long x, ys, ym, yp, buflength,
		  xstop, ystep, ystop, ydoff, ysoff;
   short *temppic;

	if( sourcepic == NULL || extractpic == NULL )
		return( CCDLIB_NoPointerError );

   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
      buflength = pictureparam->linelength*pictureparam->columnheight*3*sizeof(short);
   else
      buflength = pictureparam->linelength*pictureparam->columnheight*sizeof(short);

	if( sourcepic == extractpic )
   {
      temppic = (short*)malloc( buflength );
      if( temppic == NULL )
		   return( CCDLIB_WrongParameterError );
   }
   else
      temppic = extractpic;

   if( frametype == CCDLIB_FullframeConst && sourcepic != extractpic )
   {
      memcpy( extractpic, sourcepic, buflength );
      return( CCDLIB_Successful );
   }

	xstop = pictureparam->linelength;
	ystep = pictureparam->linelength*2;
	ystop = pictureparam->linelength*pictureparam->columnheight;
	ysoff = ystop - ystep;
	ydoff = ystop - pictureparam->linelength;

	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	{
		xstop *= 3;
		ystep *= 3;
		ystop *= 3;
		ysoff *= 3;
		ydoff *= 3;
	}

   if( frametype == CCDLIB_OddPicConst )
   {
	   for( ys=0; ys<ystop; ys+=ystep )
		   for( x=0; x<xstop; x++ )
			   temppic[ ys+x ] = sourcepic[ ys+x ];

   	for( ys=xstop, ym=0, yp=2*xstop;
	   	  yp<ystop;
		     ys+=ystep, ym+=ystep, yp+=ystep )
		   for( x=0; x<xstop; x++ )
			   temppic[ ys+x ] = (short)((sourcepic[ ym+x ]+sourcepic[ yp+x ])/2);

   	if( (pictureparam->columnheight%2) == 0 )
	   	for( x=0; x<xstop; x++ )
		   	temppic[ ydoff+x ] = sourcepic[ ysoff+x ];
   }
   else
   {
   	for( ys=xstop; ys<ystop; ys+=ystep )
	   	for( x=0; x<xstop; x++ )
		   	temppic[ ys+x ] = sourcepic[ ys+x ];

	   for( ys=2*xstop, ym=xstop, yp=3*xstop;
		     yp<ystop;
		     ys+=ystep, ym+=ystep, yp+=ystep )
		   for( x=0; x<xstop; x++ )
			   temppic[ ys+x ] = (short)((sourcepic[ ym+x ]+sourcepic[ yp+x ])>>1);

	   for( x=0; x<xstop; x++ )
		   temppic[ x ] = sourcepic[ xstop+x ];
   }

	if( sourcepic == extractpic )
   {
      memcpy( extractpic, temppic, buflength );
      free( temppic );
      return( CCDLIB_Successful );
   }

	return( CCDLIB_Successful );
}

/*** Bestimmung von Verschiebungskoordinaten ***/

/**************************************************************************
*                                                                         *
*   CCDLIB_GetCenterOfDisk : Routine zur Koordinatenbestimmung des        *
*                            geometrischen Schwerpunkts eines             *
*                            hellen Objekts.                              *
*                                                                         *
*   Parameter:  *picturerect : Arbeitsbereich im Bild                     *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               threshold    : Beruecksichtigungsschwelle Max/Untergr.    *
*               *picpuffer   : 32-Bit-Pointer auf das aktuelle Bild       *
*               *centercoord : Position des Schwerpunkts                  *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !		              *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.1, (03.10.1997, CCD-Praktikum, Sylvester 2001)          *
*   Written by: Georg Dittie, 1997, 2001                                  *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetCenterOfDisk
(
   CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_PositionStruct *centercoord,
   double *matchvalue
)
{
	long xaltpos, yaltpos, xcenter, ycenter, yupos, ydpos, xlpos, xrpos;
	long xstart, xstop, ystart, ystop, stepcount;

	long histomaximum, histomaxpos, fwhmlevel;
	double histomaxpart, histoentropy;
	long histolowestpos, histohighestpos, histohighpos, histolowpos;
	long * histogramfield;
   short *searchbuf;
   CCDLIB_PictureparamStruct searchparam;
   CCDLIB_PositionStruct startcoord;
   enum CCDLIB_ErrorEnum result;

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( picture == NULL )
		return( CCDLIB_NoPointerError );

 	histogramfield = (long *)malloc( pictureparam->dynamik * sizeof(long) );
   searchbuf = (short *)malloc( pictureparam->linelength*pictureparam->columnheight*sizeof(short) );
	if( histogramfield == NULL || searchbuf == NULL )
   {
      if( histogramfield != NULL ) free( histogramfield );
      if( searchbuf != NULL ) free( searchbuf );
		return( CCDLIB_OutOfMemoryError );
   }

 	if( CCDLIB_RetrieveHistogram( picturerect, pictureparam,
			 picture, histogramfield ) != CCDLIB_Successful )
   {
      if( histogramfield != NULL ) free( histogramfield );
      if( searchbuf != NULL ) free( searchbuf );
		return( CCDLIB_NoPointerError );
   }

	CCDLIB_EvaluateHistogram(
		histogramfield,
		pictureparam->dynamik,
		CCDLIB_Histo99_8Const,
		0,
		0,
		&histolowestpos,
		&histohighestpos,
		&histolowpos,
		&histohighpos,
		&histomaxpos,
		&histomaximum,
		&histomaxpart,
		&histoentropy );

	fwhmlevel = (long)floor(((double)histohighpos - (double)histomaxpos) * 0.4
					 + (double)histomaxpos + 0.5 );

	xstart  = picturerect->xpos;
	ystart  = picturerect->ypos;
	xstop   = picturerect->xpos + picturerect->xsize;
	ystop   = picturerect->ypos + picturerect->ysize;

   /*** filling searchbuffer ***/

   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
		CCDLIB_ExtractLuminance( picture, pictureparam, searchbuf );
   else
      memcpy( searchbuf, picture,
      	pictureparam->linelength*pictureparam->columnheight*sizeof(short) );

   searchparam = *pictureparam;
   searchparam.colorchannel = CCDLIB_BlackWhiteColor;

 	CCDLIB_NoisefilterFunction(
		picturerect, &searchparam, searchbuf, searchbuf,
      1., CCDLIB_Noise5x5MittelConst, TRUE );

   /*** looking for the brightest point ***/

   result = CCDLIB_GetCenterOfBrightness(
               picturerect, &searchparam,
               0.4, searchbuf, &startcoord );
   if( result != CCDLIB_Successful ||
       (startcoord.xpos == 0 && startcoord.ypos == 0) )
   {
      *matchvalue = 0.;
      centercoord->xpos = 0;
	   centercoord->ypos = 0;
   }
   else
   {
      xcenter = startcoord.xpos;
		ycenter = startcoord.ypos;
   }

	/*** Iteratives Suchstrahlverfahren ***/
	stepcount = 0;
   *matchvalue = 1.;
	do
	{
		xaltpos = xcenter; yaltpos = ycenter;
		for( xlpos=xcenter-3; xlpos>=xstart; xlpos-- )
			if( CCDLIB_GetCenterMajority(
               searchbuf, pictureparam, fwhmlevel,
               xlpos, 5, 3, CCDLIB_Vertical ) ) break;
		for( xrpos=xcenter+3; xrpos<xstop; xrpos++ )
			if( CCDLIB_GetCenterMajority(
               searchbuf, pictureparam, fwhmlevel,
               xrpos, 5, 3, CCDLIB_Vertical ) ) break;
		for( yupos=ycenter-3; yupos>=ystart; yupos-- )
			if( CCDLIB_GetCenterMajority(
               searchbuf, pictureparam, fwhmlevel,
               yupos, 5, 3, CCDLIB_Horizontal ) ) break;
		for( ydpos=ycenter+3; ydpos<ystop; ydpos++ )
			if( CCDLIB_GetCenterMajority(
               searchbuf, pictureparam, fwhmlevel,
               ydpos, 5, 3, CCDLIB_Horizontal ) ) break;

		xcenter = (xlpos+xrpos)/2; ycenter = (yupos+ydpos)/2;
		stepcount++;
	}
	while( (xaltpos != xcenter || yaltpos != ycenter) && stepcount < 4 );

   if( xlpos<=xstart+16 )     *matchvalue = 0.;
   else if( yupos<=ystart+16 )*matchvalue = 0.;
   else if( xrpos>=xstop-16 ) *matchvalue = 0.;
   else if( ydpos>=ystop-16 ) *matchvalue = 0.;
   else *matchvalue = 1.;

	centercoord->xpos = xcenter;
	centercoord->ypos = ycenter;

	free( histogramfield );
   free( searchbuf );

	return( CCDLIB_Successful );
}

BOOL CCDLIB_GetCenterMajority(
   short *buffer,
   CCDLIB_PictureparamStruct *picparam,
   long fwhmlevel,
   long samplecoord,
   long samplenum,
   long sampledist,
   enum CCDLIB_OrientationEnum sampledir )
{
   long i, start, stop, step, samplehalf, count = 0;

   samplehalf = samplenum / 2;
   switch( sampledir )
   {
   case CCDLIB_Horizontal:
      start = samplecoord-samplehalf*sampledist;
      stop  = samplecoord+samplehalf*sampledist;
      if( start < 0 || stop >= picparam->linelength )
      	return( TRUE );
      for( i=start; i<=stop; i+=sampledist )
         if( buffer[ i ] < fwhmlevel ) count++;
      break;
   case CCDLIB_Vertical:
      start = (samplecoord-samplehalf*sampledist)*picparam->linelength;
      stop  = (samplecoord+samplehalf*sampledist)*picparam->linelength;
      step  = picparam->linelength*sampledist;
      if( start < 0 || stop >= picparam->linelength*picparam->columnheight )
      	return( TRUE );
      for( i=start; i<=stop; i+=step )
         if( buffer[ i ] < fwhmlevel ) count++;
      break;
   }

   if( count > samplenum/2 )
      return( TRUE );
   else
      return( FALSE );
}


/*** Hilfsfunktionen ***/

long CCDLIB_GetMeanSample(
	long xcoord,
	long ycoord,
	long picwidth,
	long picheight,
	enum CCDLIB_ColorChannelEnum colorchannel,
	long samplesize,
	short *picture )
{
	long x, xstart, xstop, ys, ystart, ystop, ystep,
		  samplehalf, samplenum, greyval=0;

  	ystep  = picwidth;
   if( samplesize == 1 )
   {
		if( colorchannel != CCDLIB_RGBChannelColor )
		{
			greyval = (long)picture[xcoord+ycoord*ystep];
		}
		else
		{
         x  = xcoord;
         ys = ycoord*ystep;
			greyval += (long)floor( 0.299 * (double)picture[ ys+x+2 ] +
											0.587 * (double)picture[ ys+x+1 ] +
										   0.114 * (double)picture[ ys+x ] + 0.5 );
		}
      return( greyval );
   }
   else
   {
		samplehalf = samplesize >> 1;
		samplenum  = samplesize*samplesize;
		xstart = xcoord-samplehalf;
		xstop  = xcoord+samplehalf;
		ystart = (ycoord-samplehalf)*ystep;
		ystop  = (ycoord+samplehalf)*ystep;

		if( colorchannel == CCDLIB_RGBChannelColor )
   	{
      	xstart *= 3; xstop *= 3;
      	ystart *= 3; ystop *= 3;
      	ystep  *= 3;
   	}

		if( xcoord < samplehalf ||
			 ycoord < samplehalf ||
          xcoord >= picwidth-samplehalf ||
			 ycoord >= picheight-samplehalf )
			return( 0 );

		if( colorchannel != CCDLIB_RGBChannelColor )
		{
			for( ys=ystart; ys<=ystop; ys+=ystep )
				for( x=xstart; x<=xstop; x++ )
					greyval += (long)picture[x+ys];
		}
		else
		{
			for( ys=ystart; ys<=ystop; ys+=ystep )
				for( x=xstart; x<=xstop; x+=3 )
					greyval += (long)floor( 0.299 * (double)picture[ ys+x+2 ] +
													0.587 * (double)picture[ ys+x+1 ] +
												   0.114 * (double)picture[ ys+x ] + 0.5 );
		}
	   return( greyval/samplenum );
   }
}

/**************************************************************************
*                                                                         *
*   CCDLIB_GetCenterOfBrightness : Routine zur Koordinatenbestimmung      *
*                                  des Helligkeitsschwerpunkts            *
*                                  in einem Bild.                         *
*                                                                         *
*   Parameter:  *picturerect : Arbeitsbereich im Bild                     *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               threshold    : Beruecksichtigungsschwelle Max/Untergr.    *
*               *picpuffer   : 32-Bit-Pointer auf das aktuelle Bild       *
*               *centercoord : Position des Schwerpunkts                  *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.1 (06.05.2002 in Kirchheim)                             *
*   Written by: Georg Dittie, 1996 - 2002                                 *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetCenterOfBrightness
(
   CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
   double threshold,
	short * picture,
	CCDLIB_PositionStruct *centercoord
)
{
	double xweight, yweight, brightsum, xd, yd, pw;
	long x, ys, xstart, xstop, ystart, ystop, linestep;

	long histomaximum, histomaxpos;
	double histomaxpart, histoentropy, histolevel;
	long histolowestpos, histohighestpos;

	long * histogramfield;
   short * tempbuf;

	histogramfield = (long *)malloc( pictureparam->dynamik * sizeof(long) );
	if( histogramfield == NULL )
	{
		if( histogramfield != NULL ) free( histogramfield );
		return( CCDLIB_OutOfMemoryError );
	}

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( picture == NULL )
		return( CCDLIB_NoPointerError );

   tempbuf = picture;

	if( CCDLIB_RetrieveHistogram( picturerect, pictureparam,
			 tempbuf, histogramfield ) != CCDLIB_Successful )
		return( CCDLIB_NoPointerError );

	CCDLIB_GetMaximumInHistogram(
		histogramfield,
		pictureparam->dynamik,
		&histolowestpos,
		&histohighestpos,
		&histomaxpos,
		&histomaximum,
		&histomaxpart,
		&histoentropy );

	histolevel = ((double)histohighestpos - (double)histomaxpos) * threshold
					 + (double)histomaxpos;

	/*** The picture ist to bright ! **/

	if( histolevel >= (double)histohighestpos-10. )
	{
		free( histogramfield );
		return( CCDLIB_WrongParameterError );
	}

	xstart = picturerect->xpos;
	ystart = picturerect->ypos * pictureparam->linelength;
	xstop  = picturerect->xpos + picturerect->xsize;
	ystop  = (picturerect->ypos + picturerect->ysize) * pictureparam->linelength;
	linestep = pictureparam->linelength;

	xweight = 0.;
	yweight = 0.;
	brightsum = 0.;

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
      yd = (double)(picturerect->ypos);
		for( ys = ystart;
           ys < ystop;
			  ys += linestep )
      {
			xd = (double)xstart;
         for( x = xstart;
				  x < xstop;
              x ++ )
			{
            pw = (double)tempbuf[ ys + x ];

            if( pw >= histolevel )
				{
               xweight += (xd * pw);
					yweight += (yd * pw);

					brightsum += pw;
            }
				xd ++;
         }
			yd ++;
      }
	}
   else
	{
      xstart *= 3;
		xstop  *= 3;
      ystart *= 3;
		ystop  *= 3;
      linestep *= 3;

      /*** Gewichtung nach der ISO-Helligkeitsnorm ***/

      yd = (double)(picturerect->ypos);
		for( ys = ystart;
           ys < ystop;
			  ys += linestep )
      {
			xd = (double)(picturerect->xpos);
         for( x = xstart;
				  x < xstop;
              x += 3 )
			{
				pw = 0.299 * (double)tempbuf[ ys+x+2 ] +
					  0.587 * (double)tempbuf[ ys+x+1 ] +
					  0.114 * (double)tempbuf[ ys+x ];

            if( pw >= histolevel )
				{
               xweight += (xd * pw);
					yweight += (yd * pw);

					brightsum += pw;
            }
				xd ++;
         }
			yd ++;
      }
	}
   if( brightsum > 0. )
	{
      centercoord->xpos = (long)floor( xweight / brightsum + 0.5 );
		centercoord->ypos = (long)floor( yweight / brightsum + 0.5 );
   }
	else
   {
		centercoord->xpos = 0;
      centercoord->ypos = 0;
	}

	free( histogramfield );
	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_2DCrossCorrelation : Routine zur Koordinatenbestimmung         *
*                               des Zentralpixels eines Suchmusters       *
*                               in einem Bild.                            *
*                                                                         *
*   Diese Routine realisiert echte 2D-Kreuzkorrelation !                  *
*                                                                         *
*   Parameter:  *fitsize     : Groesse des Passmusters                    *
*               subpixelfak  : Vergroesserungsfaktor                      *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf das aktuelle Bild       *
*               *fitpuffer   : 32-Bit-Pointer auf das Suchmuster          *
*               *centercoord : Position des Schwerpunkts                  *
*               *correlation : maximaler Wert der Kreuzkorrelation        *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Das Ergebnis steht in centercoord und correlation                     *
*                                                                         *
*   Version:    2.0 (19.08.2002)                                          *
*   Written by: Georg Dittie, 1996 - 2002                                 *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_2DCrossCorrelation
(
	CCDLIB_RectsizeStruct  * fitsize,
   long subpixelfak,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	short * fitpuffer,
	CCDLIB_PositionStruct *centercoord,
	double * correlation
)
{
	long xp, ysp, xr, ysr, xpr, ypr,
		  xstart, xstop, ystart, ystop, xrstop, yrstop,
		  picrefstep, piclinestep, reflinestep, fitstep, samplestep,
		  xfithalf, yfithalf, ylinefithalf;

	double corrval, corrmax, divisor, samplenum;

   long corrxpos, corrypos;

   double refgreyval, refgreymean, refgreysqr, refsqrsum,
          picgreyval, picgreymean, picgreysqr, picsqrsum,
          picrefsqr, picrefsum;

	short * pictureptr, *piclowptr;
   short * fitbufptr;

   CCDLIB_PictureparamStruct fitparam, piclowparam;
   CCDLIB_RectangleStruct lowpassrect;

   if( picture == NULL )
	   return( CCDLIB_NoPointerError );

   piclowparam = *pictureparam;
   piclowparam.colorchannel = CCDLIB_BlackWhiteColor;

   fitstep = 4*subpixelfak;
   samplestep = 4*subpixelfak;

   fitparam = *pictureparam;
   fitparam.linelength   = fitsize->xsize;
   fitparam.columnheight = fitsize->ysize;

   xfithalf = fitsize->xsize / 2;
   yfithalf = fitsize->ysize / 2;

   samplenum = (double)(fitsize->xsize/samplestep+1)
             * (double)(fitsize->ysize/samplestep+1);

   if( (xfithalf%samplestep) != 0 || (yfithalf%samplestep) != 0 )
	   return( CCDLIB_WrongParameterError );

   piclowptr = (short *)malloc(
	   pictureparam->linelength * pictureparam->columnheight * sizeof(short) );
   if( piclowptr == NULL  )
      return( CCDLIB_OutOfMemoryError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
   {
      pictureptr = picture;
      fitbufptr  = fitpuffer;
   }
   else
   {
		pictureptr = (short *)malloc(
	      pictureparam->linelength * pictureparam->columnheight * sizeof(short) );
      fitbufptr = (short *)malloc(
         fitsize->xsize * fitsize->ysize * sizeof(short) );

      if( pictureptr == NULL || fitbufptr == NULL )
      {
	      if( piclowptr != NULL ) free( piclowptr );
	      if( pictureptr != NULL ) free( pictureptr );
	      if( fitbufptr != NULL ) free( fitbufptr );
         return( CCDLIB_OutOfMemoryError );
      }
		CCDLIB_ExtractLuminance( picture, pictureparam, pictureptr );
      CCDLIB_ExtractLuminance( fitpuffer, &fitparam, fitbufptr );
   }

   CCDLIB_InitRectangle( &lowpassrect,
      0, 0, pictureparam->linelength, pictureparam->columnheight );
   CCDLIB_QuickLowPassFilter( piclowptr, pictureptr, &piclowparam, &lowpassrect, 2 );

   xstart = xfithalf;
   ystart = yfithalf;
   xstop = pictureparam->linelength - xfithalf - 1;
   ystop = pictureparam->columnheight - yfithalf - 1;

   ystart *= pictureparam->linelength;
   ystop  *= pictureparam->linelength;

   yfithalf *= fitsize->xsize;
   ylinefithalf = fitsize->ysize/2 * pictureparam->linelength;

   /*** Grauwertsummen fuer die Referenz vorberechnen ***/

   picrefstep  = pictureparam->linelength * samplestep;
   reflinestep = fitsize->xsize * samplestep;

   xrstop = fitsize->xsize;
   yrstop = fitsize->xsize * fitsize->ysize;

   refgreymean = 0.;
   for( ysr = 0;
        ysr <= yrstop;
        ysr += reflinestep )
   {
      for( xr = 0;
           xr <= xrstop;
           xr += samplestep )
      {
         refgreymean += (double)fitbufptr[ ysr+xr ];
      }
   }
   refgreymean /= samplenum;

	refsqrsum = 0.;
   for( ysr = 0;
        ysr <= yrstop;
		  ysr += reflinestep )
	{
		for( xr = 0;
			  xr <= xrstop;
			  xr += samplestep )
		{
			refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
			refgreysqr = refgreyval * refgreyval;
			refsqrsum += refgreysqr;
		}
	}

	/*** 1. Iterationsschritt: Suche in 2er Schritten ***/

	corrmax = -1.;

	piclinestep = pictureparam->linelength * fitstep;

	for( ysp = ystart;
		  ysp < ystop;
		  ysp += piclinestep )
	{
		for( xp = xstart;
			  xp < xstop;
			  xp += fitstep )
		{
			picgreymean = 0.;
			ypr = ysp - ylinefithalf;

			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					picgreymean += (double)piclowptr[ ypr+xpr ];

					xpr += samplestep;
				}
				ypr += picrefstep;
			}
			picgreymean /= samplenum;

			picrefsum = 0.;
			picsqrsum = 0.;

			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
					picgreyval = (double)piclowptr[ ypr+xpr ] - picgreymean;

					picgreysqr = picgreyval*picgreyval;
					picsqrsum += picgreysqr;

					picrefsqr  = picgreyval*refgreyval;
					picrefsum += picrefsqr;

					xpr += samplestep;
				}
				ypr += picrefstep;
			}

			divisor = refsqrsum * picsqrsum;
			if( divisor > 0. )
				corrval = picrefsum / sqrt( divisor );
			else
				corrval = 0.;

			if( corrval > corrmax )
			{
				corrmax = corrval;
				corrxpos = xp;
				corrypos = ysp;
			}
		}
	}

	/*** Zweiter Iterationsschritt: pixelgenaue Suche ***/

	corrmax = -1.;

	piclinestep = pictureparam->linelength;

	xstart = corrxpos - fitstep;
	xstop  = corrxpos + fitstep;
	ystart = corrypos - fitstep*pictureparam->linelength;
	ystop  = corrypos + fitstep*pictureparam->linelength;

	for( ysp = ystart;
		  ysp <= ystop;
		  ysp += piclinestep )
	{
		for( xp = xstart;
			  xp <= xstop;
			  xp ++ )
		{
			picgreymean = 0.;
			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					picgreymean += (double)pictureptr[ ypr+xpr ];

					xpr += samplestep;
				}
				ypr += picrefstep;
			}
			picgreymean /= samplenum;

			picrefsum = 0.;
			picsqrsum = 0.;

			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
					picgreyval = (double)pictureptr[ ypr+xpr ] - picgreymean;

					picgreysqr = picgreyval*picgreyval;
					picsqrsum += picgreysqr;

					picrefsqr  = picgreyval*refgreyval;
					picrefsum += picrefsqr;

					xpr += samplestep;
				}
				ypr += picrefstep;
			}

			divisor = refsqrsum * picsqrsum;
			if( divisor > 0. )
				corrval = picrefsum / sqrt( divisor );
			else
				corrval = 0.;

			if( corrval > corrmax )
			{
				corrmax = corrval;
				corrxpos = xp;
				corrypos = ysp;
			}
		}
	}

	centercoord->xpos = corrxpos;
	centercoord->ypos = corrypos/pictureparam->linelength;

   *correlation = corrmax;

	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
   {
		free( pictureptr );
		free( fitbufptr );
	}
   free( piclowptr );
	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_2DQuickCorrelation : Routine zur Koordinatenbestimmung         *
*                               des Zentralpixels eines Suchmusters       *
*                               in einem Bild.                            *
*                                                                         *
*   Diese Routine realisiert echte 2D-Kreuzkorrelation, wobei bei der     *
*   Grobsuche beim ersten gefundenen Maximum gleich zum Feinsuchen        *
*   gegangen wird. Es mu also ein eindeutiges Maximum geben              *
*                                                                         *
*   Parameter:  *fitsquare   : Mae & Startposition des Passmusters       *
*               searchradius : Suchabstand um Startpunkt                  *
*               subpixelfak  : Vergroesserungsfaktor                      *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf das aktuelle Bild       *
*               *fitpuffer   : 32-Bit-Pointer auf das Suchmuster          *
*               *centercoord : Position des Schwerpunkts                  *
*               *correlation : maximaler Wert der Kreuzkorrelation        *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Das Ergebnis steht in centercoord und correlation                     *
*                                                                         *
*   Version:    1.0 vom vllig verregneten Sommer 2002                    *
*               1.1 vom schnen sonnigen Sommer 2003                      *
*   Written by: Georg Dittie, 2002-2003                                   *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_2DQuickCorrelation
(
	CCDLIB_RectangleStruct *fitsquare,
   long searchradius,
   long subpixelfak,
	CCDLIB_PictureparamStruct *pictureparam,
	short *picture,
	short *fitpuffer,
	CCDLIB_PositionStruct *centercoord,
	double *correlation
)
{
	long xp, ysp, xr, ysr, xpr, ypr,
		  xstart, xstop, ystart, ystop, xrstop, yrstop,
		  picrefstep, piclinestep, reflinestep, fitstep, samplestep,
		  xfithalf, yfithalf, ylinefithalf,
        yfit, yistart, i, ilimit, xfit, j, jlimit;

	double corrval, corrmax, divisor, samplenum;

   long corrxpos, corrypos;

   double refgreyval, refgreymean, refgreysqr, refsqrsum,
          picgreyval, picgreymean, picgreysqr, picsqrsum,
          picrefsqr, picrefsum;

	short * pictureptr, *piclowptr;
   short * fitbufptr;

   CCDLIB_PictureparamStruct fitparam, piclowparam;
   CCDLIB_RectangleStruct lowpassrect;

   if( picture == NULL )
	   return( CCDLIB_NoPointerError );

   piclowparam = *pictureparam;
   piclowparam.colorchannel = CCDLIB_BlackWhiteColor;

   fitstep = 2*subpixelfak;
   samplestep = 2*subpixelfak;

   fitparam = *pictureparam;
   fitparam.linelength   = fitsquare->xsize;
   fitparam.columnheight = fitsquare->ysize;

   xfithalf = fitsquare->xsize / 2;
   yfithalf = fitsquare->ysize / 2;

   samplenum = (double)(fitsquare->xsize/samplestep+1)
             * (double)(fitsquare->ysize/samplestep+1);

   if( (xfithalf%samplestep) != 0 || (yfithalf%samplestep) != 0 )
	   return( CCDLIB_WrongParameterError );

   piclowptr = (short *)malloc(
	   pictureparam->linelength * pictureparam->columnheight * sizeof(short) );
   if( piclowptr == NULL  )
      return( CCDLIB_OutOfMemoryError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
   {
      pictureptr = picture;
      fitbufptr  = fitpuffer;
   }
   else
   {
		pictureptr = (short *)malloc(
	      pictureparam->linelength * pictureparam->columnheight * sizeof(short) );
      fitbufptr = (short *)malloc(
         fitsquare->xsize * fitsquare->ysize * sizeof(short) );

      if( pictureptr == NULL || fitbufptr == NULL )
      {
	      if( piclowptr != NULL ) free( piclowptr );
	      if( pictureptr != NULL ) free( pictureptr );
	      if( fitbufptr != NULL ) free( fitbufptr );
         return( CCDLIB_OutOfMemoryError );
      }
		CCDLIB_ExtractLuminance( picture, pictureparam, pictureptr );
      CCDLIB_ExtractLuminance( fitpuffer, &fitparam, fitbufptr );
   }

   CCDLIB_InitRectangle( &lowpassrect,
      0, 0, pictureparam->linelength, pictureparam->columnheight );
   CCDLIB_QuickLowPassFilter( piclowptr, pictureptr, &piclowparam, &lowpassrect, 2 );

   xfit = fitsquare->xpos+fitsquare->xsize/2;
   yfit = fitsquare->ypos+fitsquare->ysize/2;

   xstart = xfit-searchradius;
   if( xstart < xfithalf ) xstart = xfithalf;
   ystart = yfit-searchradius;
   if( ystart < yfithalf ) ystart = yfithalf;
   xstop = xfit+searchradius;
   if( xstart > pictureparam->linelength-xfithalf-1 )
   	xstop = pictureparam->linelength-xfithalf-1;
   ystop = yfit+searchradius;
   if( ystart > pictureparam->columnheight-yfithalf-1 )
   	ystop = pictureparam->columnheight-yfithalf-1;

   ilimit = searchradius*2;
   jlimit = searchradius*2;

   ystart *= pictureparam->linelength;
   ystop  *= pictureparam->linelength;

   yistart = yfit*pictureparam->linelength;

   yfithalf *= fitsquare->xsize;
   ylinefithalf = fitsquare->ysize/2 * pictureparam->linelength;

   /*** Grauwertsummen fuer die Referenz vorberechnen ***/

   picrefstep  = pictureparam->linelength * samplestep;
   reflinestep = fitsquare->xsize * samplestep;

   xrstop = fitsquare->xsize;
   yrstop = fitsquare->xsize * fitsquare->ysize;

   refgreymean = 0.;
   for( ysr = 0;
        ysr <= yrstop;
        ysr += reflinestep )
   {
      for( xr = 0;
           xr <= xrstop;
           xr += samplestep )
      {
         refgreymean += (double)fitbufptr[ ysr+xr ];
      }
   }
   refgreymean /= samplenum;

	refsqrsum = 0.;
   for( ysr = 0;
        ysr <= yrstop;
		  ysr += reflinestep )
	{
		for( xr = 0;
			  xr <= xrstop;
			  xr += samplestep )
		{
			refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
			refgreysqr = refgreyval * refgreyval;
			refsqrsum += refgreysqr;
		}
	}

	/*** 1. Iterationsschritt: Suche im abwechselnden Zeilenfortschritt mit Abbruch bei corr = 0,9 ***/

	corrmax = -1.;

	piclinestep = pictureparam->linelength * fitstep;

	for( i = 0;
		  i < ilimit;
		  i++ )
	{
      if( (i%2) == 0 )
      {
         ysp = yistart+(i/2)*piclinestep;
         if( ysp >= ystop ) continue;
      }
      else
      {
         ysp = yistart-((i+1)/2)*piclinestep;
         if( ysp < ystart ) continue;
      }

	   for( j = 0;
		     j < jlimit;
		     j++ )
	   {
         if( (j%2) == 0 )
         {
            xp = xfit+(j/2)*fitstep;
            if( xp >= xstop ) continue;
         }
         else
         {
            xp = xfit-((j+1)/2)*fitstep;
            if( xp < xstart ) continue;
         }

			picgreymean = 0.;
			ypr = ysp - ylinefithalf;

			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					picgreymean += (double)piclowptr[ ypr+xpr ];

					xpr += samplestep;
				}
				ypr += picrefstep;
			}
			picgreymean /= samplenum;

			picrefsum = 0.;
			picsqrsum = 0.;

			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
					picgreyval = (double)piclowptr[ ypr+xpr ] - picgreymean;

					picgreysqr = picgreyval*picgreyval;
					picsqrsum += picgreysqr;

					picrefsqr  = picgreyval*refgreyval;
					picrefsum += picrefsqr;

					xpr += samplestep;
				}
				ypr += picrefstep;
			}

			divisor = refsqrsum * picsqrsum;
			if( divisor > 0. )
				corrval = picrefsum / sqrt( divisor );
			else
				corrval = 0.;

			if( corrval > corrmax )
			{
				corrmax = corrval;
				corrxpos = xp;
				corrypos = ysp;
			}
         if( corrmax >= 0.85 ) break;
		}
      if( corrmax >= 0.85 ) break;
	}

	/*** Zweiter Iterationsschritt: pixelgenaue Suche ***/

	corrmax = -1.;

	piclinestep = pictureparam->linelength;

	xstart = corrxpos - fitstep;
	xstop  = corrxpos + fitstep;
	ystart = corrypos - fitstep*pictureparam->linelength;
	ystop  = corrypos + fitstep*pictureparam->linelength;

	for( ysp = ystart;
		  ysp <= ystop;
		  ysp += piclinestep )
	{
		for( xp = xstart;
			  xp <= xstop;
			  xp ++ )
		{
			picgreymean = 0.;
			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					picgreymean += (double)pictureptr[ ypr+xpr ];

					xpr += samplestep;
				}
				ypr += picrefstep;
			}
			picgreymean /= samplenum;

			picrefsum = 0.;
			picsqrsum = 0.;

			ypr = ysp - ylinefithalf;
			for( ysr = 0;
				  ysr <= yrstop;
				  ysr += reflinestep )
			{
				xpr = xp - xfithalf;
				for( xr = 0;
					  xr <= xrstop;
					  xr += samplestep )
				{
					refgreyval = (double)fitbufptr[ ysr+xr ] - refgreymean;
					picgreyval = (double)pictureptr[ ypr+xpr ] - picgreymean;

					picgreysqr = picgreyval*picgreyval;
					picsqrsum += picgreysqr;

					picrefsqr  = picgreyval*refgreyval;
					picrefsum += picrefsqr;

					xpr += samplestep;
				}
				ypr += picrefstep;
			}

			divisor = refsqrsum * picsqrsum;
			if( divisor > 0. )
				corrval = picrefsum / sqrt( divisor );
			else
				corrval = 0.;

			if( corrval > corrmax )
			{
				corrmax = corrval;
				corrxpos = xp;
				corrypos = ysp;
			}
		}
	}

	centercoord->xpos = corrxpos;
	centercoord->ypos = corrypos/pictureparam->linelength;

   *correlation = corrmax;

	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
   {
		free( pictureptr );
		free( fitbufptr );
	}
   free( piclowptr );
	return( CCDLIB_Successful );
}

/*** Algorithmen zum Sussmann-Operator ***/

/************************************************************************
*																								*
*	CCDLIB_CreateMask erzeugt in einem Maskenpuffer eine Bearbeitungs-   *
*  maske fuer Bildverarbeitungsprozeduren an Planetenaufnahmen          *
*                                                                       *
*   Version:    1.1 (Idee CCD-Workshop in Mariazell, Neujahrstag 2002)  *
*   Written by: Georg Dittie, 1998 - 2002                               *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CreateMask(
   double *histoentropy,
   double *contrastrange,
	char **mask,
   CCDLIB_RectangleStruct *checkrect,
	short *picbuffer,
	CCDLIB_PictureparamStruct *picparam,
	double masklevel )
{
//static long debugcount=0;

	long brightval, buflength;
	long x, y, xs, ys, yd, xstart, xstop, ystart, ystop,
		  ystep, ysstep,
        xm1, xp1, ym1, yp1, i,
        leftlimit, rightlimit, upperlimit, lowerlimit;

	long histomaximum, histomaxpos, fwhmlevel;
	double histomaxpart;
	long histolowestpos, histohighestpos, histohighpos, histolowpos;
	long * histogramfield;
	CCDLIB_RectangleStruct picrect;
   char *puffer;

	if( picbuffer == NULL )
		return( CCDLIB_NoPointerError );

	if( masklevel > 1. ) masklevel = 1.;
	if( masklevel < 0.1 ) masklevel = 0.1;

   *histoentropy = 0;
   *contrastrange = 0;

	buflength = picparam->linelength*picparam->columnheight;
	(*mask) = (char *)malloc( buflength * sizeof(char) );
	histogramfield = (long *)malloc( picparam->dynamik * sizeof(long) );

	if( histogramfield == NULL || (*mask) == NULL )
	{
		if( histogramfield != NULL ) free( histogramfield );
		if( (*mask) != NULL ) free( *mask );
		return( CCDLIB_OutOfMemoryError );
	}

   memset( (*mask), 0, buflength * sizeof(char) );

	CCDLIB_InitRectangle(
		&picrect, CCDLIB_MINIMALRIM, CCDLIB_MINIMALRIM,
      picparam->linelength-CCDLIB_MINIMALRIM*2, picparam->columnheight-CCDLIB_MINIMALRIM*2 );
	if( CCDLIB_RetrieveHistogram( &picrect, picparam,
			 picbuffer, histogramfield ) != CCDLIB_Successful )
	{
		if( histogramfield != NULL ) free( histogramfield );
		if( (*mask) != NULL ) free( *mask );
		return( CCDLIB_NoPointerError );
	}

	CCDLIB_EvaluateHistogram(
		histogramfield,
		picparam->dynamik,
		CCDLIB_Histo99_8Const,
		0,
		0,
		&histolowestpos,
		&histohighestpos,
		&histolowpos,
		&histohighpos,
		&histomaxpos,
		&histomaximum,
		&histomaxpart,
		histoentropy );

   *contrastrange = (double)histohighpos - (double)histomaxpos;
	fwhmlevel = (long)floor( *contrastrange * masklevel
					 + (double)histomaxpos + 0.5 );

	xstart = picrect.xpos;
	ystart = picrect.ypos;
	xstop  = picrect.xpos+picrect.xsize;
	ystop  = picrect.ypos+picrect.ysize;
	ystart *= picparam->linelength;
	ystop  *= picparam->linelength;
	ystep  = picparam->linelength;

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for( y = ystart; y < ystop; y += ystep )
		{
			for( x = xstart; x < xstop; x++ )
			{
				if( picbuffer[ y+x ] > fwhmlevel )
					(*mask)[ y+x ] = 0x7F;
			}
		}
	}
	else
	{
		ysstep = ystep*3;

		for( yd = ystart, ys = ystart*3;
			  yd < ystop;
			  yd += ystep, ys += ysstep )
		{
			for( x = xstart, xs = xstart*3;
				  x < xstop;
				  x++, xs += 3 )
			{
				brightval = (long)floor(0.5+
					0.299 * (double)picbuffer[ ys+xs+2 ] +
					0.587 * (double)picbuffer[ ys+xs+1 ] +
					0.114 * (double)picbuffer[ ys+xs ] );

				if( brightval > fwhmlevel )
					(*mask)[ yd+x ] = 0x7F;
			}
		}
	}

	free( histogramfield );


	puffer = (char *)malloc( buflength * sizeof(char) );
	if( puffer != NULL )
	{
      memcpy( puffer, (*mask), buflength * sizeof(char) );

   	xstart = picrect.xpos+1;
	   ystart = picrect.ypos+1;
	   xstop  = picrect.xpos+picrect.xsize-1;
	   ystop  = picrect.ypos+picrect.ysize-1;
	   ystart *= picparam->linelength;
	   ystop  *= picparam->linelength;
	   ystep  = picparam->linelength;

      /*** Erosion zur Rauschentfernung */

	   for( y = ystart; y < ystop; y += ystep )
	   {
         ym1 = y-ystep;
         yp1 = y+ystep;
		   for( x = xstart; x < xstop; x++ )
		   {
            xm1 = x-1;
            xp1 = x+1;
			   if( (*mask)[ y+x ] == 0 )
            {
               puffer[ ym1+xm1 ] = 0;
               puffer[ ym1+x   ] = 0;
               puffer[ ym1+xp1 ] = 0;
               puffer[ y  +xm1 ] = 0;
               puffer[ y  +x   ] = 0;
               puffer[ y  +xp1 ] = 0;
               puffer[ yp1+xm1 ] = 0;
               puffer[ yp1+x   ] = 0;
               puffer[ yp1+xp1 ] = 0;
            }
         }
      }
      memcpy( (*mask), puffer, buflength * sizeof(char) );

      /*** 4 Dilatationen zur Randermittlung ***/

      for( i=0; i<4; i++ )
      {
		   for( y = ystart; y < ystop; y += ystep )
		   {
            ym1 = y-ystep;
            yp1 = y+ystep;
			   for( x = xstart; x < xstop; x++ )
			   {
               xm1 = x-1;
               xp1 = x+1;
				   if( (*mask)[ y+x ] != 0 )
               {
                  puffer[ ym1+xm1 ] = 0x7f;
                  puffer[ ym1+x   ] = 0x7f;
                  puffer[ ym1+xp1 ] = 0x7f;
                  puffer[ y  +xm1 ] = 0x7f;
                  puffer[ y  +x   ] = 0x7f;
                  puffer[ y  +xp1 ] = 0x7f;
                  puffer[ yp1+xm1 ] = 0x7f;
                  puffer[ yp1+x   ] = 0x7f;
                  puffer[ yp1+xp1 ] = 0x7f;
               }
            }
			}
         memcpy( (*mask), puffer, buflength * sizeof(char) );
		}
      free( puffer );
	}
  	xstart = picrect.xpos;
   ystart = picrect.ypos;
   xstop  = picrect.xpos+picrect.xsize;
   ystop  = picrect.ypos+picrect.ysize;
   ystart *= picparam->linelength;
   ystop  *= picparam->linelength;
   ystep  = picparam->linelength;

   leftlimit  = xstop-1;
   rightlimit = 0;
   for( y = ystart; y < ystop; y += ystep )
	{
	   for( x = xstart; x < xstop; x++ )
	   {
		   if( (*mask)[ y+x ] != 0 )
         {
            if( x<leftlimit ) leftlimit = x;
            if( x>rightlimit ) rightlimit = x;
         }
      }
   }

   upperlimit = ystop-picparam->linelength;
   lowerlimit = 0;
   for( x = xstart; x < xstop; x++ )
   {
      for( y = ystart; y < ystop; y += ystep )
	   {
		   if( (*mask)[ y+x ] != 0 )
         {
            if( y<upperlimit ) upperlimit = y;
            if( y>lowerlimit ) lowerlimit = y;
         }
      }
   }
   checkrect->xpos = leftlimit;
   checkrect->ypos = upperlimit/picparam->linelength;
   checkrect->xsize = rightlimit-leftlimit;
   checkrect->ysize = (lowerlimit-upperlimit)/picparam->linelength;
/*
sprintf( debuglog, "F:\\pic%04ld.bmp", debugcount );
SaveBitmapOnDisk( debuglog, (unsigned char *)(*mask),picparam->linelength,picparam->columnheight,8);
debugcount++;

sprintf( debuglog, "createmask X %3ld  Y %3ld  W %3ld  H %3ld",
checkrect->xpos, checkrect->ypos, checkrect->xsize, checkrect->ysize );
Debug( debuglog );
*/
  	return( CCDLIB_Successful );
}


/************************************************************************
*																								*
*	CCDLIB_EvaluatePicQuality bewertet CCD-Aufnahmen nach diversen       *
*  Bewertungskriterien:                                                 *
*                                                                       *
*  HINWEIS: Die Idee zu dieser Routine enstand bei einer hitzigen       *
*  Diskussion mit Fritz Sussmann in einer Mariazeller Pizzeria :-)      *
*                                                                       *
*	Rauschanteil im 5x5 Quadrat ( Standardabweichung )                   *
*  Mittelwert des Sobeloperators in der Maske                           *
*  Maximum des Sobeloperators in der Maske                              *
*  Belichtungsdynamik                                                   *
*                                                                       *
*  Return: TRUE, wenn Minimalbedingungen erfuellt werden                *
*                                                                       *
*   Version:    1.3 (Idee Mariazell 1998, Neujahr 2002)                 *
*   Written by: Georg Dittie, 1998 - 2002                               *
*                                                                       *
************************************************************************/

BOOL CCDLIB_EvaluatePicQuality(
	double *sobelmax,
	double *sobelmean,
	double *noiselevel,
	double *sharpindex,
   double *histoentropy,
	double *contrastval,
   double *signaltonoise,
   CCDLIB_RectangleStruct *checkrect,
	short *picbuffer,
	CCDLIB_PictureparamStruct *picparam,
   CCDLIB_RectangleStruct *evalrect )
{
	long x, ys, xks, yks, y1, coord, mcoord, pcoord, maskcount,
		  buflength, xstart, xstop, ysstart, ysstop, ysstep,
        xkstart, xkstop, ykstart, ykstop, xm, ym;
	double val, dmean, sigma, sigmar, sigmag, sigmab, sigmasum, contrastrange,
          sobelval, sobelsum, xgrad, ygrad;

	long histomaximum, histomaxpos;
	double histomaxpart;
	long histolowestpos, histohighestpos, histohighpos, histolowpos;
	CCDLIB_RectangleStruct picrect;

   BOOL maskcreate;

	short *tempbuf;
	long  *histogramfield;
   char  *mask;

	/* Vorbereitungen */

	*sobelmax = 0.;
	*sobelmean = 0.;
	*noiselevel = 0.;
	*sharpindex = 0.;
	*contrastval = 0.;
   *histoentropy = 0.;
   *signaltonoise = 0.;

 	buflength = picparam->linelength*picparam->columnheight;
	if( picparam->colorchannel == CCDLIB_RGBChannelColor )
		buflength *= 3;

	tempbuf = (short*)malloc( buflength*sizeof(short) );
	if( tempbuf == NULL  )
		return( FALSE );

   /*** Kontrastwerte und Pruefbereich ermitteln ***/

   if( evalrect == NULL )
   {
      histogramfield = NULL;

      CCDLIB_InitRectangle(
	      &picrect, CCDLIB_MINIMALRIM, CCDLIB_MINIMALRIM,
         picparam->linelength-CCDLIB_MINIMALRIM*2,
         picparam->columnheight-CCDLIB_MINIMALRIM*2 );

      if( CCDLIB_CreateMask(
         histoentropy, &contrastrange, &mask, checkrect,
         picbuffer, picparam, 0.4 ) != CCDLIB_Successful )
      {
         if( tempbuf != NULL ) free( tempbuf );
         return( FALSE );
      }
      maskcreate = TRUE;
   }
   else
   {
      *checkrect = *evalrect;
      picrect = *evalrect;
      mask = NULL;
      maskcreate = FALSE;

	   histogramfield = (long *)malloc( picparam->dynamik * sizeof(long) );
	   if( histogramfield == NULL )
      {
         if( tempbuf != NULL ) free( tempbuf );
         if( histogramfield != NULL ) free( histogramfield );
		   return( FALSE );
      }

      if( CCDLIB_RetrieveHistogram( &picrect, picparam,
	   		 picbuffer, histogramfield ) != CCDLIB_Successful )
	   {
         if( tempbuf != NULL ) free( tempbuf );
		   if( histogramfield != NULL ) free( histogramfield );
		   return( FALSE );
	   }

	   CCDLIB_EvaluateHistogram(
		   histogramfield,
		   picparam->dynamik,
		   CCDLIB_Histo99_8Const,
		   0,
		   0,
		   &histolowestpos,
		   &histohighestpos,
		   &histolowpos,
		   &histohighpos,
		   &histomaxpos,
		   &histomaximum,
		   &histomaxpart,
		   histoentropy );

      contrastrange = (double)histohighpos - (double)histomaxpos;
   }

	/* 2. Schritt Filterung */

   CCDLIB_QuickLowPassFilter( tempbuf, picbuffer, picparam, checkrect, 5 );

	/*** 3. Schritt: Auswertung des gefilterten Puffers / Rauschanalyse */

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
	   ysstep  = picparam->linelength;

		sigmasum  = 0.;
      maskcount = 0;
      sobelsum = 0.;
      *sobelmax = 0.;

		if( maskcreate == TRUE )
      {
         xstart  = checkrect->xpos;
		   xstop   = checkrect->xpos+checkrect->xsize;
		   ysstart = checkrect->ypos*picparam->linelength;
		   ysstop  = (checkrect->ypos+checkrect->ysize)*picparam->linelength;

		   for( ys = ysstart; ys < ysstop; ys += ysstep )
		   {
            ykstart = ys-ysstep;
            ykstop  = ys+ysstep;
			   for( x = xstart; x < xstop; x++ )
			   {
               if( mask[ ys+x ] != 0 )
               {
                  y1 = ys+x;
                  xkstart = x-1;
                  xkstop  = x+1;

                  xgrad = ((double)tempbuf[y1+1]-(double)tempbuf[y1-1]);
         	      ygrad = ((double)tempbuf[y1+ysstep]-(double)tempbuf[y1-ysstep]);
                  sobelval = sqrt( xgrad*xgrad+ygrad*ygrad )*0.5;

                  if( sobelval > *sobelmax ) *sobelmax = sobelval;
                  sobelsum += sobelval;

                  dmean = (double)tempbuf[y1];
                  sigma = 0.;
                  for( yks = ykstart; yks <= ykstop; yks += ysstep )
                  {
                     for( xks = xkstart; xks <= xkstop; xks++ )
                     {
                        val = (double)picbuffer[yks+xks]-dmean;
                        sigma += val*val;
                     }
                  }
                  sigmasum += sqrt( sigma/8. );

                  maskcount++;
   		      }
            }
   	   }
      }
      else
      {
         xstart  = evalrect->xpos;
		   xstop   = evalrect->xpos+evalrect->xsize;
		   ysstart = evalrect->ypos*picparam->linelength;
		   ysstop  = (evalrect->ypos+evalrect->ysize)*picparam->linelength;

		   for( ys = ysstart; ys < ysstop; ys += ysstep )
		   {
            ykstart = ys-ysstep;
            ykstop  = ys+ysstep;
			   for( x = xstart; x < xstop; x++ )
			   {
         	   y1 = ys+x;
               xkstart = x-1;
               xkstop  = x+1;

               xgrad = ((double)tempbuf[y1+1]-(double)tempbuf[y1-1]);
         	   ygrad = ((double)tempbuf[y1+ysstep]-(double)tempbuf[y1-ysstep]);
               sobelval = sqrt( xgrad*xgrad+ygrad*ygrad )*0.5;

               if( sobelval > *sobelmax ) *sobelmax = sobelval;
               sobelsum += sobelval;

               dmean = (double)tempbuf[y1];
               sigma = 0.;
               for( yks = ykstart; yks <= ykstop; yks += ysstep )
               {
                  for( xks = xkstart; xks <= xkstop; xks++ )
                  {
                     val = (double)picbuffer[yks+xks]-dmean;
                     sigma += val*val;
                  }
               }
               sigmasum += sqrt( sigma/8. );

               maskcount++;
   		   }
   	   }
      }
   }
   else
   {
	   ysstep  = picparam->linelength*3;

		sigmasum  = 0.;
      maskcount = 0;
      sobelsum = 0.;
      *sobelmax = 0.;

		if( maskcreate == TRUE )
      {
         xstart  = checkrect->xpos;
		   xstop   = (checkrect->xpos+checkrect->xsize)*3;
		   ysstart = checkrect->ypos*picparam->linelength;
		   ysstop  = (checkrect->ypos+checkrect->ysize)*picparam->linelength*3;

   		for( ys = ysstart*3, ym = ysstart;
              ys < ysstop;
              ys += ysstep, ym += picparam->linelength)
	   	{
		   	for( x = xstart*3, xm = xstart;
                 x < xstop;
                 x += 3, xm++ )
			   {
               if( mask[ ym+xm ] != 0 )
               {
                  coord = ys+x; mcoord = coord-ysstep; pcoord = coord+ysstep;
         	      xgrad = (  0.114*(double)(tempbuf[coord+3]-tempbuf[coord-3])
                           + 0.587*(double)(tempbuf[coord+4]-tempbuf[coord-2])
                           + 0.299*(double)(tempbuf[coord+5]-tempbuf[coord-1]));

         	      ygrad = (  0.114*(double)(tempbuf[pcoord]-tempbuf[mcoord])
                           + 0.587*(double)(tempbuf[pcoord+1]-tempbuf[mcoord+1])
                           + 0.299*(double)(tempbuf[pcoord+2]-tempbuf[mcoord+2]));

                  sobelval = sqrt( xgrad*xgrad+ygrad*ygrad );

                  if( sobelval > *sobelmax ) *sobelmax = sobelval;
                  sobelsum += sobelval;

                  dmean = (double)tempbuf[coord];
                  sigmab = 0.;
                  sigmag = 0.;
                  sigmar = 0.;
                  ykstart = ys-ysstep;
                  ykstop  = ys+ysstep;
                  for( yks = ykstart; yks <= ykstop; yks += ysstep )
                  {
                     xkstart = x-3;
                     xkstop  = x+3;
                     for( xks = xkstart; xks <= xkstop; xks += 3 )
                     {
                        val = (double)picbuffer[yks+xks]-dmean;
                        sigmab += val*val;
                        val = (double)picbuffer[yks+xks+1]-dmean;
                        sigmag += val*val;
                        val = (double)picbuffer[yks+xks+2]-dmean;
                        sigmar += val*val;
                     }
                  }
                  sigmasum += ( 0.114*sqrt( sigmab/8. )
                              + 0.587*sqrt( sigmag/8. )
                              + 0.299*sqrt( sigmar/8. ));

                  maskcount++;
               }
   		   }
   	   }
      }
      else
      {
         xstart  = evalrect->xpos*3;
		   xstop   = (evalrect->xpos+evalrect->xsize)*3;
		   ysstart = evalrect->ypos*picparam->linelength*3;
		   ysstop  = (evalrect->ypos+evalrect->ysize)*picparam->linelength*3;

   		for( ys = ysstart; ys < ysstop; ys += ysstep )
	   	{
		   	for( x = xstart; x < xstop; x += 3 )
			   {
               coord = ys+x; mcoord = coord-ysstep; pcoord = coord+ysstep;
         	   xgrad = (  0.114*(double)(tempbuf[coord+3]-tempbuf[coord-3])
                        + 0.587*(double)(tempbuf[coord+4]-tempbuf[coord-2])
                        + 0.299*(double)(tempbuf[coord+5]-tempbuf[coord-1]));

         	   ygrad = (  0.114*(double)(tempbuf[pcoord]-tempbuf[mcoord])
                        + 0.587*(double)(tempbuf[pcoord+1]-tempbuf[mcoord+1])
                        + 0.299*(double)(tempbuf[pcoord+2]-tempbuf[mcoord+2]));

               sobelval = sqrt( xgrad*xgrad+ygrad*ygrad );

               if( sobelval > *sobelmax ) *sobelmax = sobelval;
               sobelsum += sobelval;

               dmean = (double)tempbuf[coord];
               sigmab = 0.;
               sigmag = 0.;
               sigmar = 0.;
               ykstart = ys-ysstep;
               ykstop  = ys+ysstep;
               for( yks = ykstart; yks <= ykstop; yks += ysstep )
               {
                  xkstart = x-3;
                  xkstop  = x+3;
                  for( xks = xkstart; xks <= xkstop; xks += 3 )
                  {
                     val = (double)picbuffer[yks+xks]-dmean;
                     sigmab += val*val;
                     val = (double)picbuffer[yks+xks+1]-dmean;
                     sigmag += val*val;
                     val = (double)picbuffer[yks+xks+2]-dmean;
                     sigmar += val*val;
                  }
               }
               sigmasum += ( 0.114*sqrt( sigmab/8. )
                           + 0.587*sqrt( sigmag/8. )
                           + 0.299*sqrt( sigmar/8. ));

               maskcount++;
   		   }
   	   }
      }
   }
   if( maskcount > 0 )
   {
      *sobelmean = sobelsum / (double)maskcount;
      *noiselevel = sigmasum / (double)maskcount;
      *contrastval = contrastrange / (double)picparam->dynamik;
      if( *noiselevel != 0 )
   	   *signaltonoise = contrastrange / *noiselevel;
      else
         *signaltonoise = 0.;
   }

	free( tempbuf );
   if( histogramfield != NULL ) free( histogramfield );
   if( mask != NULL ) free( mask );

/*
sprintf( debuglog, "sobelmax = %lf", *sobelmax ); Debug( debuglog );
sprintf( debuglog, "sobelmean = %lf", *sobelmean ); Debug( debuglog );
sprintf( debuglog, "noiselevel = %lf", *noiselevel ); Debug( debuglog );
sprintf( debuglog, "contrastval = %lf", *contrastval ); Debug( debuglog );
sprintf( debuglog, "entropy = %lf", *histoentropy ); Debug( debuglog );
sprintf( debuglog, "signaltonoise = %lf", *signaltonoise ); Debug( debuglog );
*/
	return( TRUE );
}

/*** Hilfsfunktion fr Koerrelation und Qualitycheck ***/

void CCDLIB_QuickLowPassFilter(
   short *tempbuf,
   short *picbuffer,
   CCDLIB_PictureparamStruct *picparam,
   CCDLIB_RectangleStruct *checkrect,
   long width )
{
   long x, y, ys, xstart, xstop, ystart, ystop, ysstart, ysstop, xstop2,
        ysstep, ysst1, ysst2, ysst3, ysst4, coord;
   long *linbuf, *colbuf;

   if( width > 2 ) width = 4;
   else width = 2;

   if( checkrect->xpos < width ) checkrect->xpos = width;
   if( checkrect->ypos < width ) checkrect->ypos = width;
   if( checkrect->xpos+checkrect->xsize >= picparam->linelength-width )
      checkrect->xsize = picparam->linelength-checkrect->xpos-width-1;
   if( checkrect->ypos+checkrect->ysize >= picparam->columnheight-width )
      checkrect->ysize = picparam->columnheight-checkrect->ypos-width-1;

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
      linbuf = (long *)malloc( picparam->linelength * sizeof(long) );
      colbuf = (long *)malloc( picparam->columnheight * sizeof(long) );
      if( linbuf == NULL || colbuf == NULL )
      {
         if( linbuf != NULL ) free( linbuf );
         if( colbuf != NULL ) free( colbuf );
         return;
      }

      xstart  = checkrect->xpos;
	   xstop   = checkrect->xpos+checkrect->xsize;
	   ysstart = checkrect->ypos*picparam->linelength;
	   ysstop  = (checkrect->ypos+checkrect->ysize)*picparam->linelength;
	   ysstep  = picparam->linelength;

      if( width == 4 )
      {
         if( xstart < 4 ) xstart = 4;
	      if( xstop > picparam->linelength-4 ) xstop = picparam->linelength-4;

		   for( ys = ysstart; ys < ysstop; ys += ysstep )
		   {
			   for( x = xstart; x < xstop; x++ )
			   {
               coord = ys+x;
               linbuf[ x ] =
                  (( picbuffer[coord-4]+picbuffer[coord-3]
                    +picbuffer[coord-2]+picbuffer[coord-1]
                    +picbuffer[coord+1]+picbuffer[coord+2]
                    +picbuffer[coord+3]+picbuffer[coord+4])>>3);
            }
			   for( x = xstart; x < xstop; x++ )
               tempbuf[ ys+x ] = (short)linbuf[x];
         }
         xstart = checkrect->xpos;
	      xstop  = checkrect->xpos+checkrect->xsize;
	      ystart = checkrect->ypos*picparam->linelength;
	      ystop  = checkrect->ypos+checkrect->ysize;

         ysst1 = ysstep; ysst2 = ysstep*2; ysst3 = ysstep*3; ysst4 = ysstep*4;

         if( ystart < 4 ) ystart = 4;
	      if( ystop > picparam->columnheight-4)
            ystop = picparam->columnheight-4;
         ysstart = ystart*picparam->linelength;

   	   for( x = xstart; x < xstop; x++ )
		   {
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
		      {
               coord = ys+x;
               colbuf[ y ] =
                  (( tempbuf[coord-ysst4]+tempbuf[coord-ysst3]
                    +tempbuf[coord-ysst2]+tempbuf[coord-ysst1]
                    +tempbuf[coord+ysst1]+tempbuf[coord+ysst2]
                    +tempbuf[coord+ysst3]+tempbuf[coord+ysst4])>>3);
            }
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
                tempbuf[ ys+x ] = (short)colbuf[ y ];
         }
      }
      else
      {
         if( xstart < 2 ) xstart = 2;
	      if( xstop > picparam->linelength-2 ) xstop = picparam->linelength-2;

		   for( ys = ysstart; ys < ysstop; ys += ysstep )
		   {
			   for( x = xstart; x < xstop; x++ )
			   {
               coord = ys+x;
               linbuf[ x ] =
                  (( picbuffer[coord-2]+picbuffer[coord-1]
                    +picbuffer[coord+1]+picbuffer[coord+2])>>2);
            }
			   for( x = xstart; x < xstop; x++ )
               tempbuf[ ys+x ] = (short)linbuf[x];
         }
         xstart = checkrect->xpos;
	      xstop  = checkrect->xpos+checkrect->xsize;
	      ystart = checkrect->ypos;
	      ystop  = checkrect->ypos+checkrect->ysize;

         ysst1 = ysstep; ysst2 = ysstep*2;

         if( ystart < 2 ) ystart = 2;
	      if( ystop > picparam->columnheight-2)
            ystop = picparam->columnheight-2;
         ysstart = ystart*picparam->linelength;

   	   for( x = xstart; x < xstop; x++ )
		   {
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
		      {
               coord = ys+x;
               colbuf[ y ] =
                  (( tempbuf[coord-ysst2]+tempbuf[coord-ysst1]
                    +tempbuf[coord+ysst1]+tempbuf[coord+ysst2])>>2);
            }
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
                tempbuf[ ys+x ] = (short)colbuf[ y ];
         }
      }
      free( linbuf );
      free( colbuf );
   }
   else
   {
      linbuf = (long *)malloc( picparam->linelength * 3 * sizeof(long) );
      colbuf = (long *)malloc( picparam->columnheight * sizeof(long) );
      if( linbuf == NULL || colbuf == NULL )
      {
         if( linbuf != NULL ) free( linbuf );
         if( colbuf != NULL ) free( colbuf );
         return;
      }

      xstart  = checkrect->xpos;
	   xstop   = checkrect->xpos+checkrect->xsize;
	   ysstart = checkrect->ypos*picparam->linelength;
	   ysstop  = (checkrect->ypos+checkrect->ysize)*picparam->linelength;
	   ysstep  = picparam->linelength*3;

      if( width == 4 )
      {
         if( xstart < 4 ) xstart = 4;
   	   if( xstop > picparam->linelength-4 ) xstop = picparam->linelength-4;

         xstart *= 3; xstop *= 3; ysstart *= 3; ysstop *= 3;
         xstop2 = xstop+2;

    		for( ys = ysstart; ys < ysstop; ys += ysstep )
   		{
   			for( x = xstart; x < xstop; x+=3 )
   			{
               coord = ys+x;
               linbuf[x] =
                  (( picbuffer[coord-12]+picbuffer[coord- 9]
                    +picbuffer[coord- 6]+picbuffer[coord- 3]
                    +picbuffer[coord+ 3]+picbuffer[coord+ 6]
                    +picbuffer[coord+ 9]+picbuffer[coord+12])>>3);
               coord = ys+x+1;
               linbuf[x+1] =
                  (( picbuffer[coord-11]+picbuffer[coord- 8]
                    +picbuffer[coord- 5]+picbuffer[coord- 2]
                    +picbuffer[coord+ 4]+picbuffer[coord+ 7]
                    +picbuffer[coord+10]+picbuffer[coord+13])>>3);
               coord = ys+x+2;
               linbuf[x+2] =
                  (( picbuffer[coord-10]+picbuffer[coord- 7]
                    +picbuffer[coord- 4]+picbuffer[coord- 1]
                    +picbuffer[coord+ 5]+picbuffer[coord+ 8]
                    +picbuffer[coord+11]+picbuffer[coord+14])>>3);
            }
   			for( x = xstart; x < xstop2; x++ )
               tempbuf[ ys+x ] = (short)linbuf[x];
         }
         xstart  = checkrect->xpos;
   	   xstop   = checkrect->xpos+checkrect->xsize;
   	   ystart  = checkrect->ypos;
   	   ystop   = checkrect->ypos+checkrect->ysize;

         if( ystart < 4 ) ystart = 4;
   	   if( ystop > picparam->columnheight-4 )
            ystop = picparam->columnheight-4;
         ysstart = ystart*picparam->linelength;

         xstart *= 3; xstop *= 3; ysstart *= 3;
         ysst1 = ysstep; ysst2 = ysstep*2; ysst3 = ysstep*3; ysst4 = ysstep*4;

   	   for( x = xstart; x < xstop; x++ )
		   {
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
		      {
               coord = ys+x;
               colbuf[ y ] =
                  (( tempbuf[coord-ysst4]+tempbuf[coord-ysst3]
                    +tempbuf[coord-ysst2]+tempbuf[coord-ysst1]
                    +tempbuf[coord+ysst1]+tempbuf[coord+ysst2]
                    +tempbuf[coord+ysst3]+tempbuf[coord+ysst4])>>3);
            }
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
                tempbuf[ ys+x ] = (short)colbuf[ y ];
         }
      }
      else
      {
         if( xstart < 2 ) xstart = 2;
   	   if( xstop > picparam->linelength-2 ) xstop = picparam->linelength-2;

         xstart *= 3; xstop *= 3; ysstart *= 3; ysstop *= 3;
         xstop2 = xstop+2;

    		for( ys = ysstart; ys < ysstop; ys += ysstep )
   		{
   			for( x = xstart; x < xstop; x+=3 )
   			{
               coord = ys+x;
               linbuf[x] =
                  (( picbuffer[coord- 6]+picbuffer[coord- 3]
                    +picbuffer[coord+ 3]+picbuffer[coord+ 6])>>2);
               coord = ys+x+1;
               linbuf[x+1] =
                  (( picbuffer[coord- 5]+picbuffer[coord- 2]
                    +picbuffer[coord+ 4]+picbuffer[coord+ 7])>>2);
               coord = ys+x+2;
               linbuf[x+2] =
                  (( picbuffer[coord- 4]+picbuffer[coord- 1]
                    +picbuffer[coord+ 5]+picbuffer[coord+ 8])>>2);
            }
   			for( x = xstart; x < xstop2; x++ )
               tempbuf[ ys+x ] = (short)linbuf[x];
         }
         xstart  = checkrect->xpos;
   	   xstop   = checkrect->xpos+checkrect->xsize;
   	   ystart  = checkrect->ypos;
   	   ystop   = checkrect->ypos+checkrect->ysize;

         if( ystart < 2 ) ystart = 2;
   	   if( ystop > picparam->columnheight-2 )
            ystop = picparam->columnheight-2;
         ysstart = ystart*picparam->linelength;

         xstart *= 3; xstop *= 3; ysstart *= 3;
         ysst1 = ysstep; ysst2 = ysstep*2;

   	   for( x = xstart; x < xstop; x++ )
		   {
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
		      {
               coord = ys+x;
               colbuf[ y ] =
                  (( tempbuf[coord-ysst2]+tempbuf[coord-ysst1]
                    +tempbuf[coord+ysst1]+tempbuf[coord+ysst2])>>2);
            }
	   	   for( y=ystart, ys = ysstart; y < ystop; y++, ys += ysstep )
                tempbuf[ ys+x ] = (short)colbuf[ y ];
         }
      }
      free( linbuf );
      free( colbuf );
   }
}


/*** Statistische Auswertung eines einfachen Puffers ****/

void CCDLIB_GetLineStatistics(
	long  *mean,
	double *sigma,
	long  *minimum,
	long  *maximum,
	short  *values,
	long   valnum )
{
  long n;

  double meansum, variancesum, temporary;

  *mean = 0;
  *minimum = values[ 0 ];
  *maximum = values[ 0 ];
  *sigma = 0.;

  meansum = 0.;
  for( n=0; n<valnum; n++ )
  {
		meansum += (double)values[ n ];
		if( (long)values[ n ] > *maximum ) *maximum = (long)values[ n ];
		if( (long)values[ n ] < *minimum ) *minimum = (long)values[ n ];
  }
  *mean = (long)floor(0.5 + meansum / (double)valnum);

  variancesum = 0.;
  for( n=0; n<valnum; n++ )
  {
		temporary = (double)values[ n ] - (double)*mean;
		variancesum += ( temporary * temporary );
  }

  if( variancesum == 0. )
	 *sigma = 0.;
  else
	 *sigma = sqrt( variancesum / (double)( valnum-1 ) );
}

/*** Experimentierroutinen zum Rauschen in Bildern ***/

/************************************************************************
*                                                                       *
*	 CCDLIB_SetNoiseInPicture : Routine zum kuenstlichen Einfuegen von   *
*               Rauschen in vorhandene Bilder zu Testzwecken            *
*																								*
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf die Bilddaten       *
*               noisetype     :	Art des einzufuegenden Rauschens :     *
*                                                                       *
*               CCDLIB_NoiseWhiteConst:        Addiere weisses Rauschen *
*		          CCDLIB_NoiseDenseWhiteConst:   Add starkes ws. Rauschen *
*		          CCDLIB_NoiseWhiteFlatConst:    Nur weisses Rauschen     *
*		          CCDLIB_NoiseDenseWhiteFlatConst: Nur int. weisses Rau.  *
*		          CCDLIB_NoisePinkFlatConst:     Reines Rosa Rauschen     *
*		          CCDLIB_NoiseBrightConst:       Fuege weisse Pixel ein   *
*		          CCDLIB_NoiseDenseBrightConst:  Fuege viele weisse Pixel * 
*		          CCDLIB_NoiseDarkConst:         Fuege schwarze Pixel ein *
*		          CCDLIB_NoiseDenseDarkConst:    Fuege viele schwarze P.  *
*		          CCDLIB_NoiseMixedConst:        Fuege w & s Pixel ein    *
*		          CCDLIB_NoiseDenseMixedConst:   Fuege viele w & s Pixel  *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild steht an derselben Stelle wie das Ausgangsbild.    *
*   ACHTUNG: Das Ausgangsbild wird dabei ueberschrieben !               *
*                                                                       *
*   Version:    1.0 (05.06.1994)                                        *
*   Written by: Georg Dittie, 1994                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_SetNoiseInPicture
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	enum CCDLIB_NoiseTypeEnum noisetype
)
{
  long x, ys, ystop;
  short randomnum;
  short zw;
  
  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL )
	 return( CCDLIB_NoPicturePointerError );

  srand( time(NULL) );
  ystop = (picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

  for(ys = picturerect->ypos * pictureparam->linelength;
		ys < ystop;
		ys += pictureparam->linelength)
  {
	 for(x = picturerect->xpos;
		  x < picturerect->xpos + picturerect->xsize;
		  x++ )
	 {
		randomnum = (short)rand();
    
      switch( noisetype )
      {
		case CCDLIB_NoiseWhiteConst:
		  zw = (short)(picture[ x+ys ] + ((randomnum & (short)(pictureparam->dynamik/4))
									  - (short)(pictureparam->dynamik/8) ) );
		  picture[ x+ys ] = (short)(max(min( zw,(short)(pictureparam->dynamik-1)), 0) );
		  break;

		case CCDLIB_NoiseDenseWhiteConst:
		  zw = (short)(picture[ x+ys ] + ((randomnum & (short)(pictureparam->dynamik/2) )
									  - (short)(pictureparam->dynamik/4) ) );
		  picture[ x+ys ] = (short)(max(min( zw,(short)(pictureparam->dynamik-1) ), 0) );
		  break;

		case CCDLIB_NoiseWhiteFlatConst:
		  picture[ x+ys ] = (short)( (randomnum & (short)(pictureparam->dynamik/4) )
											 + (short)(pictureparam->dynamik/8*3) );
		  break;

		case CCDLIB_NoiseDenseWhiteFlatConst:
		  picture[ x+ys ] = (short)( (randomnum & (short)pictureparam->dynamik ) );
		  break;

		case CCDLIB_NoisePinkFlatConst:
		  picture[ x+ys] = (short)CCDLIB_GaussDevRandom(
									(short)(pictureparam->dynamik/2-1),
                           (short)(pictureparam->dynamik/2-1));
		  break;

		case CCDLIB_NoiseBrightConst:
		  if( randomnum >= 0x7eff )
			 picture[ x+ys ] = (short)(pictureparam->dynamik-1);
		  break;

		case CCDLIB_NoiseDenseBrightConst:
		  if( randomnum >= 0x77ff )
			 picture[ x+ys ] = (short)(pictureparam->dynamik-1);
		  break;
    
		case CCDLIB_NoiseDarkConst:
        if( randomnum <= 0x100 )
			 picture[ x+ys ] = 0;
		  break;
    
		case CCDLIB_NoiseDenseDarkConst:
		  if( randomnum <= 0x800 )
			 picture[ x+ys ] = 0;
		  break;

		case CCDLIB_NoiseMixedConst:
		  if( randomnum >= 0x7eff )
			 picture[ x+ys ] = (short)(pictureparam->dynamik-1);
		  else if( randomnum <= 0x100 )
			 picture[ x+ys ] = 0;
		  break;

		case CCDLIB_NoiseDenseMixedConst:
		  if( randomnum >= 0x77ff )
			 picture[ x+ys ] = (short)(pictureparam->dynamik-1);
		  else if( randomnum <= 0x100 )
			 picture[ x+ys ] = 0;
		  break;
		default:
			return( CCDLIB_WrongParameterError );
		}
	 }
  }

  return( CCDLIB_Successful );
}


short CCDLIB_GaussDevRandom
(
  short mean,
  short dynamik
)
{
  static BOOL iset = FALSE;
  static double gset;
  double fac, rsq, v1, v2, res;

  if( !iset )
  {
    do
    {
      v1 = 2. * (double)rand() / (double)RAND_MAX - 1.;
      v2 = 2. * (double)rand() / (double)RAND_MAX - 1.;
		rsq = v1*v1 + v2*v2;
    } while ( rsq >= 1. || rsq == 0. );

    fac  = sqrt( -2. * log(rsq) / rsq );
    gset = v1*fac;
    iset = TRUE;
    res  = v2*fac;
  }
  else
  {
    iset = FALSE;
    gset = 0.;
    res  = gset;
  }

  return( (short)(mean + (short)( res * (double)(dynamik) + 0.5 )) );
}

/************************************************************************
*																								*
*	CCDLIB_RemoveStars entfernt punktfrmige Lichtquellen im Bild        *
*                                                                       *
*   Version:    1.0 (CCD-Workshop in Kirchheim Mai 2002)                *
*   Written by: Georg Dittie, 2002                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_RemoveStars(
	short *destpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct *picparam,
	double masklevel )
{
//static long debugcount=0;

	long brightval, buflength, mval;
	long x, y, xs, ys, yd, xstart, xstop, ystart, ystop,
		  ystep, ysstep, yc, xc, ycstart, ycstep, xcstart,
        xm1, xp1, ym1, yp1, i;

	long histomaximum, histomaxpos, fwhmlevel;
	double histomaxpart, histoentropy, contrastrange;
	long histolowestpos, histohighestpos, histohighpos, histolowpos;
	long * histogramfield;
	CCDLIB_RectangleStruct picrect;
   char *mask1, *mask2, *puffer;

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPointerError );

	if( masklevel > 1. ) masklevel = 1.;
	if( masklevel < 0. ) masklevel = 0.;

	buflength = picparam->linelength*picparam->columnheight;

	mask1 = (char *)malloc( buflength * sizeof(char) );
	mask2 = (char *)malloc( buflength * sizeof(char) );
	puffer = (char *)malloc( buflength * sizeof(char) );
	histogramfield = (long *)malloc( picparam->dynamik * sizeof(long) );

	if( histogramfield == NULL || mask1 == NULL || mask2 == NULL || puffer == NULL )
	{
		if( histogramfield != NULL ) free( histogramfield );
		if( mask1 != NULL ) free( mask1 );
		if( mask2 != NULL ) free( mask2 );
      if( puffer != NULL ) free( puffer );
		return( CCDLIB_OutOfMemoryError );
	}

   memset( mask1, 0, buflength * sizeof(char) );

	CCDLIB_InitRectangle(
		&picrect, CCDLIB_MINIMALRIM, CCDLIB_MINIMALRIM,
      picparam->linelength-CCDLIB_MINIMALRIM*2, picparam->columnheight-CCDLIB_MINIMALRIM*2 );
	if( CCDLIB_RetrieveHistogram( &picrect, picparam,
			 sourcepic, histogramfield ) != CCDLIB_Successful )
	{
		if( histogramfield != NULL ) free( histogramfield );
		if( mask1 != NULL ) free( mask1 );
		if( mask2 != NULL ) free( mask2 );
      if( puffer != NULL ) free( puffer );
		return( CCDLIB_NoPointerError );
	}

	CCDLIB_EvaluateHistogram(
		histogramfield,
		picparam->dynamik,
		CCDLIB_Histo99_8Const,
		0,
		0,
		&histolowestpos,
		&histohighestpos,
		&histolowpos,
		&histohighpos,
		&histomaxpos,
		&histomaximum,
		&histomaxpart,
		&histoentropy );

   contrastrange = (double)histohighpos - (double)histomaxpos;
	fwhmlevel = (long)floor( contrastrange * masklevel
					 + (double)histomaxpos + 0.5 );

	xstart = picrect.xpos;
	ystart = picrect.ypos;
	xstop  = picrect.xpos+picrect.xsize;
	ystop  = picrect.ypos+picrect.ysize;
	ystart *= picparam->linelength;
	ystop  *= picparam->linelength;
	ystep  = picparam->linelength;

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for( y = ystart; y < ystop; y += ystep )
		{
			for( x = xstart; x < xstop; x++ )
			{
				if( sourcepic[ y+x ] > fwhmlevel )
					mask1[ y+x ] = 0x7F;
			}
		}
	}
	else
	{
		ysstep = ystep*3;

		for( yd = ystart, ys = ystart*3;
			  yd < ystop;
			  yd += ystep, ys += ysstep )
		{
			for( x = xstart, xs = xstart*3;
				  x < xstop;
				  x++, xs += 3 )
			{
				brightval = (long)floor(0.5+
					0.299 * (double)sourcepic[ ys+xs+2 ] +
					0.587 * (double)sourcepic[ ys+xs+1 ] +
					0.114 * (double)sourcepic[ ys+xs ] );

				if( brightval > fwhmlevel )
					mask1[ yd+x ] = 0x7F;
			}
		}
	}
   memcpy( mask2, mask1, buflength * sizeof(char) );

  	xstart = picrect.xpos+1;
   ystart = picrect.ypos+1;
   xstop  = picrect.xpos+picrect.xsize-1;
   ystop  = picrect.ypos+picrect.ysize-1;
   ystart *= picparam->linelength;
   ystop  *= picparam->linelength;
   ystep  = picparam->linelength;

   for( i=0; i<4; i++ )
	{
      memcpy( puffer, mask2, buflength * sizeof(char) );

	   for( y = ystart; y < ystop; y += ystep )
	   {
         ym1 = y-ystep;
         yp1 = y+ystep;
		   for( x = xstart; x < xstop; x++ )
		   {
            xm1 = x-1;
            xp1 = x+1;
			   if( puffer[ y+x ] == 0 )
            {
               mask2[ ym1+xm1 ] = 0;
               mask2[ ym1+x   ] = 0;
               mask2[ ym1+xp1 ] = 0;
               mask2[ y  +xm1 ] = 0;
               mask2[ y  +x   ] = 0;
               mask2[ y  +xp1 ] = 0;
               mask2[ yp1+xm1 ] = 0;
               mask2[ yp1+x   ] = 0;
               mask2[ yp1+xp1 ] = 0;
            }
         }
      }
   }

   for( i=0; i<4; i++ )
   {
      memcpy( puffer, mask2, buflength * sizeof(char) );
		for( y = ystart; y < ystop; y += ystep )
		{
         ym1 = y-ystep;
         yp1 = y+ystep;
		   for( x = xstart; x < xstop; x++ )
		   {
            xm1 = x-1;
            xp1 = x+1;
			   if( puffer[ y+x ] != 0 )
            {
               mask2[ ym1+xm1 ] = 0x7f;
               mask2[ ym1+x   ] = 0x7f;
               mask2[ ym1+xp1 ] = 0x7f;
               mask2[ y  +xm1 ] = 0x7f;
               mask2[ y  +x   ] = 0x7f;
               mask2[ y  +xp1 ] = 0x7f;
               mask2[ yp1+xm1 ] = 0x7f;
               mask2[ yp1+x   ] = 0x7f;
               mask2[ yp1+xp1 ] = 0x7f;
            }
         }
		}
	}
   for( i=0; i<buflength; i++ )
   {
       mval = (long)mask1[i]-(long)mask2[i];
       if( mval < 0 ) mval = 0;
       mask1[i] = (char)mval;
   }

   /* Eventuell hier noch eine Erosion/Dilatation zum Raendeentfernen TBD */

  	xstart = picrect.xpos;
   ystart = picrect.ypos;
   xstop  = picrect.xpos+picrect.xsize;
   ystop  = picrect.ypos+picrect.ysize;
   ystart *= picparam->linelength;
   ystop  *= picparam->linelength;
   ystep  = picparam->linelength;

   if( picparam->colorchannel != CCDLIB_RGBChannelColor )
   {
      for( y = ystart; y < ystop; y += ystep )
	   {
	      for( x = xstart; x < xstop; x++ )
	      {
		      if( mask1[ y+x ] == 0 )
               destpic[ y+x ] = sourcepic[ y+x ];
            else
               destpic[ y+x ] = (short)histomaxpos;
         }
      }
   }
   else
   {
      ycstart = ystart * 3;
      ycstep  = ystep * 3 ;
      xcstart = xstart * 3;
      for( y = ystart, yc = ycstart; y < ystop; y += ystep, yc += ycstep )
	   {
	      for( x = xstart, xc = xcstart; x < xstop; x++, xc += 3 )
	      {
		      if( mask1[ y+x ] == 0 )
            {
               destpic[ yc+xc   ] = sourcepic[ yc+xc   ];
               destpic[ yc+xc+1 ] = sourcepic[ yc+xc+1 ];
               destpic[ yc+xc+2 ] = sourcepic[ yc+xc+2 ];
            }
            else
            {
               destpic[ yc+xc   ] = (short)histomaxpos;
               destpic[ yc+xc+1 ] = (short)histomaxpos;
               destpic[ yc+xc+2 ] = (short)histomaxpos;
            }
         }
      }
   }

   free( mask1 );
   free( mask2 );
	free( puffer );
	free( histogramfield );

  	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_RGBCorrection : routine for shifting end resizing the blue   *
*               and red channel in RGB images                           *
*                                                                       *
*   Parameter:  * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               bluexshift    :  relative X shift against green channel *
*               blueyshift    :  relative Y shift against green channel *
*               bluesizedif   :  Size difference against green channel  *
*               redxshift     :  relative X shift against green channel *
*               redyshift     :  relative Y shift against green channel *
*               redsizedif    :  Size difference against green channel  *
*                                                                       *
*               CAUTION: *sourcepic != *destpic                         *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.1 (07.09.2002)                                        *
*   Written by: Georg Dittie, 1999 - 2002                               *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_RGBCorrection
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   long bluexshift,
   long blueyshift,
   long bluesizedif,
   long redxshift,
   long redyshift,
   long redsizedif
)
{
	long xs, xd, xc, ys, yd, yc, ystep, ystop, xstop1, ystop1,
        i, buflength, linehalf, colhalf,
        xq0, xq1, yq0, yq1, v00, v01, v10, v11, xfrac, yfrac;
   double sizefactor, xp, yp;
   short *tempbuf;
   BOOL  issamebuffer = FALSE;


	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		return( CCDLIB_WrongParameterError );

   buflength = pictureparam->linelength*pictureparam->columnheight*3;

   if( sourcepic == destpic )
   {
      tempbuf = (short*)malloc( buflength*sizeof(short) );
      if( tempbuf == NULL )
         return( CCDLIB_OutOfMemoryError );
      issamebuffer = TRUE;
   }
   else
      tempbuf = destpic;

   for( i=0; i<buflength; i+=3 )
   {
   	tempbuf[i  ] = 0;
   	tempbuf[i+1] = sourcepic[i+1];
   	tempbuf[i+2] = 0;
   }

   ystep = pictureparam->linelength*3;
   ystop = ystep*pictureparam->columnheight;
   ystop1 = ystop-ystep;
   xstop1 = ystep-3;

   linehalf = pictureparam->linelength/2;
   colhalf = pictureparam->columnheight/2;

   if( bluesizedif == 0 )
   {
      for( ys=0; ys<ystop; ys+=ystep )
      {
         yd = ys-blueyshift*ystep;
         if( yd < 0 || yd >= ystop )
         	continue;

      	for( xs=0; xs<ystep; xs+=3 )
         {
	         xd = xs-bluexshift*3;
   	      if( xd < 0 || xd >= ystep )
      	   	continue;

		   	tempbuf[ys+xs] = sourcepic[yd+xd];
         }
      }
   }
   else
   {
      sizefactor = (double)linehalf/(double)(linehalf+bluesizedif);

      for( yc=0, ys=0; ys<ystop1; ys+=ystep, yc++ )
      {
         yp = (double)(yc-colhalf)*sizefactor
         	+ (double)(colhalf-blueyshift);

			yq0 = (long)floor( yp ) * ystep;
			yq1 = yq0 + ystep;

         if( yq0 < 0 || yq1 >= ystop1 )
         	continue;

			yfrac = (long)( frac( yp )*256. );

      	for( xc=0, xs=0; xs<xstop1; xs+=3, xc++ )
         {
         	xp = (double)(xc-linehalf)*sizefactor
            	+ (double)(linehalf-bluexshift);

				xq0 = (long)floor( xp ) * 3;
				xq1 = xq0 + 3;

   	      if( xq0 < 0 || xq1 >= xstop1 )
      	   	continue;

				xfrac = (long)( frac( xp )*256. );

				v00 = (long)sourcepic[ xq0+yq0 ];
				v10 = (long)sourcepic[ xq1+yq0 ];
				v01 = (long)sourcepic[ xq0+yq1 ];
				v11 = (long)sourcepic[ xq1+yq1 ];

				tempbuf[ xs+ys ] = (short)(
					( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
					(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );
         }
      }
   }

   if( redsizedif == 0 )
   {
      for( ys=0; ys<ystop; ys+=ystep )
      {
         yd = ys-redyshift*ystep;
         if( yd < 0 || yd >= ystop )
         	continue;

      	for( xs=2; xs<ystep; xs+=3 )
         {
	         xd = xs-redxshift*3;
   	      if( xd < 0 || xd >= ystep )
      	   	continue;

		   	tempbuf[ys+xs] = sourcepic[yd+xd];
         }
      }
   }
   else
   {
      sizefactor = (double)linehalf/(double)(linehalf+redsizedif);

      for( yc=0, ys=0; ys<ystop1; ys+=ystep, yc++ )
      {
         yp = (double)(yc-colhalf)*sizefactor
         	+ (double)(colhalf-redyshift);

			yq0 = (long)floor( yp ) * ystep;
			yq1 = yq0 + ystep;

         if( yq0 < 0 || yq1 >= ystop1 )
         	continue;

			yfrac = (long)( frac( yp )*256. );

      	for( xc=0, xs=2; xs<xstop1; xs+=3, xc++ )
         {
         	xp = (double)(xc-linehalf)*sizefactor
            	+ (double)(linehalf-redxshift);

				xq0 = (long)floor( xp ) * 3 + 2;
				xq1 = xq0 + 3;

   	      if( xq0 < 0 || xq1 >= xstop1 )
      	   	continue;

				xfrac = (long)( frac( xp )*256. );

				v00 = (long)sourcepic[ xq0+yq0 ];
				v10 = (long)sourcepic[ xq1+yq0 ];
				v01 = (long)sourcepic[ xq0+yq1 ];
				v11 = (long)sourcepic[ xq1+yq1 ];

				tempbuf[ xs+ys ] = (short)(
					( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
					(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );
         }
      }
   }
   if( issamebuffer == TRUE )
   {
      memcpy( destpic, tempbuf, buflength );
      free( tempbuf );
   }

	return( CCDLIB_Successful );
}


/************************************************************************
*                                                                       *
*	 CCDLIB_ColorCorrection : routine for adjusting the RGB channels to  *
*               each other in RGB images                                *
*                                                                       *
*   Parameter:  * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               redcorr       :  Red factor                             *
*               greencorr     :  Green factor                           *
*               bluecorr      :  Blue factor                            *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 vom Nelkensamstag 2002                              *
*   Written by: Georg Dittie, 2002                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_ColorCorrection
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   long redcorr,
   long greencorr,
   long bluecorr
)
{
	long i, j, k, buflength, val;
   short *redtab, *greentab, *bluetab;
   double rfak, gfak, bfak;

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		return( CCDLIB_WrongParameterError );

   redtab   = (short *)malloc(32768L*sizeof(short));
   greentab = (short *)malloc(32768L*sizeof(short));
   bluetab  = (short *)malloc(32768L*sizeof(short));
   if( redtab == NULL || greentab == NULL || bluetab == NULL )
   {
      if( redtab != NULL ) free( redtab );
      if( greentab != NULL ) free( greentab );
      if( bluetab != NULL ) free( bluetab );
      return( CCDLIB_OutOfMemoryError );
   }

   rfak = pow( 10., (double)redcorr/500. );
   gfak = pow( 10., (double)greencorr/500. );
   bfak = pow( 10., (double)bluecorr/500. );

   for( i=0; i<32768; i++ )
   {
      val = (long)((double)i*rfak+0.5);
      if( val > 32767 ) val = 32767;
      redtab[i] = (short)val;

      val = (long)((double)i*gfak+0.5);
      if( val > 32767 ) val = 32767;
      greentab[i] = (short)val;

      val = (long)((double)i*bfak+0.5);
      if( val > 32767 ) val = 32767;
      bluetab[i] = (short)val;
   }

   buflength = pictureparam->linelength*pictureparam->columnheight*3;
   for( i=0, j=1, k=2; i<buflength; i+=3, j+=3, k+=3 )
   {
   	destpic[i] = bluetab[sourcepic[i]];
   	destpic[j] = greentab[sourcepic[j]];
   	destpic[k] = redtab[sourcepic[k]];
   }

   free( redtab );
   free( greentab );
   free( bluetab );
	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_SeparatedMaskedFilterOp : Universelle Filteroperation fuer   *
*               separierbare Operationen. Die Maske ist quadratisch     *
*                                                                       *
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               maskdiameter  :	Durchmesser des Filtermaskenkreis      *
*               rectdiameter  :	Durchmesser des Rechteckanteils        *
*               filtdiameter  :	Durchmesser des Bandpassfilters        *
*               CCDLIB_OperationTypeEnum : Art der Filteroperation      *
*               CCDLIB_BandpassFiltEnum : Art der Bandpassfilterung     *
*                                                                       *
*               ACHTUNG: *diameter darf nur ungerade Werte annehmen!    *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (08.10.1998)                                        *
*   Written by: Georg Dittie, 1998                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_SeparatedMaskedFilterOp
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype,
   void (* workmsg)()
)
{
	double *mask, *line, *col;
	double temporary, tempb, tempg, tempr, dynamikmone;

	long x, xk, y, yk, ys, xc,
		  xstart, xstartk, xstop, xstopk,
		  ystart, ystartk, ystop, ystopk,
		  ystep, ystartc, ystopc,
		  indexhalf, xindexhalf;


	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( effiency < 0.01 || effiency > 100. )
	{
		return( CCDLIB_CopyRegion(
			destpic, sourcepic, pictureparam, picturerect ) );
	}
	dynamikmone = (double)(pictureparam->dynamik - 1);
	indexhalf = maskdiameter/2;

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		ystep  = pictureparam->linelength;
		xstart = (long)(picturerect->xpos);
		xstartk = (long)(picturerect->xpos+indexhalf);
		ystart = (long)(picturerect->ypos)*ystep;
		ystartk = (long)(picturerect->ypos+indexhalf);
		xstop  = (long)(picturerect->xpos+picturerect->xsize);
		xstopk = (long)(picturerect->xpos+picturerect->xsize-indexhalf);
		ystop  = (long)(picturerect->ypos+picturerect->ysize)*ystep;
		ystopk  = (long)(picturerect->ypos+picturerect->ysize-indexhalf);
		ystartc = (long)(picturerect->ypos);
		ystopc = (long)(picturerect->ypos+picturerect->ysize);
		xindexhalf = indexhalf;
	}
	else
	{
		ystep  = pictureparam->linelength*3;
		xstart = (long)(picturerect->xpos)*3;
		xstartk = (long)(picturerect->xpos+indexhalf)*3;
		ystart = (long)(picturerect->ypos)*ystep;
		ystartk = (long)(picturerect->ypos+indexhalf);
		xstop  = (long)(picturerect->xpos+picturerect->xsize)*3;
		xstopk = (long)(picturerect->xpos+picturerect->xsize-indexhalf)*3;
		ystop  = (long)(picturerect->ypos+picturerect->ysize)*ystep;
		ystopk  = (long)(picturerect->ypos+picturerect->ysize-indexhalf);
		ystartc = (long)(picturerect->ypos);
		ystopc = (long)(picturerect->ypos+picturerect->ysize);
		xindexhalf = indexhalf*3;
	}

	line = ( double * )malloc( ystep * sizeof( double ) );
	col  = ( double * )malloc( pictureparam->columnheight * sizeof( double ) );
	mask = ( double * )malloc( maskdiameter * sizeof( double ) );

	if( line == NULL || col == NULL || mask == NULL )
	{
		if( line != NULL ) free( line );
		if( col  != NULL ) free( col );
		if( mask != NULL ) free( mask );

		return( CCDLIB_OutOfMemoryError );
	}

	CCDLIB_GetSeparableMask(
		mask,
		maskdiameter,
		rectdiameter,
		filtdiameter,
		effiency,
		operationtype,
		filtertype );

	for( ys = ystart;
		  ys < ystop;
		  ys += ystep )
	{
      if( workmsg != NULL && (ys/ystep)%64 == 0 ) (*workmsg)( ys, ystop*2 );

		for( x = xstart; x < xstop; x++ )
			line[ x ] = (double)sourcepic[ x+ys ];

		if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		{
			for( x = xstartk; x < xstopk; x++ )
			{
				temporary = 0.;
				for( xk = -xindexhalf;
					  xk <= xindexhalf;
					  xk++ )
				{
					temporary += (line[ x+xk ] * mask[ xk + indexhalf ]);
				}
				destpic[ x+ys ] = (short)floor(
					max( 0, min( dynamikmone, temporary ) ) + 0.5 );
			}
		}
		else
		{
			for( x = xstartk; x < xstopk; x+=3 )
			{
				tempb=0.; tempg=0.; tempr=0.;
				for( xk = -xindexhalf, xc=0;
					  xk <= xindexhalf;
					  xk += 3, xc++ )
				{
					tempb += (line[ x+xk   ] * mask[ xc ]);
					tempg += (line[ x+xk+1 ] * mask[ xc ]);
					tempr += (line[ x+xk+2 ] * mask[ xc ]);
				}
				destpic[ x+ys ] = (short)floor(
					max( 0, min( dynamikmone, tempb ) ) + 0.5 );
				destpic[ x+ys+1 ] = (short)floor(
					max( 0, min( dynamikmone, tempg ) ) + 0.5 );
				destpic[ x+ys+2 ] = (short)floor(
					max( 0, min( dynamikmone, tempr ) ) + 0.5 );
			}
		}
	}

	for( x = xstart; x < xstop; x ++ )
	{
      if( workmsg != NULL && (ys/ystep)%64 == 0 ) (*workmsg)( x+xstop, xstop*2 );

		for( y=ystartc, ys=ystart; y<ystopc; y++, ys+=ystep )
			col[ y ] = (double)destpic[ x+ys ];

		for( y = ystartk, ys = ystartk*ystep;
			  y < ystopk;
			  y++, ys += ystep )
		{
			temporary = 0.;
			for( yk = -indexhalf;
				  yk <= indexhalf;
				  yk++ )
			{
				temporary += (col[ y+yk ] * mask[ yk + indexhalf ]);
			}
			destpic[ x+ys ] = (short)floor(
				max( 0, min( dynamikmone, temporary ) ) + 0.5 );
		}
	}

	free( line );
	free( col );
	free( mask );

	return( CCDLIB_Successful );
}

enum CCDLIB_ErrorEnum CCDLIB_GetSeparableMask(
	double *mask,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype )
{
	long x, xs, filthalf, filtdif, indexhalf;

	double masksum, maskwidth, divfaktor, centralval, radius,
			 maskradius, rectradius, filtradius, filtmasksum, filtfactor,
			 arg, rectant;
	double *filtmask;

	indexhalf = maskdiameter/2;

	filtmask = NULL;
	maskradius = (double)maskdiameter / 2.;
	filtradius = (double)filtdiameter / 2.;
	rectradius = (double)rectdiameter / 2.;
	masksum = 0.;
	maskwidth = 3./(double)(indexhalf);
	filtmasksum = 0.;

	if( filtertype != CCDLIB_NoBandpassConst )
	{
		filtmask = (double*)malloc(filtdiameter*sizeof(double));
		if( filtmask != NULL )
		{
			filthalf = filtdiameter / 2;
			filtdif = indexhalf-filthalf;
			rectant = filtradius/3.;
			for( x=0; x<filtdiameter; x++ )
			{
				radius = (double)(x-filthalf);
				switch( operationtype )
				{
				case CCDLIB_FiltRectangleConst:
					filtmask[ x ] = 1.;
					break;
				case CCDLIB_FiltTriangleConst:
					filtmask[ x ] = radius/filtradius;
					break;
				case CCDLIB_FiltGaussConst:
					filtmask[ x ] = exp( -(radius*radius)/rectant );
					break;
				default:
					return( CCDLIB_WrongParameterError );
				}
				filtmasksum += fabs( filtmask[ x ] );
			}
		}
	}

	rectant = (maskradius-rectradius)/3.;
	for( x=0; x<maskdiameter; x++ )
	{
		radius = (double)(x-indexhalf);
		switch( operationtype )
		{
		case CCDLIB_SharpGaussConst:
			mask[ x ] = -exp( -(radius*radius)/(maskradius/3.) );
			break;
		case CCDLIB_SharpRectangleConst:
			mask[ x ] = -1.;
			break;
		case CCDLIB_SharpTriangleConst:
			mask[ x ] = -radius/(double)indexhalf;
			break;
		case CCDLIB_SharpGaussRectConst:
			if( radius < rectradius )
				mask[ x ] = -1.;
			else
			{
				arg = radius-rectradius;
				mask[ x ] = -exp( -(arg*arg)/rectant );
			}
			break;
		case CCDLIB_MeanValueConst:
			mask[ x ] = 1.;
			break;
		case CCDLIB_LorentzMaskConst:
			mask[ x ] = -1./(radius*radius*maskwidth+1.);
			break;
		default:
			return( CCDLIB_WrongParameterError );
		}
		masksum += fabs( mask[ x ] );
	}

	if( filtmasksum == 0 )
	{
		centralval = masksum * (1. + 1./effiency);
		divfaktor = centralval - masksum;
		mask[ indexhalf ] += centralval;
	}
	else
	{
		filtfactor = masksum/filtmasksum * (1. + 1./effiency);
		divfaktor  = filtmasksum * filtfactor - masksum;
		for( x=0, xs=filtdif; x<maskdiameter; x++, xs++ )
		{
			mask[ xs ] += filtmask[ x ]*filtfactor;
		}
	}

	for( x=0; x<maskdiameter; x++ )
		mask[ x ] /= divfaktor;

	if( filtmask != NULL ) free( filtmask );

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_MaskedFilterOperation : Universelle Filteroperation fuer     *
*               nicht separierbare Operationen. Die Maske ist echt      *
*               kreisfoermig !                                          *
*                                                                       *
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               maskdiameter  :	Durchmesser des Filtermaskenkreis      *
*               rectdiameter  :	Durchmesser des Rechteckanteils        *
*               filtdiameter  :	Durchmesser des Bandpassfilters        *
*               effiency      :  Verstrkung gegenber Normierung       *
*               adaptiv       :  gradientenabhngige Verstrkung        *
*
*               CCDLIB_OperationTypeEnum : Art der Filteroperation      *
*               CCDLIB_BandpassFiltEnum : Art der Bandpassfilterung     *
*                                                                       *
*               ACHTUNG: maskdiameter darf nur ungerade Werte annehmen! *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.3 ( Allerheiligentag 1999 )                           *
*   Version:    2.0 ( 23. 8. 2002 )                                     *
                                                                        *
*   Written by: Georg Dittie, 1994-2002                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_MaskedFilterOperation
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
   BOOL adaptiv,
   long astigmatism,
   double astigvector,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype,
   void (* workmsg)()
)
{
	long x, xk, yk, ys, xks, yks, ystep, i, j, n, sobelswitch,  matrixsize,
		  xstart, x1stop, ystart, y1stop, xsstart, ysstart,
		  index, indexhalf, indexturn, bufferindex, xindexhalf, yindexhalf;
   CCDLIB_PictureparamStruct sobelparam;
	double temporary, dynamikmone, dynamikhalf, tempb, tempg, tempr, maskval, eff;
   double **mask[16], **imask;
	short  sortarray[ CCDLIB_ConvolutionWidthConst * CCDLIB_ConvolutionWidthConst ];
   short *bufferline[ CCDLIB_ConvolutionWidthConst ];
   short *sobelbuffer;

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( effiency < 0.01 )
		return( CCDLIB_CopyRegion(
			destpic, sourcepic, pictureparam, picturerect ) );
   if( effiency <= 1. ) adaptiv = FALSE;
   if( effiency > 100. ) effiency = 100.;


	if( operationtype	== CCDLIB_MeanValueConst ||
		 operationtype == CCDLIB_MaskedMedianConst )
      astigmatism = 0;

   if( astigmatism > 1 )
      matrixsize = maskdiameter + (astigmatism/2)*2;
   else
      matrixsize = maskdiameter;


	if( matrixsize > CCDLIB_ConvolutionWidthConst )
		return( CCDLIB_WrongParameterError );

	if( operationtype >= CCDLIB_OpNGradientConst )
   {
      adaptiv = FALSE;
      astigmatism = 0;
		maskdiameter = 3;
      matrixsize = 3;
   }
	dynamikmone = (double)(pictureparam->dynamik - 1);
	dynamikhalf = (double)(pictureparam->dynamik)/2.;

	indexhalf  = matrixsize / 2;
	indexturn  = indexhalf + 1;

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		ystep = pictureparam->linelength;
		xstart = (long)(picturerect->xpos+indexhalf);
		ystart = (long)(picturerect->ypos+indexhalf) * ystep;
		x1stop = (long)(picturerect->xpos+picturerect->xsize-indexhalf);
		y1stop = (long)(picturerect->ypos+picturerect->ysize-indexhalf) * ystep;
		xindexhalf =  matrixsize/2;
		yindexhalf = (matrixsize/2)*ystep;
	}
	else
	{
		ystep = pictureparam->linelength*3;
		xstart = (long)(picturerect->xpos+indexhalf)*3;
		ystart = (long)(picturerect->ypos+indexhalf)*ystep;
		x1stop = (long)(picturerect->xpos+picturerect->xsize-indexhalf)*3;
		y1stop = (long)(picturerect->ypos+picturerect->ysize-indexhalf)*ystep;
		xindexhalf = (matrixsize/2)*3;
		yindexhalf = (matrixsize/2)*ystep;
	}

	bufferline[ 0 ] = (short *)malloc( ystep * sizeof(short) * matrixsize );
	if( bufferline[ 0 ] == NULL )
		return( CCDLIB_OutOfMemoryError );

	for( i=1; i<matrixsize; i++ )
		bufferline[ i ] = bufferline[ 0 ] + i*ystep;

   if( adaptiv == TRUE )
   {
      sobelbuffer = (short *)malloc(
         pictureparam->linelength*pictureparam->columnheight*sizeof(short) );
      if( sobelbuffer == NULL )
	   {
		   free( bufferline[0] );
		   return( CCDLIB_OutOfMemoryError );
	   }

      for( i=0; i<16; i++ )
      {
	      if( !CCDLIB_Matrixalloc( &mask[i], matrixsize ) )
	      {
            free( sobelbuffer );
		      free( bufferline[0] );
            for( j=0; j<i; j++ )
               CCDLIB_Matrixfree( mask[j] );
		      return( CCDLIB_OutOfMemoryError );
         }

         eff = (effiency-1.) * (double)i/16. + 1.;
   	   CCDLIB_GetNotSeparableMask(
	   	   mask[i], maskdiameter, rectdiameter, filtdiameter,
		      eff, astigmatism, astigvector, operationtype, filtertype );
      }

      if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
         CCDLIB_ExtractLuminance( sourcepic, pictureparam, sobelbuffer );
      else
         memcpy( sobelbuffer, sourcepic,
            pictureparam->linelength*pictureparam->columnheight*sizeof(short) );

      sobelparam = *pictureparam;
      sobelparam.colorchannel = CCDLIB_BlackWhiteColor;
      CCDLIB_CalcSobel( sobelbuffer, sobelbuffer, &sobelparam );
      CCDLIB_QuickEqualize( sobelbuffer, &sobelparam );
   }
   else
   {
	   if( !CCDLIB_Matrixalloc( &mask[0], matrixsize ) )
	   {
		   free( bufferline[0] );
		   return( CCDLIB_OutOfMemoryError );
	   }
   	CCDLIB_GetNotSeparableMask(
	   	mask[0], maskdiameter, rectdiameter, filtdiameter,
		   effiency, astigmatism, astigvector, operationtype, filtertype );
   }

	index = 0;
	for( ys = ystart, ysstart = ystart-yindexhalf;
        ys < y1stop;
        ys += ystep, ysstart += ystep )
	{
      if( workmsg != NULL ) (*workmsg)( (y1stop-ystart)/ystep, FALSE );

 		bufferindex = index % indexturn;

		if( index >= indexturn )
		{
			for( x = xstart; x < x1stop; x++ )
				destpic[ ys - indexturn*ystep + x ] = bufferline[ bufferindex ][ x ];
		}

		if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		{
			for( x = xstart, xsstart = xstart-xindexhalf;
              x < x1stop;
              x++, xsstart++ )
			{
            if( adaptiv == TRUE )
            {
               sobelswitch = (long)( log((double)sobelbuffer[ ys+x ]+1.)*1.4427 + 0.5 );

               switch( sobelswitch )
               {
               case 0:
                  imask = mask[15]; break;
               case 1:
                  imask = mask[14]; break;
               case 2:
                  imask = mask[13]; break;
               case 3:
                  imask = mask[12]; break;
               case 4:
                  imask = mask[11]; break;
               case 5:
                  imask = mask[10]; break;
               case 6:
                  imask = mask[9]; break;
               case 7:
                  imask = mask[8]; break;
               case 8:
                  imask = mask[7]; break;
               case 9:
                  imask = mask[6]; break;
               case 10:
                  imask = mask[5]; break;
               case 11:
                  imask = mask[4]; break;
               case 12:
                  imask = mask[3]; break;
               case 13:
                  imask = mask[2]; break;
               case 14:
                  imask = mask[1]; break;
               default:
                  imask = mask[0]; break;
               }
            }
            else
            {
               imask = mask[0];
            }

            switch( operationtype )
				{
	         case CCDLIB_SharpGaussConst:
				case CCDLIB_SharpRectangleConst:
				case CCDLIB_SharpTriangleConst:
				case CCDLIB_SharpGaussRectConst:
				case CCDLIB_MeanValueConst:
				case CCDLIB_LorentzMaskConst:
					temporary= 0.;
 					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for(xks = xsstart, xk=0;
							 xk  <  matrixsize;
                      xks++, xk++ )
						{
		         	   maskval = imask[ yk ][ xk ];
 						 	if( maskval != 0 )
 								temporary += (double)sourcepic[ xks + yks ] * maskval;
                  }
					}
					bufferline[ bufferindex ][ x ] = (short)(
						max( 0, min( dynamikmone, temporary ) ) );
					break;

				case CCDLIB_OpNGradientConst:
				case CCDLIB_OpNEGradientConst:
				case CCDLIB_OpEGradientConst:
				case CCDLIB_OpSEGradientConst:
				case CCDLIB_OpSGradientConst:
				case CCDLIB_OpSWGradientConst:
				case CCDLIB_OpWGradientConst:
				case CCDLIB_OpNWGradientConst:
					temporary= 0.;
 					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for(xks = xsstart, xk=0;
							 xk  <=  matrixsize;
                      xks++, xk++ )
						{
		         	   maskval = imask[ yk ][ xk ];
							if( maskval != 0 )
								temporary += (double)sourcepic[ xks + yks ]*maskval;
						}
					}
					bufferline[ bufferindex ][ x ] = (short)(
						max( 0, min( dynamikmone, temporary+dynamikhalf ) ) );
					break;

				case CCDLIB_MaskedMedianConst:
					for( yks = ysstart, n=0, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for(xks = xsstart, xk=0;
							 xk  <  matrixsize;
                      xks++, xk++ )
						{
							if( imask[ yk ][ xk ] != 0 )
							{
								sortarray[ n ] = (short)sourcepic[ xks + yks ];
								n++;
							}
						}
					}
					bufferline[ bufferindex ][ x ] = (short)(
						CCDLIB_GetMedianFromArray( sortarray, n ) );
					break;
				}
			}
		}
		else  /** RGB-Bilder maskieren **/
		{
			for( x = xstart, xsstart = xstart-xindexhalf;
              x < x1stop;
              x += 3, xsstart += 3 )
			{
            if( adaptiv == TRUE )
            {
               sobelswitch = (long)( log((double)sobelbuffer[ (ys+x)/3 ]+1.)*1.4427 + 0.5 );

               switch( sobelswitch )
               {
               case 0:
                  imask = mask[15]; break;
               case 1:
                  imask = mask[14]; break;
               case 2:
                  imask = mask[13]; break;
               case 3:
                  imask = mask[12]; break;
               case 4:
                  imask = mask[11]; break;
               case 5:
                  imask = mask[10]; break;
               case 6:
                  imask = mask[9]; break;
               case 7:
                  imask = mask[8]; break;
               case 8:
                  imask = mask[7]; break;
               case 9:
                  imask = mask[6]; break;
               case 10:
                  imask = mask[5]; break;
               case 11:
                  imask = mask[4]; break;
               case 12:
                  imask = mask[3]; break;
               case 13:
                  imask = mask[2]; break;
               case 14:
                  imask = mask[1]; break;
               default:
                  imask = mask[0]; break;
               }
            }
            else
            {
               imask = mask[0];
            }

				switch( operationtype )
				{
            case CCDLIB_SharpGaussConst:
				case CCDLIB_SharpRectangleConst:
				case CCDLIB_SharpTriangleConst:
				case CCDLIB_SharpGaussRectConst:
				case CCDLIB_MeanValueConst:
				case CCDLIB_LorentzMaskConst:
					tempb=0.; tempg=0.; tempr=0.;
					for( yks = ysstart, yk=0;
						  yk  < matrixsize;
						  yks += ystep, yk++ )
					{
						for(xks = xsstart, xk=0;
							 xk < matrixsize;
							 xks+=3, xk++ )
						{
		         	   maskval = imask[ yk ][ xk ];
							if( maskval != 0 )
							{
								tempb += (double)sourcepic[ xks   +yks ]*maskval;
								tempg += (double)sourcepic[ xks+1 +yks ]*maskval;
								tempr += (double)sourcepic[ xks+2 +yks ]*maskval;
							}
						}
					}
					bufferline[ bufferindex ][ x ] = (short)(
						max( 0, min( dynamikmone, tempb ) ) );
					bufferline[ bufferindex ][ x+1 ] = (short)(
						max( 0, min( dynamikmone, tempg ) ) );
					bufferline[ bufferindex ][ x+2 ] = (short)(
						max( 0, min( dynamikmone, tempr ) ) );
					break;

				case CCDLIB_OpNGradientConst:
				case CCDLIB_OpNEGradientConst:
				case CCDLIB_OpEGradientConst:
				case CCDLIB_OpSEGradientConst:
				case CCDLIB_OpSGradientConst:
				case CCDLIB_OpSWGradientConst:
				case CCDLIB_OpWGradientConst:
				case CCDLIB_OpNWGradientConst:
					tempb=0.; tempg=0.; tempr=0.;
					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for(xks = xsstart, xk=0;
							 xk  <  matrixsize;
							 xks+=3, xk++ )
						{
 		         	   maskval = imask[ yk ][ xk ];
							if( maskval != 0 )
							{
								tempb += (double)sourcepic[ xks   +yks ]*maskval;
								tempg += (double)sourcepic[ xks+1 +yks ]*maskval;
								tempr += (double)sourcepic[ xks+2 +yks ]*maskval;
							}
						}
					}
					bufferline[ bufferindex ][ x ] = (short)(
						max( 0, min( dynamikmone, tempb+dynamikhalf ) ) );
					bufferline[ bufferindex ][ x+1 ] = (short)(
						max( 0, min( dynamikmone, tempg+dynamikhalf ) ) );
					bufferline[ bufferindex ][ x+2 ] = (short)(
						max( 0, min( dynamikmone, tempr+dynamikhalf ) ) );
					break;

				case CCDLIB_MaskedMedianConst:
               n = 0;
					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for( xks = xsstart, xk=0;
							  xk  <  matrixsize;
							  xks += 3, xk++ )
						{
							if( imask[ yk ][ xk ] != 0 )
							{
								sortarray[ n ] = (short)sourcepic[ xks + yks ];
								n++;
							}
						}
					}
					bufferline[ bufferindex ][ x ] = (short)(
						CCDLIB_GetMedianFromArray( sortarray, n ) );

               n = 0;
					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for( xks = xsstart, xk=0;
							  xk  <  matrixsize;
							  xks += 3, xk++ )
						{
							if( imask[ yk ][ xk ] != 0 )
							{
								sortarray[ n ] = (short)sourcepic[ xks + yks + 1 ];
								n++;
							}
						}
					}
					bufferline[ bufferindex ][ x+1 ] = (short)(
						CCDLIB_GetMedianFromArray( sortarray, n ) );

               n = 0;
					for( yks = ysstart, yk=0;
						  yk  <  matrixsize;
						  yks += ystep, yk++ )
					{
						for( xks = xsstart, xk=0;
							  xk  <  matrixsize;
							  xks += 3, xk++ )
						{
							if( imask[ yk ][ xk ] != 0 )
							{
								sortarray[ n ] = (short)sourcepic[ xks + yks + 2 ];
								n++;
							}
						}
					}
					bufferline[ bufferindex ][ x+2 ] = (short)(
						CCDLIB_GetMedianFromArray( sortarray, n ) );
					break;
				}
			}
		}
		index++;
	}

	for( ys = y1stop;
		  ys < y1stop + indexturn*ystep;
		  ys += ystep)
	{
		for( x = xstart; x < x1stop; x++ )
			destpic[ ys - indexturn*ystep + x ] =
				bufferline[ index%indexturn ][ x ];
		index++;
	}

	free( bufferline[ 0 ] );
   if( adaptiv == TRUE )
   {
      free( sobelbuffer );
      for( i=0; i<16; i++ )
         CCDLIB_Matrixfree( mask[i] );
   }
   else
	   CCDLIB_Matrixfree( mask[0] );

	return( CCDLIB_Successful );
}

enum CCDLIB_ErrorEnum CCDLIB_GetNotSeparableMask(
	double **mask,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
   long astigmatism,
   double astigposwin,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype )
{
	long x, y, xs, ys, xa, ya, xm, ym,
        i, j, filthalf, filtdif, indexhalf,
        matrixsize;

	double masksum, maskwidth, divfaktor, centralval, radius,
			 maskradius, rectradius, filtradius, filtmasksum, filtfactor,
			 arg, rectant, f_astig, astigrad, xar, yar, xsqr, ysqr;
	double **filtmask;


	if( operationtype	== CCDLIB_MeanValueConst ||
		 operationtype == CCDLIB_MaskedMedianConst )
      astigmatism = 0;

	filtmask = NULL;
	maskradius = (double)maskdiameter / 2.;
	filtradius = (double)filtdiameter / 2.;
	rectradius = (double)rectdiameter / 2.;
	masksum = 0.;
	filtmasksum = 0.;
   if( astigmatism > 1 )
      matrixsize = maskdiameter+(astigmatism/2)*2;
   else
      matrixsize = maskdiameter;
	indexhalf = matrixsize/2;
	maskwidth = 9./(double)(indexhalf*indexhalf);


	if( operationtype < CCDLIB_OpNGradientConst )
	{
      /* Berechnung der Tiefpassmaske ***/
		if( filtertype != CCDLIB_NoBandpassConst )
		{
			if( CCDLIB_Matrixalloc( &filtmask, filtdiameter ) )
			{
				filthalf = filtdiameter / 2;
				filtdif = indexhalf-filthalf;
				rectant = filtradius/3.;
				for( y=0; y<filtdiameter; y++ )
				{
					ysqr = (double)(y-filthalf)+.5;
					ysqr = ysqr*ysqr;

					for( x=0; x<filtdiameter; x++ )
					{
						xsqr = (double)(x-filthalf)+.5;
						xsqr = xsqr*xsqr;
						radius = sqrt( xsqr+ysqr );
						if( radius <= filtradius )
						{
							switch( operationtype )
							{
							case CCDLIB_FiltRectangleConst:
								filtmask[ y ][ x ] = 1.;
								break;
							case CCDLIB_FiltTriangleConst:
								filtmask[ y ][ x ] = radius/filtradius;
								break;
							case CCDLIB_FiltGaussConst:
								filtmask[ y ][ x ] = exp( -(radius*radius)/rectant );
								break;
							}
							filtmasksum += fabs( filtmask[ y ][ x ] );
						}
						else
							filtmask[ y ][ x ] = 0.;
					}
				}
			}
		}

      /*** eigentliche Maskenberechnung ***/
      if( astigmatism <= 1 )
      {
		   rectant = (maskradius-rectradius)/3.;
		   for( y=0; y<maskdiameter; y++ )
		   {
			   ysqr = (double)(y-indexhalf)+.5;
			   ysqr = ysqr*ysqr;

			   for( x=0; x<maskdiameter; x++ )
			   {
				   xsqr = (double)(x-indexhalf)+.5;
				   xsqr = xsqr*xsqr;
				   radius = sqrt( xsqr+ysqr );

				   if( radius <= maskradius )
				   {
					   switch( operationtype )
					   {
		            case CCDLIB_SharpGaussConst:
			            mask[ y ][ x ] = -exp( -(radius*radius)/(maskradius/3.) );
                     break;
					   case CCDLIB_SharpRectangleConst:
						   mask[ y ][ x ] = -1.;
						   break;
					   case CCDLIB_SharpTriangleConst:
						   mask[ y ][ x ] = -radius/(double)indexhalf;
						   break;
					   case CCDLIB_SharpGaussRectConst:
						   if( radius < rectradius )
							   mask[ y ][ x ] = -1.;
						   else
						   {
							   arg = radius-rectradius;
							   mask[ y ][ x ] = -exp( -(arg*arg)/rectant );
						   }
						   break;
					   case CCDLIB_MeanValueConst:
					   case CCDLIB_MaskedMedianConst:
						   mask[ y ][ x ] = 1.;
						   break;
					   case CCDLIB_LorentzMaskConst:
						   mask[ y ][ x ] = -1./((double)(xsqr*ysqr)*maskwidth+1.);
						   break;
					   }
					   masksum += fabs( mask[ y ][ x ] );
				   }
				   else
					   mask[ y ][ x ] = 0.;
			   }
		   }
      }
      else
      {
		   for( y=0; y<matrixsize; y++ )
   			for( x=0; x<matrixsize; x++ )
   				mask[ y ][ x ] = 0.;

		   rectant = (maskradius-rectradius)/3.;

         for( i=0; i<astigmatism; i++ )
         {
            astigrad = (double)(astigmatism-i-1) - (double)(astigmatism-1)/2.;

            xar = astigrad*cos(astigposwin)+(double)(indexhalf-maskradius);
            yar = (double)(indexhalf-maskradius)-astigrad*sin(astigposwin);

            for( j=0; j<4; j++ )
            {
               switch( j )
               {
               case 0:
                  f_astig = frac(xar)*frac(yar);
                  xa = (long)xar;
                  ya = (long)yar;
                  break;
               case 1:
                  f_astig = (1.-frac(xar))*frac(yar);
                  xa = (long)xar+1;
                  ya = (long)yar;
                  break;
               case 2:
                  f_astig = frac(xar)*(1.-frac(yar));
                  xa = (long)xar;
                  ya = (long)yar+1;
                  break;
               case 3:
                  f_astig = (1.-frac(xar))*(1.-frac(yar));
                  xa = (long)xar+1;
                  ya = (long)yar+1;
                  break;
               }

		         for( y=0, ym=ya; y<maskdiameter; y++, ym++ )
		         {
                  if( ym < 0 ) continue;
                  if( ym >= matrixsize ) continue;
			         ysqr = (double)(y-maskradius)+.5;
			         ysqr = ysqr*ysqr;

			         for( x=0, xm=xa; x<maskdiameter; x++, xm++ )
			         {
                     if( xm < 0 ) continue;
                     if( xm >= matrixsize ) continue;
				         xsqr = (double)(x-maskradius)+.5;
				         xsqr = xsqr*xsqr;
				         radius = sqrt( xsqr+ysqr );

				         if( radius <= maskradius )
				         {
					         switch( operationtype )
					         {
		                  case CCDLIB_SharpGaussConst:
			                  mask[ ym ][ xm ] -= exp( -(radius*radius)/(maskradius/3.) )*f_astig;
                           break;
					         case CCDLIB_SharpRectangleConst:
						         mask[ ym ][ xm ] -= f_astig;
						         break;
					         case CCDLIB_SharpTriangleConst:
						         mask[ ym ][ xm ] -= radius/(double)indexhalf*f_astig;
						         break;
					         case CCDLIB_SharpGaussRectConst:
						         if( radius < rectradius )
							         mask[ ym ][ xm ] -= f_astig;
						         else
						         {
							         arg = radius-rectradius;
							         mask[ ym ][ xm ] -= exp( -(arg*arg)/rectant )*f_astig;
						         }
						         break;
					         case CCDLIB_LorentzMaskConst:
						         mask[ ym ][ xm ] -= 1./((double)(xsqr*ysqr)*maskwidth+1.)*f_astig;
						         break;
					         }
   				      }
			         }
		         } /* Ende maske fr ein pixel ***/
            } /* Vier Pixel-Gruppe */
         } /* alle astigmatischen Punkte ... */
         for( y=0; y<matrixsize; y++ )
   			for( x=0; x<matrixsize; x++ )
   				masksum += fabs(mask[ y ][ x ]);
      }

      /*** Rauschfilteranteil (Tiefpass) berlagern ***/

		if( filtmasksum == 0 )
		{
			switch( operationtype )
			{
         case CCDLIB_SharpGaussConst:
			case CCDLIB_SharpRectangleConst:
			case CCDLIB_SharpTriangleConst:
			case CCDLIB_SharpGaussRectConst:
			case CCDLIB_LorentzMaskConst:
				centralval = masksum * (1. + 1./effiency);
				divfaktor = centralval - masksum;
				mask[ indexhalf ][ indexhalf ] += centralval;
				break;
			default:
				divfaktor = masksum;
				break;
			}
		}
		else
		{
			filtfactor = masksum/filtmasksum * (1. + 1./effiency);
			for( y=0, ys=filtdif; y<filtdiameter; y++, ys++ )
			{
				for( x=0, xs=filtdif; x<filtdiameter; x++, xs++ )
				{
					mask[ ys ][ xs ] += filtmask[ y ][ x ]*filtfactor;
				}
			}

			divfaktor = 0;
			for( y=0; y<matrixsize; y++ )
			{
				for( x=0; x<matrixsize; x++ )
				{
					divfaktor += mask[ y ][ x ];
				}
			}
		}

		for( y=0; y<matrixsize; y++ )
		{
			for( x=0; x<matrixsize; x++ )
			{
				mask[ y ][ x ] /= divfaktor;
			}
		}
	}
	else
	{
		switch( operationtype )
		{
		case CCDLIB_OpNGradientConst:
			mask[ 0 ][ 0 ] = 1.;  mask[ 0 ][ 1 ] = 1.;  mask[ 0 ][ 2 ] = 1.;
			mask[ 1 ][ 0 ] = 0.;  mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = 0.;
			mask[ 2 ][ 0 ] = -1.; mask[ 2 ][ 1 ] = -1.; mask[ 2 ][ 2 ] = -1.;
			break;
		case CCDLIB_OpNEGradientConst:
			mask[ 0 ][ 0 ] = 0.;  mask[ 0 ][ 1 ] = 1.;  mask[ 0 ][ 2 ] = 1.;
			mask[ 1 ][ 0 ] = -1.; mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = 1.;
			mask[ 2 ][ 0 ] = -1.; mask[ 2 ][ 1 ] = -1.; mask[ 2 ][ 2 ] = 0.;
			break;
		case CCDLIB_OpEGradientConst:
			mask[ 0 ][ 0 ] = -1.; mask[ 0 ][ 1 ] = 0.;  mask[ 0 ][ 2 ] = 1.;
			mask[ 1 ][ 0 ] = -1.; mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = 1.;
			mask[ 2 ][ 0 ] = -1.; mask[ 2 ][ 1 ] = 0.;  mask[ 2 ][ 2 ] = 1.;
			break;
		case CCDLIB_OpSEGradientConst:
			mask[ 0 ][ 0 ] = -1.; mask[ 0 ][ 1 ] = -1.; mask[ 0 ][ 2 ] = 0.;
			mask[ 1 ][ 0 ] = -1.; mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = 1.;
			mask[ 2 ][ 0 ] = 0.;  mask[ 2 ][ 1 ] = 1.;  mask[ 2 ][ 2 ] = 1.;
			break;
		case CCDLIB_OpSGradientConst:
			mask[ 0 ][ 0 ] = -1.; mask[ 0 ][ 1 ] = -1.; mask[ 0 ][ 2 ] = -1.;
			mask[ 1 ][ 0 ] = 0.;  mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = 0.;
			mask[ 2 ][ 0 ] = 1.;  mask[ 2 ][ 1 ] = 1.;  mask[ 2 ][ 2 ] = 1.;
			break;
		case CCDLIB_OpSWGradientConst:
			mask[ 0 ][ 0 ] = 0.;  mask[ 0 ][ 1 ] = -1.; mask[ 0 ][ 2 ] = -1.;
			mask[ 1 ][ 0 ] = 1.;  mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = -1.;
			mask[ 2 ][ 0 ] = 1.;  mask[ 2 ][ 1 ] = 1.;  mask[ 2 ][ 2 ] = 0.;
			break;
		case CCDLIB_OpWGradientConst:
			mask[ 0 ][ 0 ] = 1.;  mask[ 0 ][ 1 ] = 0.;  mask[ 0 ][ 2 ] = -1.;
			mask[ 1 ][ 0 ] = 1.;  mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = -1.;
			mask[ 2 ][ 0 ] = 1.;  mask[ 2 ][ 1 ] = 0.;  mask[ 2 ][ 2 ] = -1.;
			break;
		case CCDLIB_OpNWGradientConst:
			mask[ 0 ][ 0 ] = 1.;  mask[ 0 ][ 1 ] = 1.;  mask[ 0 ][ 2 ] = 0.;
			mask[ 1 ][ 0 ] = 1.;  mask[ 1 ][ 1 ] = 0.;  mask[ 1 ][ 2 ] = -1.;
			mask[ 2 ][ 0 ] = 0.;  mask[ 2 ][ 1 ] = -1.; mask[ 2 ][ 2 ] = -1.;
			break;
		}
	}
	if( filtmask != NULL ) CCDLIB_Matrixfree( filtmask );

	return( CCDLIB_Successful );
}

/*
static short CCDLIB_GetMedianFromArray
(
  short *values_pd,
  long valuenumber_n
)
{
  	short t;
   long  i, j, istop;
   BOOL  tausch;

   istop = valuenumber_n-1;
   do
   {
      tausch = FALSE;
      for( i=0, j=1; i<istop; i++, j++ )
      {
         if( values_pd[ i ] < values_pd[ j ] )
         {
      	   t              = values_pd[ i ];
      	   values_pd[ i ] = values_pd[ j ];
			   values_pd[ j ] = t;

            tausch = TRUE;
         }
      }
   } while( tausch == TRUE );

   return( values_pd[ valuenumber_n/2 ] );
}
*/

static short CCDLIB_GetMedianFromArray
(
  short *values_pd,
  long valuenumber_n
)
{
  short v,t;
  long  i,j, l,r, k;

  l = 0;
  r = valuenumber_n-1;
  k = valuenumber_n/2;

  while( r>l )
  {
	 v = values_pd[ r ];
	 i = l - 1;
	 j = r;

	 do
    {
      do
      {
		  i++;
      }
      while( values_pd[ i ] < v );

		do
      {
        j--;
      }
      while( values_pd[ j ] > v );

      t              = values_pd[ i ];
      values_pd[ i ] = values_pd[ j ];
		values_pd[ j ] = t;
    }
    while( j > i );

    values_pd[ j ] = values_pd[ i ];
	 values_pd[ i ] = values_pd[ r ];
	 values_pd[ r ] = t;

	 if( i >= k ) r=i-1;
    if( i <= k ) l=i+1;

  }
  v = values_pd[ k ];

  return( v );
}


/************************************************************************
*                                                                       *
*	 CCDLIB_LineFilterOperation : Universelle Filteroperation fuer das   *
*               zeilenweise Filtern von Bildern                         *
*                                                                       *
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               maskdiameter  :	Durchmesser des Filtermaskenkreis      *
*               CCDLIB_TiefpassTypEnum : Art der Filteroperation        *
*                                                                       *
*               ACHTUNG: maskdiameter darf nur ungerade Werte annehmen! *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild steht an derselben Stelle wie das Ausgangsbild.    *
*   ACHTUNG: Das Ausgangsbild wird dabei ueberschrieben !               *
*                                                                       *
*   Version:    1.1 (23.09.1997)                             32 Bit     *
*   Written by: Georg Dittie, 1994-1997                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_LineFilterOperation
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	unsigned long maskdiameter,
	enum CCDLIB_OperationTypeEnum operationtype
)
{
	long x, yk, ys, yks, i,
		  xstart, x1stop, y1stop, ystep,
		  index, indexhalf, indexturn, bufferindex;

	double temporary, dynamikmone, divfaktor;

	short  sortarray[ CCDLIB_ConvolutionWidthConst ];

	short * bufferline[ CCDLIB_ConvolutionWidthConst ];

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( maskdiameter > CCDLIB_ConvolutionWidthConst )
		return( CCDLIB_WrongParameterError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		ystep = pictureparam->linelength;
	else
		ystep = pictureparam->linelength*3;

	dynamikmone = (double)(pictureparam->dynamik - 1);
	divfaktor   = 1. / (double)maskdiameter;

	bufferline[ 0 ] = (short *)malloc(
		ystep * sizeof(short) * CCDLIB_ConvolutionWidthConst );

	if( bufferline[ 0 ] == NULL )
		return( CCDLIB_OutOfMemoryError );

	for( i=0; i<CCDLIB_ConvolutionWidthConst; i++ )
	{
		bufferline[ i ] = bufferline[ 0 ] + i * ystep;
	}

	indexhalf  = maskdiameter / 2;
	indexturn  = indexhalf + 1;
	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		x1stop = (long)(picturerect->xpos + picturerect->xsize);
		xstart = (long)(picturerect->xpos) + indexhalf;
	}
	else
	{
		x1stop = (long)(picturerect->xpos + picturerect->xsize)*3;
		xstart = (long)(picturerect->xpos + indexhalf)*3;
	}
	y1stop   = (long)(picturerect->ypos + picturerect->ysize - indexhalf) * ystep;

	index = 0;
	for( ys = (long)(picturerect->ypos + indexhalf) * ystep;
		  ys < y1stop;
		  ys += ystep )
	{
		bufferindex = index % indexturn;

		if( index >= indexturn )
		{
			for( x = xstart; x < x1stop; x++ )
				destpic[ ys - indexturn*ystep + x ] = bufferline[ bufferindex ][ x ];
		}

		for( x = xstart; x < x1stop; x++ )
		{
			switch( operationtype )
			{
			case CCDLIB_LPLinemeanConst:
				temporary= 0L;
				for( yks = -indexhalf*ystep;
					  yks <= indexhalf*ystep;
					  yks += ystep )
				{
					temporary += (double)sourcepic[ x + yks ] * divfaktor;
				}
				bufferline[ bufferindex ][ x ] = (short)(
					max( 0, min( dynamikmone, temporary ) ) );
				break;

			case CCDLIB_LPLinemedianConst:
				for( yks = -indexhalf*ystep, yk = 0;
					  yks <= indexhalf*ystep;
					  yks += ystep, yk++ )
				{
					sortarray[ yk ] = (short)sourcepic[ x + yks ];
				}
				bufferline[ bufferindex ][ x ] = (short)(
					CCDLIB_GetMedianFromArray( sortarray, maskdiameter ) );
				break;
			default:
				return( CCDLIB_WrongParameterError );
			}
		}
		index++;
	}

	for( ys = y1stop;
		  ys < y1stop + indexhalf*ystep;
		  ys += ystep)
	{
		for( x = xstart; x < x1stop; x++ )
		{
			destpic[ ys - indexturn*ystep + x ] =
				bufferline[ index%indexturn ][ x ];
		}
		index++;
	}

	free( bufferline[ 0 ] );

  	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_RemoveHotpixel : Routine entfernt einzelne bekannte heie    *
*               Pixel nach einer Koordinatentabelle                     *
*																								*
*   Parameter:  * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               hotpixelpos   :	Zeiger auf Koordinatenfeld             *
*               hotpixelnum   :	Anzahl der zu entfernenden Pixel       *
*               pseudoflag    :	Pseudoseparation nicht s. Filtertypen  *
*                                                                       *
*               ACHTUNG: filterwidth und filterheight darf nur ungerade *
*                        Werte annehmen! 											*
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild kann an derselben Stelle wie das Ausgangsbild      *
*   stehen, muss aber nicht: Beides geht.                               *
*                                                                       *
*   Version:    1.0 (7.09.2002)                              32 Bit     *
*   Written by: Georg Dittie, 2002                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_RemoveHotpixel
(
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	CCDLIB_PositionStruct *hotpixelpos,
   long hotpixelnum
)
{
   long i, xpos, ypos, xm1, xp1, ym1, ystep, yp1;
   short sortvals[8];

   ystep = pictureparam->linelength;
   for( i=0; i<hotpixelnum; i++ )
   {
      xpos = hotpixelpos[i].xpos;
      ypos = hotpixelpos[i].ypos;

      if( xpos<1 || xpos>=ystep-1 )
         continue;
      if( ypos<1 || xpos>=pictureparam->columnheight-1 )
         continue;

      xm1 = xpos-1;
      xp1 = xpos+1;
      ym1 = (ypos-1)*ystep;
      yp1 = (ypos+1)*ystep;
      ypos *= ystep;

      sortvals[0] = sourcepic[ xm1+ym1 ];
      sortvals[1] = sourcepic[ xm1+ypos ];
      sortvals[2] = sourcepic[ xm1+yp1 ];
      sortvals[3] = sourcepic[ xpos+ym1 ];
      sortvals[4] = sourcepic[ xpos+yp1 ];
      sortvals[5] = sourcepic[ xp1+ym1 ];
      sortvals[6] = sourcepic[ xp1+ypos ];
      sortvals[7] = sourcepic[ xp1+yp1 ];

      destpic[ xpos+ypos ] = CCDLIB_GetMedianFromArray( sortvals, 8 );
   }
   
  	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_NoisefilterFunction : Routine zum Tiefpassfiltern von ver-   *
*               rauschten Bildern mit separierbaren und nichtseparier-  *
*               baren Filterfunktionen                                  *
*																								*
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*               * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*               filtertype    :	Typ des separablen Tiefpass            *
*                                                                       *
*               ACHTUNG: filterwidth und filterheight darf nur ungerade *
*                        Werte annehmen! 											*
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild kann an derselben Stelle wie das Ausgangsbild      *
*   stehen, muss aber nicht: Beides geht.                               *
*                                                                       *
*   Version:    1.2 (22.09.1997)                              32 Bit    *
*   Written by: Georg Dittie, 1994-1997                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_NoisefilterFunction
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	double sigmalimit,
	enum CCDLIB_NoiseFilterEnum filtertype,
	BOOL pseudoflag
)
{
	double conv1Dh[ CCDLIB_ConvolutionWidthConst ];
	double conv1Dv[ CCDLIB_ConvolutionWidthConst ];
	double **conv2D;
	double temporary, tempr, tempg, tempb, dynamikmone;

	short *bufferline[ CCDLIB_ConvolutionWidthConst ];
	short sortarray[ CCDLIB_ConvolutionWidthConst * CCDLIB_ConvolutionWidthConst ];

	long x, xk, xc, y, yk, yc, ys, xks, yks, i, n,
		  xstart, xstartk, ystart, ysstart, ystartk,
		  xstop, ystop, ysstop, xstopk, ystopk, ystep,
		  windexhalf, hindexhalf, hindexturn, index, bufferindex;
	long filterwidth, filterheight;

	enum CCDLIB_ErrorEnum result;

	double *line;
	double *col;


	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	switch( filtertype )
	{
	case CCDLIB_Noise3x3MedianConst:
	case CCDLIB_Noise3x3SigmaConst:
	case CCDLIB_Noise3x3MittelConst:
		filterwidth=3; filterheight=3;
		break;
	case CCDLIB_Noise5x3MedianConst:
		filterwidth=5; filterheight=3;
		break;
	case CCDLIB_Noise5x5MedianConst:
	case CCDLIB_Noise5x5MittelConst:
	case CCDLIB_Noise5x5SigmaConst:
		filterwidth=5; filterheight=5;
		break;
	case CCDLIB_Noise7x5MedianConst:
		filterwidth=7; filterheight=5;
		break;
	case CCDLIB_NoiseZeilenMedianConst:
		return ( CCDLIB_LineFilterOperation( picturerect, pictureparam,
					 sourcepic, destpic, 3, CCDLIB_LPLinemedianConst ) );
	case CCDLIB_NoiseZeilenMittelConst:
		return ( CCDLIB_LineFilterOperation( picturerect, pictureparam,
					 sourcepic, destpic, 3, CCDLIB_LPLinemeanConst ) );
	default:
		return( CCDLIB_WrongParameterError );
	}

	windexhalf = filterwidth / 2;
	hindexhalf = filterheight / 2;
	dynamikmone = (double)(pictureparam->dynamik - 1);

	/* Zunaechst separable Verfahren */

	if( pseudoflag == TRUE &&
		 filtertype != CCDLIB_Noise3x3SigmaConst &&
		 filtertype != CCDLIB_Noise5x5SigmaConst )
	{
		if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
		{
			line = ( double * )malloc( pictureparam->linelength * sizeof( double ) );
			col  = ( double * )malloc( pictureparam->columnheight * sizeof( double ) );
		}
		else
		{
			line = ( double * )malloc( pictureparam->linelength * sizeof( double ) *3);
			col  = ( double * )malloc( pictureparam->columnheight * sizeof( double ));
		}

		if( line == NULL || col == NULL )
		{
			if( line != NULL ) free( line );
			if( col  != NULL ) free( col );

			return( CCDLIB_OutOfMemoryError );
		}

		/* Schleifengrenzen fuer separable Verfahren */

		xstart   = (long)(picturerect->xpos);
		ystart   = (long)(picturerect->ypos);
		ysstart  = (long)(picturerect->ypos * pictureparam->linelength);
		xstartk  = (long)(picturerect->xpos)+windexhalf;
		ystartk  = (long)(picturerect->ypos)+hindexhalf;
		xstop    = (long)(picturerect->xpos + picturerect->xsize);
		ystop    = (long)(picturerect->ypos + picturerect->ysize);
		ysstop   = (long)(picturerect->ypos + picturerect->ysize) * pictureparam->linelength;
		ystep    = (long)pictureparam->linelength;
		xstopk   = xstop-windexhalf;
		ystopk   = ystop-hindexhalf;

		if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
		{
			xstart*=3; xstartk*=3; xstop*=3; xstopk*=3;
			ysstart*=3; ysstop*=3; ystep*=3;
		}

		result = CCDLIB_GetSeparableNoiseFilterMask(
			filterwidth, filterheight, filtertype, conv1Dh, conv1Dv );

		if( result != CCDLIB_Successful )
		{
			free( line );
			free( col );
			return( result );
		}

		/* Horizontale Bearbeitung */

		for( ys = ysstart; ys < ysstop; ys += ystep )
		{
			for( x = xstart; x < xstop; x++ )
			{
				line[ x ] = (double)sourcepic[ x+ys ];
			}

			if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
			{
				switch( filtertype )
				{
				case CCDLIB_Noise3x3MittelConst:
				case CCDLIB_Noise5x5MittelConst:

					for( x = xstartk; x < xstopk; x ++ )
					{
						temporary = 0L;
						for( xk=x-windexhalf, xc=0; xc<filterwidth; xk++, xc++ )
						{
							temporary += line[ xk ] * conv1Dh[ xc ];
						}
						destpic[ ys+x ] = (short)( temporary<0 ? 0 : temporary );
					}
					break;
				case CCDLIB_Noise3x3MedianConst:
				case CCDLIB_Noise5x3MedianConst:
				case CCDLIB_Noise5x5MedianConst:
				case CCDLIB_Noise7x5MedianConst:

					for( x = xstartk; x < xstopk; x ++ )
					{
						for( xk=x-windexhalf, xc=0; xc<filterwidth; xk++, xc++ )
						{
							sortarray[ xc ] = line[ xk ];
						}
						destpic[ ys+x ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, filterwidth ) );
					}
					break;
				default:
					return( CCDLIB_WrongParameterError );
				}
			}
			else
			{
				switch( filtertype )
				{
				case CCDLIB_Noise3x3MittelConst:
				case CCDLIB_Noise5x5MittelConst:

					for( x = xstartk; x < xstopk; x +=3 )
					{
						tempr=0L; tempg=0L; tempb=0L;
						for( xk=x-windexhalf*3, xc=0; xc<filterwidth; xk+=3, xc++ )
						{
							tempr += line[ xk+2 ] * conv1Dh[ xc ];
							tempg += line[ xk+1 ] * conv1Dh[ xc ];
							tempb += line[ xk   ] * conv1Dh[ xc ];
						}
                  destpic[ ys+x ]   = (short)( tempb<0 ? 0 : tempb );
                  destpic[ ys+x+1 ] = (short)( tempg<0 ? 0 : tempg );
                  destpic[ ys+x+2 ] = (short)( tempr<0 ? 0 : tempr );
					}
					break;
				case CCDLIB_Noise3x3MedianConst:
				case CCDLIB_Noise5x3MedianConst:
				case CCDLIB_Noise5x5MedianConst:
				case CCDLIB_Noise7x5MedianConst:

					for( x = xstartk; x < xstopk; x += 3 )
					{
						for( xk=x-windexhalf*3, xc=0; xc<filterwidth; xk+=3, xc++ )
						{
							sortarray[ xc ] = (short)line[ xk ];
						}
						destpic[ ys+x ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, filterwidth ) );

						for( xk=x-windexhalf*3+1, xc=0; xc<filterwidth; xk+=3, xc++ )
						{
							sortarray[ xc ] = (short)line[ xk ];
						}
						destpic[ ys+x+1 ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, filterwidth ) );

						for( xk=x-windexhalf*3+2, xc=0; xc<filterwidth; xk+=3, xc++ )
						{
							sortarray[ xc ] = (short)line[ xk ];
						}
						destpic[ ys+x+2 ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, filterwidth ) );
					}
					break;
				default:
					return( CCDLIB_WrongParameterError );
				}
			}
		}

		/* Vertikale Bearbeitung */

		for( x = xstart;
			  x < xstop;
			  x ++ )
		{
			for( y = ystart, ys = ysstart;
				  y < ystop;
				  y++, ys += ystep )
			{
				col[ y ] = (double)sourcepic[ x+ys ];
			}

			switch( filtertype )
			{
			case CCDLIB_Noise3x3MittelConst:
			case CCDLIB_Noise5x5MittelConst:

				for( y = ystartk, ys = ystartk * ystep;
					  y < ystopk;
					  y ++, ys += ystep )
				{
					temporary = 0L;
					for( yk=y-hindexhalf, yc=0; yc<filterheight; yk++, yc++ )
					{
						temporary += col[ yk ] * conv1Dv[ yc ];
					}
					destpic[ x+ys ] = (short)( temporary<0 ? 0 : temporary );
				}
				break;
			case CCDLIB_Noise3x3MedianConst:
			case CCDLIB_Noise5x3MedianConst:
			case CCDLIB_Noise5x5MedianConst:
			case CCDLIB_Noise7x5MedianConst:

				for( y = ystartk, ys = ystartk * ystep;
					  y < ystopk;
					  y ++, ys += ystep )
				{
					for( yk=y-hindexhalf, yc=0; yc<filterheight; yk++, yc++ )
					{
						sortarray[ yc ] = (short)col[ yk ];
					}
					destpic[ x+ys ] = (short)(
						CCDLIB_GetMedianFromArray( sortarray, filterheight ) );
				}
				break;
			default:
				return( CCDLIB_WrongParameterError );
			}
		}

		free( line );
		free( col );
	}
	else     /* Nicht separable Filterverfahren */
	{
		/* Schleifengrenzen fuer nicht separable Verfahren */

		xstart   = (long)(picturerect->xpos)+windexhalf;
		ysstart  = (long)(picturerect->ypos+hindexhalf) * pictureparam->linelength;
		xstop    = (long)(picturerect->xpos + picturerect->xsize - windexhalf);
		ysstop   = (long)(picturerect->ypos + picturerect->ysize - hindexhalf) * pictureparam->linelength;
		ystep    = (long)pictureparam->linelength;

		if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
		{
			xstart*=3;  xstop*=3;
			ysstart*=3; ysstop*=3; ystep*=3;
		}

		bufferline[ 0 ] = (short *)malloc(
			ystep * sizeof(short) * CCDLIB_ConvolutionWidthConst );

		if( bufferline[ 0 ] == NULL )
			return( CCDLIB_OutOfMemoryError );

		for( i = 0; i < CCDLIB_ConvolutionWidthConst; i ++ )
		{
			bufferline[ i ] = bufferline[ 0 ] + i * ystep;
		}

		hindexturn  = hindexhalf + 1;

		if( !CCDLIB_Matrixalloc( &conv2D, CCDLIB_ConvolutionWidthConst ) )
		{
			free( bufferline[0] );
			return( CCDLIB_OutOfMemoryError );
		}
		CCDLIB_GetNotSeparableNoiseFilterMask(
			filterwidth, filterheight, filtertype, conv2D );

		index = 0;
		for( ys = ysstart; ys < ysstop; ys += ystep)
		{
			bufferindex = index % hindexturn;

			if( index >= hindexturn )
			{
				for( x = xstart; x < xstop; x++ )
				{
					destpic[ ys - hindexturn*ystep + x ] =
						bufferline[ bufferindex ][ x ];
				}
			}

			if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
			{
				for( x = xstart; x < xstop; x++ )
				{
					switch( filtertype )
					{
					case CCDLIB_Noise3x3MittelConst:
					case CCDLIB_Noise5x5MittelConst:
						temporary= 0L;
						for(yks = -hindexhalf*ystep, yk=0;
							 yks <= hindexhalf*ystep;
							 yks += ystep, yk++ )
						{
							for(xks = -windexhalf, xk=0;
								 xks <= windexhalf;
								 xks++, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
									temporary += (double)sourcepic[ x+xks + ys+yks ] * conv2D[ yk ][ xk ];
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							max( 0, min( dynamikmone, temporary ) ) );

						break;

					case CCDLIB_Noise3x3MedianConst:
					case CCDLIB_Noise5x3MedianConst:
					case CCDLIB_Noise5x5MedianConst:
					case CCDLIB_Noise7x5MedianConst:
						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf, xk=0;
								  xks <= windexhalf;
								  xks ++, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, n ) );

						break;
					case CCDLIB_Noise3x3SigmaConst:
					case CCDLIB_Noise5x5SigmaConst:
						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf, xk=0;
								  xks <= windexhalf;
								  xks ++, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							CCDLIB_RemoveGreaterThanSigmaFromArray(
							sortarray, sourcepic[ x + ys ], n, sigmalimit ) );

						break;
					default:
						return( CCDLIB_WrongParameterError );
					}
				}
			}
			else    /* Nichtseparables Filtern von RGB-Bildern */
			{
				for( x = xstart; x < xstop; x+=3 )
				{
					switch( filtertype )
					{
					case CCDLIB_Noise3x3MittelConst:
					case CCDLIB_Noise5x5MittelConst:
						tempb=0.; tempg=0.; tempr=0.;
						for(yks = -hindexhalf*ystep, yk=0;
							 yks <= hindexhalf*ystep;
							 yks += ystep, yk++ )
						{
							for(xks = -windexhalf*3, xk=0;
								 xks <= windexhalf*3;
								 xks+=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									tempb += (double)sourcepic[ x+xks + ys+yks   ] * conv2D[ yk ][ xk ];
									tempg += (double)sourcepic[ x+xks + ys+yks+1 ] * conv2D[ yk ][ xk ];
									tempr += (double)sourcepic[ x+xks + ys+yks+2 ] * conv2D[ yk ][ xk ];
								}
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							max( 0, min( dynamikmone, tempb ) ) );
						bufferline[ bufferindex ][ x+1 ] = (short)(
							max( 0, min( dynamikmone, tempg ) ) );
						bufferline[ bufferindex ][ x+2 ] = (short)(
							max( 0, min( dynamikmone, tempr ) ) );

						break;

					case CCDLIB_Noise3x3MedianConst:
					case CCDLIB_Noise5x3MedianConst:
					case CCDLIB_Noise5x5MedianConst:
					case CCDLIB_Noise7x5MedianConst:
						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, n ) );

						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3+1, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x+1 ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, n ) );

						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3+2, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x+2 ] = (short)(
							CCDLIB_GetMedianFromArray( sortarray, n ) );

						break;
					case CCDLIB_Noise3x3SigmaConst:
					case CCDLIB_Noise5x5SigmaConst:
						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x ] = (short)(
							CCDLIB_RemoveGreaterThanSigmaFromArray(
							sortarray, sourcepic[ x + ys ], n, sigmalimit ) );

						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3+1, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x+1 ] = (short)(
							CCDLIB_RemoveGreaterThanSigmaFromArray(
							sortarray, sourcepic[ x+1 + ys ], n, sigmalimit ) );

						for( yks = -hindexhalf*ystep, n=0, yk=0;
							  yks <= hindexhalf*ystep;
							  yks += ystep, yk++ )
						{
							for( xks = -windexhalf*3+2, xk=0;
								  xks <= windexhalf*3;
								  xks +=3, xk++ )
							{
								if( conv2D[ yk ][ xk ] != 0 )
								{
									sortarray[ n ] = sourcepic[ x+xks + ys+yks ];
									n++;
								}
							}
						}
						bufferline[ bufferindex ][ x+2 ] = (short)(
							CCDLIB_RemoveGreaterThanSigmaFromArray(
							sortarray, sourcepic[ x+ 2 + ys ], n, sigmalimit ) );

						break;
					default:
						return( CCDLIB_WrongParameterError );
					}
				}
			}
			index++;
		}

		for(ys = ysstop;
			 ys < ysstop + hindexhalf*ystep;
			 ys += ystep)
		{
			for( x = xstart;
				  x < xstop;
				  x++ )
			{
				destpic[ ys - hindexturn*ystep + x ] =
					bufferline[ index%hindexturn ][ x ];
			}
			index++;
		}

      CCDLIB_Matrixfree( conv2D );
		free( bufferline[ 0 ] );
	}

	return( CCDLIB_Successful );
}

enum CCDLIB_ErrorEnum CCDLIB_GetSeparableNoiseFilterMask(
	long filterwidth,
	long filterheight,
	enum CCDLIB_NoiseFilterEnum filtertype,
	double *conv1Dh,
	double *conv1Dv )
{
	long i;

	switch( filtertype )
	{
	case CCDLIB_Noise3x3MedianConst:
	case CCDLIB_Noise5x3MedianConst:
	case CCDLIB_Noise5x5MedianConst:
	case CCDLIB_Noise7x5MedianConst:
		for( i=0; i<filterwidth; i++ )
			conv1Dh[ i ] = 1.;
		for( i=0; i<filterheight; i++ )
			conv1Dv[ i ] = 1.;
		break;
	case CCDLIB_Noise3x3MittelConst:
	case CCDLIB_Noise5x5MittelConst:
		for( i=0; i<filterwidth; i++ )
			conv1Dh[ i ] = 1. / (double)filterwidth;
		for( i=0; i<filterheight; i++ )
			conv1Dv[ i ] = 1. / (double)filterheight;
		break;
	default:
		return( CCDLIB_WrongParameterError );
	}

	return( CCDLIB_Successful );
}

enum CCDLIB_ErrorEnum CCDLIB_GetNotSeparableNoiseFilterMask(
	long filterwidth,
	long filterheight,
	enum CCDLIB_NoiseFilterEnum filtertype,
	double **conv2D
)
{
	long x, y, hh, wh, xdif, ydif, xsqr, ysqr;
	double areasum;

	hh = filterheight/2;
	wh = filterwidth/2;

	switch( filtertype )
	{
	case CCDLIB_Noise3x3MedianConst:
	case CCDLIB_Noise3x3SigmaConst:
	case CCDLIB_Noise3x3MittelConst:
		for( y=0; y<filterheight; y++ )
		{
			ydif = y-hh;
			ysqr = ydif*ydif;
			for( x=0; x<filterwidth; x++ )
			{
			  xdif = x-wh;
			  xsqr = xdif*xdif;

			  if( ysqr+xsqr < 2 )
				  conv2D[y][x] = 1.;
			  else
				  conv2D[y][x] = 0.;
			}
		}
		break;
	case CCDLIB_Noise5x3MedianConst:
	case CCDLIB_Noise5x5SigmaConst:
	case CCDLIB_Noise5x5MedianConst:
	case CCDLIB_Noise5x5MittelConst:
		for( y=0; y<filterheight; y++ )
		{
			ydif = y-hh;
			ysqr = ydif*ydif;
			for( x=0; x<filterwidth; x++ )
			{
			  xdif = x-wh;
			  xsqr = xdif*xdif;

			  if( ysqr+xsqr < 4 )
				  conv2D[y][x] = 1.;
			  else
				  conv2D[y][x] = 0.;
			}
		}
		break;
	case CCDLIB_Noise7x5MedianConst:
		for( y=0; y<filterheight; y++ )
		{
			ydif = y-hh;
			ysqr = ydif*ydif;
			for( x=0; x<filterwidth; x++ )
			{
			  xdif = x-wh;
			  xsqr = xdif*xdif;

			  if( ysqr+xsqr < 6 )
				  conv2D[y][x] = 1.;
			  else
				  conv2D[y][x] = 0.;
			}
		}
		break;
	default:
		return( CCDLIB_WrongParameterError );
	}

	switch( filtertype )
	{
	case CCDLIB_Noise3x3MittelConst:
	case CCDLIB_Noise5x5MittelConst:
		areasum = 0.;
		for( y=0; y<filterheight; y++ )
		{
			for( x=0; x<filterwidth; x++ )
			{
			  areasum += conv2D[y][x];
			}
		}
		for( y=0; y<filterheight; y++ )
		{
			for( x=0; x<filterwidth; x++ )
			{
			  conv2D[y][x] /= areasum;
			}
		}
		break;
	default:
		return( CCDLIB_WrongParameterError );
	}

	return( CCDLIB_Successful );
}

short CCDLIB_RemoveGreaterThanSigmaFromArray(
	short *sortarray,
	short greyvalue,
	long arraylength,
	double sigmalimit
)
{
	double mean = 0., sigmasum = 0., dif, sigma;
	long i;

	for( i=0; i<arraylength; i++ )
		mean += (double)sortarray[ i ];
   mean /= (double)arraylength;

	for( i=0; i<arraylength; i++ )
	{
		dif = (double)sortarray[ i ] - mean;
		sigmasum += (dif*dif);
	}

	if( sigmasum != 0. )
		sigma = sqrt( sigmasum / (double)(arraylength-1) );
	else
		return( (short)floor(mean+0.5) );

	if( fabs( (double)greyvalue-mean ) > sigma*sigmalimit )
		return( (short)floor(mean+0.5) );
	else
		return( greyvalue );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_ExtractLuminance : Routine zum Berechnen des Luminanzanteils *
*               nach ISO-Norm aus einem RGB-Farbbild                    *
*																								*
*   Parameter:  * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*               * destpic     :  32-Bit-Pointer auf die Luminanzdaten   *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild kann nicht an derselben Stelle wie das             *
*   Ausgangsbild stehen !                                               *
*                                                                       *
*   Version:    1.0 (12.05.1996)                                        *
*   Written by: Georg Dittie, 1996                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_ExtractLuminance
(
  short * sourcepicture,
  CCDLIB_PictureparamStruct * pictureparam,
  short * destpicture
)
{
   long i, ic, istop;
   double lumval;

   if( destpicture == NULL || sourcepicture == NULL )
	   return( CCDLIB_NoPicturePointerError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor ||
		 destpicture == sourcepicture )
		return( CCDLIB_WrongParameterError );

	istop = pictureparam->linelength*pictureparam->columnheight;

	ic = 0;
	for( i=0; i<istop; i++ )
	{
		lumval = 0.299*(double)sourcepicture[ ic + 2 ] +
					0.587*(double)sourcepicture[ ic + 1 ] +
					0.114*(double)sourcepicture[ ic ];

		destpicture[ i ] = (short)floor( lumval + 0.5 );

		ic += 3;
	}

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_SeparateColors : Routine zum Separieren einzelner Farbkanle *
*               aus einem RGB-Farbbild                                  *
*																								*
*   Parameter:  * sourcepic   :  32-Bit-Pointer auf die Quellbilddaten  *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*               * destpic     :  32-Bit-Pointer auf die Luminanzdaten   *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild kann nicht an derselben Stelle wie das             *
*   Ausgangsbild stehen !                                               *
*                                                                       *
*   Version:    1.0 (3.03.2001)                                        *
*   Written by: Georg Dittie, 2001                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_SeparateColors
(
   short * sourcepicture,
   CCDLIB_PictureparamStruct * pictureparam,
   enum CCDLIB_ColorSeparationEnum separmethod,
   short * channel1,
   short * channel2,
   short * channel3
)
{
   long i, ic, istop;

   if( channel1 == NULL || sourcepicture == NULL )
	   return( CCDLIB_NoPicturePointerError );
   if( separmethod == CCDLIB_SeparRGBConst )
      if( channel2 == NULL || channel3 == NULL )
	      return( CCDLIB_NoPicturePointerError );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor ||
		 channel1 == sourcepicture )
		return( CCDLIB_WrongParameterError );
   if( separmethod == CCDLIB_SeparRGBConst )
      if( channel2 == sourcepicture || channel3 == sourcepicture )
         return( CCDLIB_WrongParameterError );

	istop = pictureparam->linelength*pictureparam->columnheight;

   if( separmethod == CCDLIB_SeparLuminanceConst )
   {
      CCDLIB_ExtractLuminance(
         sourcepicture, pictureparam, channel1 );
      return( CCDLIB_Successful );
   }

	ic = 0;
   switch( separmethod )
   {
   case CCDLIB_SeparRGBConst:
	   for( i=0; i<istop; i++ )
	   {
		   channel1[ i ] = sourcepicture[ ic+2 ];
		   channel2[ i ] = sourcepicture[ ic+1 ];
		   channel3[ i ] = sourcepicture[ ic ];
		   ic += 3;
	   }
      break;
	case CCDLIB_SeparRedConst:
	   for( i=0; i<istop; i++ )
	   {
		   channel1[ i ] = sourcepicture[ ic+2 ];
		   ic += 3;
	   }
      break;
	case CCDLIB_SeparGreenConst:
	   for( i=0; i<istop; i++ )
	   {
		   channel1[ i ] = sourcepicture[ ic+1 ];
		   ic += 3;
	   }
      break;
	case CCDLIB_SeparBlueConst:
	   for( i=0; i<istop; i++ )
	   {
		   channel1[ i ] = sourcepicture[ ic ];
		   ic += 3;
	   }
      break;
   }

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_CombineColors : Routine zum Kombinieren einzelner Bilder     *
*               zu einem RGB-Farbbild oder rot/blau-Anaglyphenbild      *
*																								*
*   Parameter:  * destpic     :  32-Bit-Pointer auf die Zielbilddaten   *
*					 * redpic      :  32-Bit-Pointer auf die Quellbilddaten  *
*					 * greenpic    :  32-Bit-Pointer auf die Quellbilddaten  *
*					 * bluepic     :  32-Bit-Pointer auf die Quellbilddaten  *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*               method        :  Art der Bildverarbeitung               *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultatbild kann nicht an derselben Stelle wie das             *
*   Ausgangsbild stehen !                                               *
*                                                                       *
*   Version:    1.0 (24.08.2002)                                        *
*   Written by: Georg Dittie, 2002                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CombineColors
(
   short * destpic,
   CCDLIB_PictureparamStruct *pictureparam,
   enum CCDLIB_ColorCombineEnum combinemethod,
   short * redpic,
   short * greenpic,
   short * bluepic
)
{
   long i, ic, istop;

   if( destpic == NULL || redpic == NULL || bluepic == NULL )
	   return( CCDLIB_NoPicturePointerError );
   if( combinemethod == CCDLIB_CombineRGBConst )
   {
      if(  greenpic == NULL )
	      return( CCDLIB_NoPicturePointerError );
   }

	istop = pictureparam->linelength*pictureparam->columnheight;

	ic = 0;
   switch( combinemethod )
   {
	case CCDLIB_CombineRGBConst:
	   for( i=0; i<istop; i++ )
	   {
		   destpic[ ic   ] = bluepic[ i ];
		   destpic[ ic+1 ] = greenpic[ i ];
		   destpic[ ic+2 ] = redpic[ i ];
		   ic += 3;
	   }
      break;
	case CCDLIB_BWAnaglyphConst:
	   for( i=0; i<istop; i++ )
	   {
		   destpic[ ic   ] = bluepic[ i ];
		   destpic[ ic+1 ] = 0;
		   destpic[ ic+2 ] = redpic[ i ];
		   ic += 3;
	   }
      break;
	case CCDLIB_ColorAnaglyphConst:
 	   for( i=0; i<istop; i++ )
	   {
		   destpic[ ic   ] = bluepic[ ic ];
		   destpic[ ic+1 ] = (short)((bluepic[ ic+1 ]+redpic[ ic+1 ])/2);
		   destpic[ ic+2 ] = redpic[ ic+2 ];
		   ic += 3;
	   }
      break;
   }

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_FlatfieldCorrection: Routine zur Flatfieldkorrektur eines    *
*               Bildbereichs                                            *
*																								*
*   Parameter:  * destpicture    :  32-Bit-Pointer auf das Zielbild     *
*               * sourcepicture  :  32-Bit-Pointer auf das Quellbild    *
*               * flatfield      :  32-Bit-Pointer auf das Flatfield    *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (11.05.1996)                                        *
*   Written by: Georg Dittie, 1996                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_FlatfieldCorrection
(
  short * destpicture,
  short * sourcepicture,
  short * flatfield,
  CCDLIB_PictureparamStruct * pictureparam
)
{
   long   i, buflength;
	double flatmean;

   if( destpicture == NULL || sourcepicture == NULL || flatfield == NULL )
	   return( CCDLIB_NoPicturePointerError );

   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
      buflength = pictureparam->linelength*pictureparam->columnheight*3L;
   else
      buflength = pictureparam->linelength*pictureparam->columnheight;

   flatmean = 0.;
   for( i=0; i<buflength; i++ )
      flatmean += (double)flatfield[ i ];
   flatmean /= (double)buflength;

   for( i=0; i<buflength; i++ )
      if( flatfield[i] != 0 )
         destpicture[ i ] = (short)floor(
            (double)sourcepicture[ i ] * flatmean / (double)flatfield[ i ] + 0.5 );
      else
         destpicture[ i ] = sourcepicture[ i ];

  return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_DarkframeCorrection: Routine zur Darkframekorrektur eines    *
*               Bildbereichs                                            *
*																								*
*   Parameter:  * destpicture    :  32-Bit-Pointer auf das Zielbild     *
*               * sourcepicture  :  32-Bit-Pointer auf das Quellbild    *
*               * darkframe      :  32-Bit-Pointer auf das Flatfield    *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (11.05.1996)                                        *
*   Written by: Georg Dittie, 1996                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_DarkframeCorrection
(
  short * destpicture,
  short * sourcepicture,
  short * darkframe,
  CCDLIB_PictureparamStruct * pictureparam
)
{
   long  i, buflength;
   short temp;

   if( destpicture == NULL || sourcepicture == NULL || darkframe == NULL )
	   return( CCDLIB_NoPicturePointerError );

   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
      buflength = pictureparam->linelength*pictureparam->columnheight*3L;
   else
		buflength = pictureparam->linelength*pictureparam->columnheight;

   for( i=0; i<buflength; i++ )
   {
      temp = (short)(sourcepicture[ i ] - darkframe[ i ]);
      destpicture[ i ] = (short)( temp < 0 ? 0 : temp );
   }
   return( CCDLIB_Successful );
}


/************************************************************************
*                                                                       *
*	 CCDLIB_CountBrightsteps: Routine zum Ermitteln der Anzahl von Grau- *
*               bzw. Luminanz und Farbstufen eines Bildbereichs         *
*																								*
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf die Bilddaten       *
*               * greycount   :	Anzahl der Grauwerte                   *
*               * redcount    :	Anzahl der Rotwerte                    *
*               * greencount  :	Anzahl der Gruenwerte                  *
*               * bluecount   :	Anzahl der Blauwerte                   *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (11.10.1998)                                        *
*   Version:    1.1 (30.8.2002)                                         *
*   Written by: Georg Dittie, 1998 - 2002                               *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CountBrightsteps
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	long * greycount,
	long * redcount,
	long * greencount,
	long * bluecount,
   long * nullcount,
   long * negcount,
   long * greymax
)
{
	long x, ys, i, xstop, ystop, ystep, val;
	long *greyhist, *redhist, *greenhist, *bluehist;
   double dval;

	*greycount  = 0;
	*redcount   = 0;
	*greencount = 0;
	*bluecount  = 0;
   *nullcount  = 0;
	*negcount   = 0;
   *greymax    = 0;

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( picture == NULL )
		return( CCDLIB_NoPicturePointerError );

	xstop = (long)(picturerect->xpos + picturerect->xsize);
	ystop = (long)(picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

	greyhist = (long*)malloc( pictureparam->dynamik*sizeof(long) );
	if( greyhist == NULL )
		return( CCDLIB_OutOfMemoryError );
	memset( greyhist, 0, pictureparam->dynamik*sizeof(long) );

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for(ys = (long)(picturerect->ypos * pictureparam->linelength);
			 ys < ystop;
			 ys += pictureparam->linelength)
		{
			for(x = (long)(picturerect->xpos);
				 x < xstop;
				 x++ )
			{
            val = (long)picture[ ys+x ];
            if( val == 0 )
               (*nullcount)++;
            if( val < 0 )
            {
               (*negcount)++;
               val = 0;
            }
            if( val > 32767 ) val = 32767;
				greyhist[ val ] ++;
			}
		}
		for( i=0; i<pictureparam->dynamik; i++ )
			if( greyhist[ i ] > 0 )
         {
            *greymax = i;
				(*greycount)++;
         }
	}
	else  /* Luminanz-Histogram nach ISO-Norm */
	{
		redhist   = (long*)malloc( pictureparam->dynamik*sizeof(long) );
		greenhist = (long*)malloc( pictureparam->dynamik*sizeof(long) );
		bluehist  = (long*)malloc( pictureparam->dynamik*sizeof(long) );
		if( redhist == NULL || greenhist == NULL || bluehist == NULL )
		{
			if( greyhist != NULL )  free( greyhist );
			if( redhist != NULL )   free( redhist );
			if( greenhist != NULL ) free( greenhist );
			if( bluehist != NULL )  free( bluehist );
			return( CCDLIB_OutOfMemoryError );
		}
		memset( greyhist, 0, pictureparam->dynamik*sizeof(long) );
		memset( greyhist, 0, pictureparam->dynamik*sizeof(long) );
		memset( greyhist, 0, pictureparam->dynamik*sizeof(long) );

		xstop *= 3;
		ystop *= 3;
		ystep = pictureparam->linelength*3;
		for(ys = (long)(picturerect->ypos * pictureparam->linelength);
			 ys < ystop;
			 ys += ystep )
		{
			for(x = (long)(picturerect->xpos);
				 x < xstop;
				 x+=3 )
			{
            dval = 0.114*(double)picture[ ys+x ] +
					    0.587*(double)picture[ ys+x+1 ] +
					    0.299*(double)picture[ ys+x+2 ] + 0.5;
            if( dval < 0. ) dval = 0.;
            if( dval > 32767. ) dval = 32767.;
				greyhist[ (long)floor( dval ) ] ++;

            val = (long)picture[ ys+x ];
            if( val == 0 )
               (*nullcount)++;
            if( val < 0 )
            {
               (*negcount)++;
               val = 0;
            }
            if( val > 32767 ) val = 32767;
				bluehist[ val ] ++;

            val = (long)picture[ ys+x+1 ];
            if( val == 0 )
               (*nullcount)++;
            if( val < 0 )
            {
               (*negcount)++;
               val = 0;
            }
            if( val > 32767 ) val = 32767;
				greenhist[ val ] ++;

            val = (long)picture[ ys+x+2 ];
            if( val == 0 )
               (*nullcount)++;
            if( val < 0 )
            {
               (*negcount)++;
               val = 0;
            }
            if( val > 32767 ) val = 32767;
				redhist[ val ] ++;
			}
		}
		for( i=0; i<pictureparam->dynamik; i++ )
		{
			if( greyhist[ i ] > 0 )
         {
            *greymax = i;
				(*greycount)++;
         }
			if( redhist[ i ] > 0 )
				(*redcount)++;
			if( greenhist[ i ] > 0 )
				(*greencount)++;
			if( bluehist[ i ] > 0 )
				(*bluecount)++;
		}
	}

	if( greyhist != NULL )  free( greyhist );
	if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
	{
		if( redhist != NULL )   free( redhist );
		if( greenhist != NULL ) free( greenhist );
		if( bluehist != NULL )  free( bluehist );
	}

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_RetrieveHistogram: Routine zum Ermitteln eines Grauwertver-  *
*               teilungshistogramms eines Bildbereichs                  *
*																								*
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf die Bilddaten       *
*               * histogramfield :	Pointer auf Histogrammwerte         *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Das Resultat steht an der Stelle, wo histogramfield hin pointet.    *
*                                                                       *
*   Version:    1.1 (22.09.1997)                                        *
*   Written by: Georg Dittie, 1994-1997                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_RetrieveHistogram
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  long * histogramfield
)
{
  long x, ys, i, val, xstart, ystart, xstop, ystop, ystep;
  double dval;

  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL )
	 return( CCDLIB_NoPicturePointerError );

  if( histogramfield == NULL )
	 return( CCDLIB_NoPointerError );

  for( i=0; i<(long)pictureparam->dynamik; i++ )
	 histogramfield[ i ] = 0;

  xstart = picturerect->xpos;
  ystart = picturerect->ypos * pictureparam->linelength;
  xstop = picturerect->xpos + picturerect->xsize;
  ystop = (picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for(ys = ystart;
			 ys < ystop;
			 ys += pictureparam->linelength)
		{
			for(x = xstart;
				 x < xstop;
				 x++ )
			{
            val = (long)picture[ ys+x ];
            if( val < 0 ) val = 0;
            if( val >= pictureparam->dynamik ) val = pictureparam->dynamik-1;
				histogramfield[ val ] ++;
			}
		}
	}
	else  /* Luminanz-Histogram nach ISO-Norm */
	{
		xstart *= 3;
		ystart *= 3;
		xstop *= 3;
		ystop *= 3;
		ystep = pictureparam->linelength*3;
		for(ys = ystart;
			 ys < ystop;
			 ys += ystep )
		{
			for(x = xstart;
				 x < xstop;
				 x+=3 )
			{
            dval = 0.114*(double)picture[ ys+x ] +
					    0.587*(double)picture[ ys+x+1 ] +
					    0.299*(double)picture[ ys+x+2 ] + 0.5;
            if( dval >= (double)pictureparam->dynamik ) dval = (double)(pictureparam->dynamik-1);
            if( dval < 0. ) dval = 0.;
				histogramfield[ (long)floor( dval ) ] ++;
			}
		}
	}

	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_GetMaximumInHistogram: Routine zum Adhoc-Auswerten eines     *
*               Grauwertverteilungshistogramms eines Bildbereichs.      *
*																								*
*   Parameter:  * histogramfield :	Pointer auf Histogrammwerte         *
*               dynamik          :  Anzahl der Graustufen               *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*               * lowestposition  : Kleinster besetzter Grauwert        *
*               * highestposition : Groesster besetzter Grauwert        *
*               * maximumposition : Haeufgster besetzter Grauwert       *
*               * maximalvalue    : Anzahl der Pixel mit haeuf. Grauw.  *
*               * maximumpart     : Anteil des haeuf. Grauwert am Bild  *
*               * entropy         : Grauwert-Entropie des Bildes        *
*                                                                       *
*   Version:    1.0 (05.06.1994)                                        *
*   Written by: Georg Dittie, 1994                                      *
*                                                                       *
*	 ACHTUNG: Diese Funktion gibt es nur noch aus Kompatibilitaets-      *
*            gruenden. Bitte nicht mehr fuer Neuentwicklungen nutzen !  *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetMaximumInHistogram
(
  long * histogramfield,
  long dynamik,
  long * lowestposition,
  long * highestposition,
  long * maximumposition,
  long * maximalvalue,
  double * maximumpart,
  double * entropy
)
{
  long i, histogramsum;

  BOOL minfound, maxfound;

  if( histogramfield == NULL )
	 return( CCDLIB_NoPointerError );

  minfound = FALSE;
  maxfound = FALSE;

  *maximumposition = 1;
  *maximalvalue = histogramfield[ 1 ];
  histogramsum  = 0;

  *lowestposition = 0;
  for( i=1; i<dynamik; i++ )
  {
	 histogramsum += histogramfield[ i ];

	 if( histogramfield[ i ] > 0 && minfound == FALSE )
    {
      minfound = TRUE;
		*lowestposition = i;
    }

	 if( histogramfield[ i ] > *maximalvalue )
    {
		*maximalvalue = histogramfield[ i ];
		*maximumposition = i;
    }
  }

  *highestposition = dynamik-1;
  for( i=dynamik-1; i>0; i-- )
  {
	 if( histogramfield[ i ] > 0 && maxfound == FALSE )
    {
		*highestposition = i;
      maxfound = TRUE;
    }
  }

  *maximumpart = (double)*maximalvalue/(double)histogramsum;
  *entropy     = 1./log( 2. )
                 * log( (double)labs( *highestposition - *lowestposition + 1 ) );

  return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_EvaluateHistogram: Routine zum Adhoc-Auswerten eines         *
*               Grauwertverteilungshistogramms eines Bildbereichs.      *
*																								*
*   Parameter:  * histogramfield :	Pointer auf Histogrammwerte         *
*               dynamik          :  Anzahl der Graustufen               *
*					 interval         :  Enum zur Intervalwahl               *
*               hotpixelcount    :  Anzahl auszuschliessender Hotpixel  *
*               deadpixelcount   :  Anzahl auszuschliessender Deadpixel *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*               * lowestposition  : Kleinster besetzter Grauwert        *
*               * highestposition : Groesster besetzter Grauwert        *
*               * lowposition     : Kleinster Grauwert des Intervals    *
*               * highposition    : Groesster Grauwert des Intervals    *
*               * maximumposition : Haeufgster besetzter Grauwert       *
*               * maximalvalue    : Anzahl der Pixel mit haeuf. Grauw.  *
*               * maximumpart     : Anteil des haeuf. Grauwert am Bild  *
*               * entropy         : Grauwert-Entropie des Bildes        *
*                                                                       *
*   Version:    1.3 (19.08.2002)                                        *
*   Written by: Georg Dittie, 1994-2002                                 *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_EvaluateHistogram
(
  long * histogramfield,
  long dynamik,
  enum CCDLIB_HistogramrangeEnum interval,
  long hotpixelnum,
  long deadpixelnum,
  long * lowestposition,
  long * highestposition,
  long * lowposition,
  long * highposition,
  long * maximumposition,
  long * maximalvalue,
  double * maximumpart,
  double * entropy
)
{
	long i, histogramsum, histolowsum, histohighsum, valcount;

	if( histogramfield == NULL )
		return( CCDLIB_NoPointerError );

   valcount      = 0;

	*lowestposition = 0;
	histogramsum  = 0;
	for( i=0; i<dynamik; i++ )
	{
		histogramsum += histogramfield[ i ];
		if( histogramsum > deadpixelnum )
		{
			*lowestposition = i;
			break;
		}
	}
	*highestposition = dynamik-1;
	histogramsum     = 0;
	for( i=dynamik-1; i>=0; i-- )
	{
		histogramsum += histogramfield[ i ];
		if( histogramsum > hotpixelnum )
		{
			*highestposition = i;
			break;
		}
   }
   if( *highestposition-*lowestposition <= 1 )
   {
  		*lowposition = *lowestposition;
  		*highposition = *highestposition;
  		*maximumposition = *lowestposition;
  		*maximalvalue = dynamik;
  		*maximumpart = 1.;
  		*entropy = 0.;
      return( CCDLIB_Successful );
	}

   *maximumposition = *lowestposition;
   *maximalvalue = 0;
   histogramsum = 0;
	for( i=*lowestposition; i<*highestposition; i++ )
	{
      histogramsum += histogramfield[ i ];
		if( histogramfield[ i ] > *maximalvalue )
		{
			*maximalvalue = histogramfield[ i ];
         *maximumposition = i;
		}
      if( histogramfield[ i ] != 0 )
         valcount++;
   }
   *maximumpart = (double)(*maximalvalue)/(double)histogramsum;
	*entropy     = 1./log( 2. ) * log( (double)labs( valcount + 1 ) );

	switch( interval )
	{
	case CCDLIB_HistoExaktConst:
		*highposition = *highestposition;
		*lowposition  = *lowestposition;
      return( CCDLIB_Successful );
	case CCDLIB_Histo99_8Const:
		histohighsum = (long)floor(histogramsum*0.001+0.5);
		histolowsum  = (long)floor(histogramsum*0.999+0.5);
		break;
	case CCDLIB_Histo99_6Const:
		histohighsum = (long)floor(histogramsum*0.002+0.5);
		histolowsum  = (long)floor(histogramsum*0.998+0.5);
		break;
	case CCDLIB_Histo99Const:
		histohighsum = (long)floor(histogramsum*0.005+0.5);
		histolowsum  = (long)floor(histogramsum*0.995+0.5);
		break;
	case CCDLIB_Histo98Const:
		histohighsum = (long)floor(histogramsum*0.01+0.5);
		histolowsum  = (long)floor(histogramsum*0.99+0.5);
		break;
	case CCDLIB_Histo96Const:
		histohighsum = (long)floor(histogramsum*0.02+0.5);
		histolowsum  = (long)floor(histogramsum*0.98+0.5);
		break;
	case CCDLIB_Histo90Const:
		histohighsum = (long)floor(histogramsum*0.05+0.5);
		histolowsum  = (long)floor(histogramsum*0.95+0.5);
		break;
	case CCDLIB_Histo80Const:
		histohighsum = (long)floor(histogramsum*0.1+0.5);
		histolowsum  = (long)floor(histogramsum*0.9+0.5);
		break;
	default:
		return( CCDLIB_WrongParameterError );
	}

	*highposition = *highestposition;
   *lowposition  = *lowestposition;
	histogramsum  = 0;
	for( i=*highestposition; i>=*lowestposition; i-- )
	{
		histogramsum += histogramfield[ i ];
		if( histogramsum < histohighsum )
		{
			*highposition = i;
		}
		if( histogramsum < histolowsum )
		{
			*lowposition = i;
		}
	}

	return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_GetLocalStatistics: Routine zum Ermitteln von Mittelwert       *
*               und Standardabweichung in einer Bildregion                *
*                                                                         *
*   Parameter:  *picturerect : Koordinate der oberen linken Ecke des      *
*                              Arbeitsbereiches im Bild                   *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*               * mean  :           arithmetischer Mittelwert             *
*               * sigma :           N-1 Standardabweichung                *
*                                                                         *
*   Version:    1.0 (05.06.1994)                                          *
*   Written by: Georg Dittie, 1994                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetLocalStatistics
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	double * mean,
   double * sigma
)
{
  long x, ys, xstop, ystop;

  long meansum, meanvalue, variance, temporary, pixelnum;
  double variancesum;

  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL )
  {
	 *mean = 0.;
	 *sigma = 0.;
	 return( CCDLIB_NoPicturePointerError );
  }

  xstop = (long)(picturerect->xpos + picturerect->xsize);
  ystop = (long)(picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

  pixelnum = (long)picturerect->xsize * (long)picturerect->ysize;

  *mean = 0;
  for(ys = (long)(picturerect->ypos * pictureparam->linelength);
		ys < ystop;
		ys += pictureparam->linelength)
  {
    meansum = 0;
	 for(x = (long)(picturerect->xpos);
		  x < xstop;
		  x++ )
    {
		meansum += (long)picture[ ys+x ];
    }
    *mean += (double)meansum;
  }
  if( *mean == 0. )
  {
    *mean = 0.;
    *sigma = 0.;
	 return( CCDLIB_Successful );
  }

  *mean /= (double)pixelnum;

  meanvalue = (long)floor( *mean + 0.5 );
  variancesum = 0.;
  for(ys = (long)(picturerect->ypos * pictureparam->linelength);
		ys < ystop;
		ys += pictureparam->linelength)
  {
	 variance = 0;
	 for(x = (long)(picturerect->xpos);
		  x < xstop;
		  x++ )
    {
		temporary = (long)picture[ ys+x ] - meanvalue;
      variance += ( temporary * temporary );
    }
	 variancesum += (double)variance;
  }

  if( variancesum == 0. )
	 *sigma = 0.;
  else
	 *sigma = sqrt( variancesum / (double)( pixelnum-1 ) );

  return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcHistogramLUT: Routine zum Berechnen einer Look Up Table    *
*                            zur Kontrastkorrektur. Nicht gebrauchte      *
*                            Pointer koennen auf NULL gesetzt werden.     *
*                                                                         *
*   Parameter:  *histogramLUT  : berechnete LUT                           *
*               dynamik        : Anzahl der Graustufen                    *
*               kontart        : Methode der LUT-Berechnung               *
*               gamma          : Gammaeinstellung                         *
*               wendepunkt     : Wendepunkt bei S-Kurve                   *
*               highgamma      : Gamma oberer Abschnitt                   *
*               lowgamma       : Gamma unterer Abschnitt                  *
*               equalize       : Equalizing mit ausfuehren                *
*               equfactor      : Toplevel fr Equalizing                  *
*               interval       : Interval fuer Equalizing                 *
*               *histogramfield: Pointer auf Histogrammwerte              *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.1 (Allerheiligen 2000)                32 Bit            *
*   Written by: Georg Dittie, 1997 - 2000                                 *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcHistogramLUT
(
	short * histogramLUT,
	long dynamik,
	enum CCDLIB_HistogramModifierEnum modifier,
	double gamma,
	long   wendepunkt,
	double highgamma,
	double lowgamma,
   long boostlevel,
	BOOL equalize,
   double equfactor,
	enum CCDLIB_HistogramrangeEnum interval,
   long backgroundlevel,
	long * histogramfield )
{
	double xnormfaktor, ynormfaktor,
			 xhighnormfak, yhighnormfak, xlownormfak, ylownormfak, ywendepkt;
	long i, histodif, histosum, aktsum;
	long lowestposition, highestposition,
		  lowposition, highposition,
		  maximumposition, maximalvalue;
	double maximumpart, entropy, histval;

	if( histogramLUT == NULL ) return( CCDLIB_NoPointerError );

	if( equalize == TRUE ||	modifier == CCDLIB_HistogramConst )
	{
		if( histogramfield == NULL ) return( CCDLIB_NoPointerError );

		CCDLIB_EvaluateHistogram(
			histogramfield, dynamik, interval, 0, 0,
			&lowestposition, &highestposition,
			&lowposition, &highposition,
			&maximumposition, &maximalvalue,
			&maximumpart, &entropy );
		if( wendepunkt <= lowposition )  wendepunkt = lowposition+1;
		if( wendepunkt >= highposition ) wendepunkt = highposition-1;

      highestposition = (long)((double)highestposition/equfactor);
      if( highestposition >= dynamik )
         highestposition = dynamik-1;
      highposition = (long)((double)highposition/equfactor);
      if( highposition >= dynamik )
         highposition = dynamik-1;
	}
	else
	{
		lowestposition = 0; highestposition = dynamik-1;
		lowposition = 0;    highposition = dynamik-1;
	}

   if( backgroundlevel >= highposition )
   	backgroundlevel = highposition-1;
   if( backgroundlevel > lowposition )
   	lowposition = backgroundlevel;
	histodif = highposition-lowposition;

	switch( modifier )
	{
	case CCDLIB_LinearConst:
		ynormfaktor = (double)(dynamik-1)/(double)(histodif);
		break;
	case CCDLIB_S_SchlagConst:
		ywendepkt = (long)floor( (double)((wendepunkt-lowposition)*(dynamik-1))/
										 (double)histodif + 0.5);
		xlownormfak  = (double)(wendepunkt-lowposition);
		ylownormfak  = (double)ywendepkt;
		xhighnormfak = (double)(highposition-wendepunkt);
		yhighnormfak = (double)(dynamik-1-ywendepkt);
		break;
	case CCDLIB_GammaKorrekturConst:
		xnormfaktor = (double)histodif;
		ynormfaktor = (double)dynamik-1.;
		break;
	case CCDLIB_LogarithmusConst:
		xnormfaktor = (double)(histodif-1)/(M_E*M_E-1.);
		ynormfaktor = (double)(dynamik-1)/2.;
		break;
	case CCDLIB_ExponentConst:
		xnormfaktor = (double)(histodif-1)/2.;
		ynormfaktor = (double)(dynamik-1)/(M_E*M_E-1.);
		break;
	case CCDLIB_HistogramConst:
		histosum = 0;
		for( i=lowposition; i<highposition; i++ )
			histosum += histogramfield[i];
		ynormfaktor = (double)(dynamik-1)/(double)(histosum);
		break;
	case CCDLIB_HistoBoostConst:
   	if( boostlevel <= lowposition )
      	boostlevel = lowposition+1;
   	if( boostlevel >= highposition )
      	boostlevel = highposition-1;
		ywendepkt = (long)floor( (double)((boostlevel-lowposition)*(dynamik-1))/
										 (double)histodif + 0.5);
		for( i=boostlevel; i<highposition; i++ )
			histosum += histogramfield[i];
		yhighnormfak = (double)(dynamik-1-ywendepkt)/(double)(histosum);
      ylownormfak  = (double)ywendepkt/(double)(boostlevel-lowposition);
		break;
	default:
		return( CCDLIB_WrongParameterError );
	}

	for( i=0; i<lowposition; i++ )
		histogramLUT[ i ] = 0;
	for( i=lowposition, aktsum=0; i<highposition; i++ )
	{
		switch( modifier )
		{
		case CCDLIB_LinearConst:
			histogramLUT[ i ] = (short)floor
				(ynormfaktor * (double)(i-lowposition) + 0.5);
			break;
		case CCDLIB_S_SchlagConst :
			if( i<wendepunkt )
				histogramLUT[ i ] = (short)floor
					(ylownormfak * pow( (double)(i-lowposition)/xlownormfak, lowgamma ) + 0.5);
			else
				histogramLUT[ i ] = ywendepkt + (short)floor
					(yhighnormfak * pow( (double)(i-wendepunkt)/xhighnormfak, highgamma ) + 0.5);
			break;
		case CCDLIB_GammaKorrekturConst :
			histogramLUT[ i ] = (short)floor
				(ynormfaktor * pow( (double)(i-lowposition)/xnormfaktor, gamma ) + 0.5);
			break;
		case CCDLIB_LogarithmusConst :
			histogramLUT[ i ] = (short)floor
				(ynormfaktor * log( (double)(i-lowposition)/xnormfaktor + 1. ) + 0.5);
			break;
		case CCDLIB_ExponentConst :
			histogramLUT[ i ] = (short)floor
				(ynormfaktor * ( exp( (double)(i-lowposition)/xnormfaktor ) - 1.) + 0.5);
			break;
		case CCDLIB_HistogramConst :
			histval = ynormfaktor * (double)aktsum;
			aktsum += histogramfield[i];
         if( histval > 32767. ) histval = 32767.;
         histogramLUT[ i ] = (short)floor( histval+0.5 );
			break;
		case CCDLIB_HistoBoostConst :
         if( i < boostlevel )
         {
	        	histval = ylownormfak * (double)(i-lowposition);
         }
         else
         {
            histval = (yhighnormfak * (double)aktsum ) + ywendepkt;
				aktsum += histogramfield[i];
         }
         if( histval > 32767. ) histval = 32767.;
         histogramLUT[ i ] = (short)floor( histval+0.5 );
			break;
		default:
			return( CCDLIB_WrongParameterError );
		}
	}
	for( i=highposition; i<dynamik; i++ )
		histogramLUT[ i ] = (short)(dynamik-1);

	return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_ModifyHistogram: Routine zum Veraendern der Grauwertvertei-    *
*               lung eines Bildausschnitts. Die Routine arbeitet mit  	  *
*               einer vorher zu fuellenden HistogramLUT.                  *
*                                                                         *
*   Parameter:  *picturerect :  Koordinate der oberen linken Ecke des     *
*                               Arbeitsbereiches im Bild                  *
*               *pictureparam:  Systemparameter des zu bearbeitenden      *
*                               Bildes                                    *
*               *sourcepic   :  32-Bit-Pointer auf die Bildquelle         *
*               *destpic     :  32-Bit-Pointer auf das Bildziel           *
*               *histogramLUT:  LUT zur Grauwerttransformation            *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesC: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerErrorConst:   Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.2 (21.09.1997)                              32 Bit      *
*   Written by: Georg Dittie, 1994-1997                                   *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_ModifyHistogram
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	short * histogramLUT )
{
	long x, ys, xstop, ystop, ystep;

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( sourcepic == NULL || destpic == NULL)
		return( CCDLIB_NoPicturePointerError );

	if( histogramLUT == NULL )
		return( CCDLIB_NoPointerError );

	xstop = (long)(picturerect->xpos + picturerect->xsize);
	ystop = (long)(picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

	if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for(ys = (long)(picturerect->ypos * pictureparam->linelength);
			 ys < ystop;
			 ys += pictureparam->linelength)
		{
			for(x = (long)(picturerect->xpos);
				 x < xstop;
				 x++ )
			{
				destpic[ x+ys ] = histogramLUT[ sourcepic[ x+ys ] ];
			}
		}
	}
	else
	{
		xstop *= 3;
		ystop *= 3;
		ystep = pictureparam->linelength*3;
		for( ys = picturerect->ypos * ystep;
			  ys < ystop;
			  ys += ystep )
		{
			for(x = (long)(picturerect->xpos)*3;
				 x < xstop;
				 x += 3 )
			{
				destpic[ x+ys ]   = histogramLUT[ sourcepic[ x+ys ] ];
				destpic[ x+ys+1 ] = histogramLUT[ sourcepic[ x+ys+1 ] ];
				destpic[ x+ys+2 ] = histogramLUT[ sourcepic[ x+ys+2 ] ];
			}
		}
	}

	return( CCDLIB_Successful );
}

enum CCDLIB_ErrorEnum CCDLIB_QuickEqualize
(
   short *picbuffer,
   CCDLIB_PictureparamStruct *pictureparam
)
{
   long i, buflength, min, max, val, dm1, dif;
   double fak;
   short *histogramLUT;

	if( picbuffer == NULL )
		return( CCDLIB_NoPicturePointerError );

   max = 0;
   min = pictureparam->dynamik-1;
   dm1 = pictureparam->dynamik-1;

   histogramLUT = (short *)malloc( 32768L*sizeof(short) );
   if( histogramLUT == NULL )
      return( CCDLIB_OutOfMemoryError );

   if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
   {
      buflength = pictureparam->linelength*pictureparam->columnheight;
      for( i=0; i<buflength; i++ )
      {
         if( (long)picbuffer[i] > max ) max = (long)picbuffer[i];
         if( (long)picbuffer[i] < min ) min = (long)picbuffer[i];
      }

      dif = max-min;
      fak = (double)dm1/(double)dif;
      for( i=0; i<min; i++ )
         histogramLUT[i] = 0;
      for( i=min; i<max; i++ )
         histogramLUT[i] = (short)((double)(i-min)*fak+0.5);
      for( i=max; i<pictureparam->dynamik; i++ )
         histogramLUT[i] = (short)dm1;

      for( i=0; i<buflength; i++ )
         picbuffer[i] = histogramLUT[ picbuffer[i] ];
   }
   else
   {
      buflength = pictureparam->linelength*pictureparam->columnheight*3;
      for( i=0; i<buflength; i+=3 )
      {
         val = (long)(  (double)picbuffer[i]*0.114
                      + (double)picbuffer[i+1]*0.587
                      + (double)picbuffer[i+2]*0.299 );
         if( val > max ) max = val;
         if( val < min ) min = val;
      }
      dif = max-min;
      fak = (double)dm1/(double)dif;
      for( i=0; i<min; i++ )
         histogramLUT[i] = 0;
      for( i=min; i<max; i++ )
         histogramLUT[i] = (short)((double)(i-min)*fak+0.5);
      for( i=max; i<pictureparam->dynamik; i++ )
         histogramLUT[i] = (short)dm1;

      for( i=0; i<buflength; i++ )
         picbuffer[i] = histogramLUT[ picbuffer[i] ];
   }

   free( histogramLUT );
	return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_SubtractBackground: Routine zum Abzug eines erhoehten Unter-   *
*               grunds. Dabei wird davon ausgegangen, das der Unter-      *
*               flaechenmaessig das Bild beherrscht.                      *
*                                                                         *
*   Parameter:  *picturerect :  Koordinate der oberen linken Ecke des     *
*                               Arbeitsbereiches im Bild                  *
*               *pictureparam:  Systemparameter des zu bearbeitenden      *
*                               Bildes                                    *
*               *sourcepic   :  32-Bit-Pointer auf die Bildquelle         *
*               *destpic     :  32-Bit-Pointer auf das Bildziel           *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Das Resultatbild steht an derselben Stelle wie das Ausgangsbild.      *
*   ACHTUNG: Das Ausgangsbild wird dabei ueberschrieben !                 *
*                                                                         *
*   Version:    1.1 (21.09.1997)                                          *
*   Written by: Georg Dittie, 1997                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_SubtractBackground
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic
)
{
  long x, ys, i, xstop, ystop;

  long histomaximum, histomaxpos;
  double histomaxpart, histoentropy, yfak;
  long histolowestpos, histohighestpos;

  enum CCDLIB_ErrorEnum result;

  long * histogramfield;
  short * translatetab;

  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

	if( sourcepic == NULL || destpic == NULL)
		return( CCDLIB_NoPicturePointerError );

  histogramfield = (long *)malloc( pictureparam->dynamik * sizeof(long) );
  translatetab   = (short *)malloc( pictureparam->dynamik * sizeof(short) );

  if( histogramfield == NULL || translatetab == NULL )
  {
	 if( histogramfield != NULL ) free( histogramfield );
	 if( translatetab   != NULL ) free( translatetab );

	 return( CCDLIB_OutOfMemoryError );
  }

  result = CCDLIB_RetrieveHistogram(
	 picturerect,
	 pictureparam,
	 sourcepic,
	 histogramfield );

  if( result != CCDLIB_Successful )
  {
	 free( histogramfield );
	 free( translatetab );
	 return( result );
  }

  result = CCDLIB_GetMaximumInHistogram(
	 histogramfield,
	 pictureparam->dynamik,
	 &histolowestpos,
	 &histohighestpos,
	 &histomaxpos,
	 &histomaximum,
	 &histomaxpart,
	 &histoentropy );

  if( result != CCDLIB_Successful )
  {
	 free( histogramfield );
	 free( translatetab );
	 return( result );
  }

  yfak = (double)(pictureparam->dynamik-1)/(double)(pictureparam->dynamik-histomaxpos);

  for( i=0; i<histomaxpos; i++ )
  {
	 translatetab[ i ] = 0;
  }
  for( i=histomaxpos; i<pictureparam->dynamik; i++ )
  {
	 translatetab[ i ] = (short)
		floor( (double)(i - histomaxpos) * yfak );
  }

  xstop = (long)(picturerect->xpos + picturerect->xsize);
  ystop = (long)(picturerect->ypos + picturerect->ysize) * pictureparam->linelength;

  for(ys = (long)(picturerect->ypos * pictureparam->linelength);
		ys < ystop;
		ys += pictureparam->linelength)
  {

	 for(x = (long)(picturerect->xpos);
		  x < xstop;
		  x++ )
	 {
		destpic[ x+ys ] = translatetab[ sourcepic[ x+ys ] ];
	 }
  }

  free( histogramfield );
  free( translatetab );

  return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_Equalize: Routine zum schnellen Equalizing des Kontrastes.     *
*                                                                         *
*   Parameter:  *pictureparam:  Systemparameter des zu bearbeitenden      *
*                               Bildes                                    *
*               *sourcepic   :  32-Bit-Pointer auf die Bildquelle         *
*               *destpic     :  32-Bit-Pointer auf das Bildziel           *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Das Resultatbild steht an derselben Stelle wie das Ausgangsbild.      *
*   ACHTUNG: Das Ausgangsbild wird dabei ueberschrieben !                 *
*                                                                         *
*   Version:    1.1 (vom schnen Juli 2003)                               *
*   Written by: Georg Dittie, 2001, 2003                                  *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_Equalize
(
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
   enum CCDLIB_GainEnum gaintype,
   double gamma,
   enum CCDLIB_EqualizeEnum subtracttype
)
{
   long i, ystop, pdm1;

   long histomaximum, histomaxpos, histodif, histoback;
   double histomaxpart, histoentropy, xfak, yfak;
   long histolowestpos, histohighestpos;

   enum CCDLIB_ErrorEnum result;
   CCDLIB_RectangleStruct picturerect;

   long * histogramfield;
   short * translatetab;

	if( sourcepic == NULL || destpic == NULL)
		return( CCDLIB_NoPicturePointerError );

   histogramfield = (long *)malloc( pictureparam->dynamik * sizeof(long) );
   translatetab   = (short *)malloc( pictureparam->dynamik * sizeof(short) );

   if( histogramfield == NULL || translatetab == NULL )
   {
	   if( histogramfield != NULL ) free( histogramfield );
	   if( translatetab   != NULL ) free( translatetab );

	   return( CCDLIB_OutOfMemoryError );
   }
   CCDLIB_InitRectangle( &picturerect,
      CCDLIB_MINIMALRIM, CCDLIB_MINIMALRIM,
      pictureparam->linelength-CCDLIB_MINIMALRIM*2,
      pictureparam->columnheight-CCDLIB_MINIMALRIM*2 );
   pdm1 = pictureparam->dynamik-1;

   ystop = pictureparam->linelength*pictureparam->columnheight;
   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
      ystop *= 3;

   result = CCDLIB_RetrieveHistogram(
	   &picturerect,
	   pictureparam,
	   sourcepic,
	   histogramfield );

   if( result != CCDLIB_Successful )
   {
	   free( histogramfield );
	   free( translatetab );
	   return( result );
   }

   result = CCDLIB_GetMaximumInHistogram(
	   histogramfield,
	   pictureparam->dynamik,
	   &histolowestpos,
	   &histohighestpos,
	   &histomaxpos,
	   &histomaximum,
	   &histomaxpart,
	   &histoentropy );

   if( result != CCDLIB_Successful )
   {
	   free( histogramfield );
	   free( translatetab );
	   return( result );
   }

   if( subtracttype == CCDLIB_NoChangeConst )
   {
   	histoback = histolowestpos;
      histodif  = histohighestpos-histolowestpos;
   }
   else
   {
   	histoback = histomaxpos;
      histodif  = histohighestpos-histomaxpos;
   }
   if( histodif <= 1 )
   {
      if( sourcepic != destpic )
      	memcpy( destpic, sourcepic, ystop*sizeof(short) );
   	return( CCDLIB_Successful );
   }

   switch( gaintype )
   {
	case CCDLIB_GainLinearConst:
     	yfak = (double)(pdm1)/(double)(histodif);
      break;
	case CCDLIB_GainLogConst:
		xfak = (double)(histodif-1)/(M_E*M_E-1.);
		yfak = pdm1/2.;
      break;
	case CCDLIB_Gain2LogConst:
		xfak = (double)(histodif-1)/(M_E*M_E*M_E*M_E-1.);
		yfak = pdm1/4.;
      break;
	case CCDLIB_GainGammaConst:
		xfak = (double)(histodif);
		yfak = pdm1;
     	break;
	}
   for( i=0; i<histoback; i++ )
      translatetab[ i ] = 0;
   for( i=histoback; i<histohighestpos; i++ )
     	switch( gaintype )
      {
		case CCDLIB_GainLinearConst:
	      translatetab[ i ] = (short)
		      floor( (double)(i - histoback) * yfak );
         break;
		case CCDLIB_GainLogConst:
		case CCDLIB_Gain2LogConst:
			translatetab[ i ] = (short)floor
				(yfak * log( (double)(i-histoback)/xfak + 1. ) + 0.5);
         break;
		case CCDLIB_GainGammaConst:
			translatetab[ i ] = (short)floor
				(yfak * pow( (double)(i-histoback)/xfak, gamma ) + 0.5);
       	break;
		}
   for( i=histohighestpos; i<pictureparam->dynamik; i++ )
       translatetab[ i ] = (short)pdm1;

   for( i = 0; i < ystop; i++ )
		destpic[ i ] = translatetab[ sourcepic[ i ] ];

   free( histogramfield );
   free( translatetab );

   return( CCDLIB_Successful );
}



/*** calculation of artificial backgrounds ***/

/************************************************************************
*                                                                       *
*	 CCDLIB_CalcArtificialBackground : Routine zum Erzeugen eines        *
*               kuenstlichen Untergrundes mit 2D-Ausgleichspolynomen    *
*																								*
*   Parameter:  * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf die Bilddaten       *
*               * background  :	32-Bit-Pointer auf den Untergrund      *
*               polygrad      :	Grad des 2D-Polynoms (0 - 4)           *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (31. 8. 2000)                                       *
*   Written by: Georg Dittie, 2000                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcArtificialBackground
(
   short * background,
	short * picture,
	CCDLIB_PictureparamStruct * pictureparam,
	long polygrad
)
{
   double xval[64], yval[64], zval[64], koeff[9], erg,
          maximumpart, entropy, xd, yd;
   long lowestposition, highestposition, lowposition, highposition,
        maximumposition, maximalvalue, i, r, g, b, x, y, buflength;
	long *histofield;
   short *tempbuf;
   CCDLIB_RectangleStruct evalrect;
   CCDLIB_PictureparamStruct tempparam;


	if( picture == NULL || background == NULL )
		return( CCDLIB_NoPicturePointerError );

   if( polygrad < 0 || polygrad > 4 )
      return( CCDLIB_WrongParameterError );

   buflength = pictureparam->linelength*pictureparam->columnheight;
   CCDLIB_InitRectangle(
      &evalrect,
      CCDLIB_MINIMALRIM, CCDLIB_MINIMALRIM,
      pictureparam->linelength-CCDLIB_MINIMALRIM*2,
      pictureparam->columnheight-CCDLIB_MINIMALRIM*2 );
   tempparam = *pictureparam;
   tempparam.colorchannel = CCDLIB_BlackWhiteColor;

   if( polygrad == 0 )
   {
   	histofield = (long*)malloc( pictureparam->dynamik*sizeof(long) );
	   if( histofield == NULL )
		   return( CCDLIB_OutOfMemoryError );
	   memset( histofield, 0, pictureparam->dynamik*sizeof(long) );

      if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
      {
         CCDLIB_RetrieveHistogram(
            &evalrect, pictureparam, picture, histofield );

         CCDLIB_EvaluateHistogram(
            histofield,
            pictureparam->dynamik,
            CCDLIB_Histo99_8Const,
            0,
            0,
            &lowestposition,
            &highestposition,
            &lowposition,
            &highposition,
            &maximumposition,
            &maximalvalue,
            &maximumpart,
            &entropy );

         for( i=0; i<buflength; i++ )
            background[i] = (short)maximumposition;
      }
      else
      {
         tempbuf = (short*)malloc( buflength*sizeof(short) );
         if( tempbuf == NULL )
         {
            free( histofield );
            return( CCDLIB_OutOfMemoryError );
         }

         for( i=0, b=0; i<buflength; i++, b+=3 )
            tempbuf[i] = picture[b];

         CCDLIB_RetrieveHistogram(
            &evalrect, &tempparam, tempbuf, histofield );

         CCDLIB_EvaluateHistogram(
            histofield,
            pictureparam->dynamik,
            CCDLIB_Histo99_8Const,
            0,
            0,
            &lowestposition,
            &highestposition,
            &lowposition,
            &highposition,
            &maximumposition,
            &maximalvalue,
            &maximumpart,
            &entropy );

         for( i=0, b=0; i<buflength; i++, b+=3 )
            background[b] = (short)maximumposition;

         for( i=0, g=1; i<buflength; i++, g+=3 )
            tempbuf[i] = picture[g];

         CCDLIB_RetrieveHistogram(
            &evalrect, &tempparam, tempbuf, histofield );

         CCDLIB_EvaluateHistogram(
            histofield,
            pictureparam->dynamik,
            CCDLIB_Histo99_8Const,
            0,
            0,
            &lowestposition,
            &highestposition,
            &lowposition,
            &highposition,
            &maximumposition,
            &maximalvalue,
            &maximumpart,
            &entropy );

         for( i=0, g=1; i<buflength; i++, g+=3 )
            background[g] = (short)maximumposition;

         for( i=0, r=2; i<buflength; i++, r+=3 )
            tempbuf[i] = picture[r];

         CCDLIB_RetrieveHistogram(
            &evalrect, &tempparam, tempbuf, histofield );

         CCDLIB_EvaluateHistogram(
            histofield,
            pictureparam->dynamik,
            CCDLIB_Histo99_8Const,
            0,
            0,
            &lowestposition,
            &highestposition,
            &lowposition,
            &highposition,
            &maximumposition,
            &maximalvalue,
            &maximumpart,
            &entropy );

         for( i=0, r=2; i<buflength; i++, r+=3 )
            background[r] = (short)maximumposition;

         free( tempbuf );
      }
   }
   else
   {
      if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
      {
         CCDLIB_GetHistoMatrix(
	         &evalrect, pictureparam, picture, xval, yval, zval );

         CCDLIB_2DRegression( polygrad, 64, xval, yval, zval, koeff);

         i = 0;
         for( y=0, yd=0.; y<pictureparam->columnheight; y++, yd+=0.001 )
         {
            for( x=0, xd=0.; x<pictureparam->linelength; x++, xd+=0.001 )
            {
               erg = CCDLIB_Eval2DPolynom( polygrad, xd, yd, koeff )*1000.;
               if( erg > 32767. ) erg = 32767.;
               if( erg < 0. ) erg = 0.;
               background[i] = (short)floor( erg+0.5 );

               i++;
            }
         }
      }
      else
      {
         tempbuf = (short*)malloc( buflength*sizeof(short) );
         if( tempbuf == NULL )
         {
            free( histofield );
            return( CCDLIB_OutOfMemoryError );
         }

         for( i=0, b=0; i<buflength; i++, b+=3 )
            tempbuf[i] = picture[b];

         CCDLIB_GetHistoMatrix(
	         &evalrect, &tempparam, tempbuf, xval, yval, zval );

         CCDLIB_2DRegression( polygrad, 64, xval, yval, zval, koeff);

         b = 0;
         for( y=0, yd=0.; y<pictureparam->columnheight; y++, yd+=0.001 )
         {
            for( x=0, xd=0.; x<pictureparam->linelength; x++, xd+=0.001 )
            {
               erg = CCDLIB_Eval2DPolynom( polygrad, xd, yd, koeff )*1000.;
               if( erg > 32767. ) erg = 32767.;
               if( erg < 0. ) erg = 0.;
               background[b] = (short)floor( erg+0.5 );
               b += 3;
            }
         }

         for( i=0, g=1; i<buflength; i++, g+=3 )
            tempbuf[i] = picture[g];

         CCDLIB_GetHistoMatrix(
	         &evalrect, &tempparam, tempbuf, xval, yval, zval );

         CCDLIB_2DRegression( polygrad, 64, xval, yval, zval, koeff);

         g = 1;
         for( y=0, yd=0.; y<pictureparam->columnheight; y++, yd+=0.001 )
         {
            for( x=0, xd=0.; x<pictureparam->linelength; x++, xd+=0.001 )
            {
               erg = CCDLIB_Eval2DPolynom( polygrad, xd, yd, koeff )*1000.;
               if( erg > 32767. ) erg = 32767.;
               if( erg < 0. ) erg = 0.;
               background[g] = (short)floor( erg+0.5 );
               g += 3;
            }
         }

         for( i=0, r=2; i<buflength; i++, r+=3 )
            tempbuf[i] = picture[r];

         CCDLIB_GetHistoMatrix(
	         &evalrect, &tempparam, tempbuf, xval, yval, zval );

         CCDLIB_2DRegression( polygrad, 64, xval, yval, zval, koeff);

         r = 2;
         for( y=0, yd=0.; y<pictureparam->columnheight; y++, yd+=0.001 )
         {
            for( x=0, xd=0.; x<pictureparam->linelength; x++, xd+=0.001 )
            {
               erg = CCDLIB_Eval2DPolynom( polygrad, xd, yd, koeff )*1000.;
               if( erg > 32767. ) erg = 32767.;
               if( erg < 0. ) erg = 0.;
               background[r] = (short)floor( erg+0.5 );
               r += 3;
            }
         }

         free( tempbuf );
      }
   }
	return( CCDLIB_Successful );
}

/************************************************************************
*                                                                       *
*	 CCDLIB_GetHistoMatrix: Routine zum Ermitteln der Stuetzwertetripel  *
*               zum Berechnen eines kuenstlichen 2D-Ausgleichspolynoms  *
*																								*
*   Parameter:  * picturerect :  Koordinate der oberen linken Ecke des  *
*                                Arbeitsbereiches im Bild               *
*					 * pictureparam:  Systemparameter des zu bearbeitenden   *
*                                Bildes                                 *
*               * picture     :  32-Bit-Pointer auf die Bilddaten       *
*               * xvalues     :	Anzahl der Grauwerte                   *
*               * yvalues     :	Anzahl der Rotwerte                    *
*               * zvalues     :	Anzahl der Gruenwerte                  *
*                                                                       *
*  WICHTIG: Funktioniert nur mit Grauwertbildern bzw Farbauszgen       *
*           Das Ergebnis ist in X und Y um 100. geteilt, in Z um 1000.  *
*           Das geschieht aus numerischen Stabilitaetsgruenden          *
*                                                                       *
*	 Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung   *
*                                                                       *
*               CCDLIB_Successful:            Alles OK !					   *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen  *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse       *
*					 CCDLIB_NoPointerError:        Nicht initialisierte Ptr. *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen  *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher         *
*                                                                       *
*   Version:    1.0 (31. 8. 2000)                                       *
*   Written by: Georg Dittie, 2000                                      *
*                                                                       *
************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_GetHistoMatrix
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	double * xvalues,
	double * yvalues,
	double * zvalues
)
{
	long x, y, n;
   double xstart, xstep, ystart, ystep, maximumpart, entropy;
	long *histofield;
   long lowestposition, highestposition, lowposition, highposition,
        maximumposition, maximalvalue;
   CCDLIB_RectangleStruct evalrect;

	if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
		return( CCDLIB_WrongRectangleValuesError );

	if( picture == NULL || xvalues == NULL || yvalues == NULL || zvalues == NULL )
		return( CCDLIB_NoPicturePointerError );

   if( pictureparam->colorchannel == CCDLIB_RGBChannelColor )
      return( CCDLIB_WrongParameterError );

	histofield = (long*)malloc( pictureparam->dynamik*sizeof(long) );
	if( histofield == NULL )
		return( CCDLIB_OutOfMemoryError );
	memset( histofield, 0, pictureparam->dynamik*sizeof(long) );

   xstart = (double)picturerect->xpos;
   xstep  = (double)picturerect->xsize / 8.;
   ystart = (double)picturerect->ypos;
   ystep  = (double)picturerect->ysize / 8.;

   n = 0;
   for( y=0; y<8; y++ )
   {
      for( x=0; x<8; x++ )
      {
         xvalues[ n ] = (xstart+((double)x+0.5)*xstep)/1000.;
         yvalues[ n ] = (ystart+((double)y+0.5)*ystep)/1000.;

         CCDLIB_InitRectangle(
            &evalrect,
            picturerect->xpos+x*picturerect->xsize/8,
            picturerect->ypos+y*picturerect->ysize/8,
            picturerect->xsize/8, picturerect->ysize/8 );

         CCDLIB_RetrieveHistogram(
            &evalrect, pictureparam, picture, histofield );

         CCDLIB_EvaluateHistogram(
            histofield,
            pictureparam->dynamik,
            CCDLIB_Histo99_8Const,
            0,
            0,
            &lowestposition,
            &highestposition,
            &lowposition,
            &highposition,
            &maximumposition,
            &maximalvalue,
            &maximumpart,
            &entropy );

         zvalues[ n ] = (double)maximumposition/1000.;

         n++;
      }
   }

   free( histofield );
	return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_SubtractShiftDifference: Abziehen eines verschobenen Teils     *
*               von einem Originalbild: Sichtbarmachung von Strukturen    *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *sourcepic   : 32-Bit-Pointer auf die Quellbilddaten      *
*               *destpic     : 32-Bit-Pointer auf die Zielbilddaten       *
*               xcenter      : X-Koordinate des Drehpunktes               *
*               ycenter      : Y-Koordinate des Drehpunktes               *
*               shiftval     : Verschiebungsstrecke (Pixel / Grad)        *
*               subintensity : Anteil des zu subtrahierenden Bildes       *
*                                                                         *
*               CCDLIB_ShiftTypeEnum : Art der Shiftoperation             *
*                                                                         *
*               CCDLIB_RotConstAngle   -> Rotation um einen konst. Win.   *
*               CCDLIB_RotConstTangent -> Rot. mit konst. Tangente        *
*               CCDLIB_ShiftConstRad   -> Shift mit konst. Radius         *
*                                                                         *
*               ACHTUNG: sourcepic darf nicht destpic sein !              *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (02.10.1997, 1. CCD-Praktikum Kirchheim)              *
*   Written by: Georg Dittie, 1997                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_SubtractShiftDifference
(
	CCDLIB_PictureparamStruct * picparam,
	short * sourcepic,
	short * destpic,
	long xcenter,
	long ycenter,
	double shiftval,
	double subintensity,
	enum CCDLIB_ShiftTypeEnum shifttype
)
{
	long xs, xs1, xd, xf, yf, yss, yss1, yd, yds, xstop, ystop, ystep,
		  xcent, ycent, xlimit, ylimit, xneu, yneu,
		  greyval, blueval, greenval, redval,
		  xm0, xm1, xm2, xm3;
	double xdist, ydist, rdist, angle, platformval, tempval;

	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( sourcepic == destpic || subintensity < 0. || subintensity > 1. )
		return( CCDLIB_WrongParameterError );

	xstop = picparam->linelength;
	ystop = picparam->linelength*picparam->columnheight;
	ystep = picparam->linelength;
	xcent = xcenter<<8;
	ycent = ycenter<<8;
	xlimit = xstop<<8;
	ylimit = (picparam->columnheight)<<8;

	platformval = (double)picparam->dynamik/2.*subintensity;
	if( shifttype == CCDLIB_RotConstAngleConst )	shiftval *= M_PI/180.;

	if( picparam->colorchannel != CCDLIB_RGBChannelColor )
	{
		for( yd=0, yds=0; yds<ystop; yd++, yds+=ystep )
		{
			ydist = (double)(yd-ycenter);
			for( xd=0; xd<xstop; xd++ )
			{
				xdist = (double)(xd-xcenter);
				rdist = sqrt( xdist*xdist + ydist*ydist );

				if( rdist <= 0.5 )
				{
					destpic[ yds+xd ] = sourcepic[ yds+xd ];
					continue;
				}

				switch( shifttype )
				{
				case CCDLIB_RotConstAngleConst:
					rdist *= 256.;
					angle = atan2( ydist, xdist )+shiftval;
					break;
				case CCDLIB_RotConstTangentConst:
					if( rdist <= shiftval )
						angle = atan2( ydist, xdist )+M_PI_2;
					else
						angle = atan2( ydist, xdist )+M_PI_2*shiftval/rdist;
					rdist *= 256.;
					break;
				case CCDLIB_ShiftConstRadConst:
					rdist = (rdist+shiftval) * 256.;
					if( rdist < 0. ) rdist = 0.;
					angle = atan2( ydist, xdist );
					break;
				default:
					return( CCDLIB_WrongParameterError );
				}
				xneu  = (long)(rdist*cos(angle))+xcent;
				yneu  = (long)(rdist*sin(angle))+ycent;

				if( xneu >= 0 && xneu < xlimit &&
					 yneu >= 0 && yneu < ylimit )
				{
					yf = yneu & 0xFF;
					xf = xneu & 0xFF;

					xs  = xneu >> 8;
					xs1 = xs+1;

					yss  = (yneu >> 8) * ystep;
					yss1 = yss + ystep;

					/* Bilineare Interpolation aus George Wolberg "Digital Image Warping" */

					xm0 = (long)sourcepic[ yss +xs  ];
					xm1 = (long)sourcepic[ yss +xs1 ];
					xm2 = (long)sourcepic[ yss1+xs  ];
					xm3 = (long)sourcepic[ yss1+xs1 ];

					greyval = (
						( (xm0<<16) + (xm1-xm0)*(xf<<8) + (((xm2-xm0)<<8) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 16 );
				}
				else
					greyval = 0;

				 tempval = (double)sourcepic[ yds+xd ]
								- subintensity*(double)greyval + platformval;
				 if( tempval < 0. )
					destpic[ yds+xd ] = 0;
				 else if( tempval >= (double)picparam->dynamik )
					destpic[ yds+xd ] = (short)(picparam->dynamik-1);
				 else
					destpic[ yds+xd ] = (short)(tempval);
			}
		}
	}
	else
	{
		xstop *= 3;
		ystop *= 3;
		ystep *= 3;

		for( yd=0, yds=0; yds<ystop; yd++, yds+=ystep )
		{
			ydist = (double)(yd-ycenter);
			for( xd=0; xd<xstop; xd+=3 )
			{
				xdist = (double)(xd/3-xcenter);
				rdist = sqrt( xdist*xdist + ydist*ydist );

				if( rdist <= 0.5 )
				{
					destpic[ yds+xd   ] = sourcepic[ yds+xd   ];
					destpic[ yds+xd+1 ] = sourcepic[ yds+xd+1 ];
					destpic[ yds+xd+2 ] = sourcepic[ yds+xd+2 ];
					continue;
				}
				switch( shifttype )
				{
				case CCDLIB_RotConstAngleConst:
					rdist *= 256.;
					angle = atan2( ydist, xdist )+shiftval;
					break;
				case CCDLIB_RotConstTangentConst:
					if( rdist <= shiftval )
						angle = atan2( ydist, xdist )+M_PI_2;
					else
						angle = atan2( ydist, xdist )+M_PI_2*shiftval/rdist;
					rdist *= 256.;
					break;
				case CCDLIB_ShiftConstRadConst:
					rdist = (rdist+shiftval) * 256.;
					if( rdist < 0. ) rdist = 0.;
					angle = atan2( ydist, xdist );
					break;
				default:
					return( CCDLIB_WrongParameterError );
				}
				xneu  = (long)(rdist*cos(angle))+xcent;
				yneu  = (long)(rdist*sin(angle))+ycent;

				if( xneu >= 0 && xneu < xlimit &&
					 yneu >= 0 && yneu < ylimit )
				{
					yf = yneu & 0xFF;
					xf = xneu & 0xFF;

					xs  = (xneu >> 8) * 3;
					xs1 = xs + 3;

					yss  = (yneu >> 8) * ystep;
					yss1 = yss + ystep;

					/* Bilineare Interpolation aus George Wolberg "Digital Image Warping" */

					xm0 = (long)sourcepic[ yss +xs  ];
					xm1 = (long)sourcepic[ yss +xs1 ];
					xm2 = (long)sourcepic[ yss1+xs  ];
					xm3 = (long)sourcepic[ yss1+xs1 ];

					blueval = (
						( (xm0<<16) + (xm1-xm0)*(xf<<8) + (((xm2-xm0)<<8) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 16 );

					xm0 = (long)sourcepic[ yss +xs +1 ];
					xm1 = (long)sourcepic[ yss +xs1+1 ];
					xm2 = (long)sourcepic[ yss1+xs +1 ];
					xm3 = (long)sourcepic[ yss1+xs1+1 ];

					greenval = (
						( (xm0<<16) + (xm1-xm0)*(xf<<8) + (((xm2-xm0)<<8) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 16 );

					xm0 = (long)sourcepic[ yss +xs +2 ];
					xm1 = (long)sourcepic[ yss +xs1+2 ];
					xm2 = (long)sourcepic[ yss1+xs +2 ];
					xm3 = (long)sourcepic[ yss1+xs1+2 ];

					redval = (
						( (xm0<<16) + (xm1-xm0)*(xf<<8) + (((xm2-xm0)<<8) +
						(xm3-xm2-xm1+xm0)*xf)*yf ) >> 16 );
				}
				else
				{
					blueval  = 0;
					greenval = 0;
					redval   = 0;
				}
				tempval = (double)sourcepic[ yds+xd ]
					-subintensity*(double)blueval +platformval;
				if( tempval < 0. )
					destpic[ yds+xd ] = 0;
				else if( tempval >= (double)picparam->dynamik )
					destpic[ yds+xd ] = (short)(picparam->dynamik-1);
				else
					destpic[ yds+xd ] = (short)(tempval);

				tempval = (double)sourcepic[ yds+xd+1 ]
					-subintensity*(double)greenval +platformval;
				if( tempval < 0. )
					destpic[ yds+xd+1 ] = 0;
				else if( tempval >= (double)picparam->dynamik )
					destpic[ yds+xd+1 ] = (short)(picparam->dynamik-1);
				else
					destpic[ yds+xd+1 ] = (short)(tempval);

				tempval = (double)sourcepic[ yds+xd+2 ]
					-subintensity*(double)redval +platformval;
				if( tempval < 0. )
					destpic[ yds+xd+2 ] = 0;
				else if( tempval >= (double)picparam->dynamik )
					destpic[ yds+xd+2 ] = (short)(picparam->dynamik-1);
				else
					destpic[ yds+xd+2 ] = (short)(tempval);
			}
		}
	}
	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcXYGradientFromPic : Routine zum Berechnen des volstaen-    *
*               digen Gradientenvektors aus einem Grauwertbild            *
*                                                                         *
*   Parameter:  *picturerect : Koordinate der oberen linken Ecke des      *
*                              Arbeitsbereiches im Bild                   *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*                                                                         *
*               ACHTUNG: maskdiameter darf nur ungerade Werte annehmen!   *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*               *gradient    :  32-Bit-Pointer auf die Gradientenwerte    *
*               Ein Gradientenwert ist eine Union aus einem 32-Bit-Wert   *
*               und dem X- und Y-Anteil des Vektors als short             *
*                                                                         *
*   Version:    1.0 (05.06.1994)                                          *
*   Written by: Georg Dittie, 1994                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcXYGradientFromPic
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_GradientUnion * gradient
)
{
  long x, ys, ysm1, ysp1, xstart, xstop, ystart, ystop;
  
  short grad;
  
  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL || gradient == NULL )
	 return( CCDLIB_NoPicturePointerError );

  xstart = picturerect->xpos + 1;
  xstop  = picturerect->xpos + picturerect->xsize - 1;
  ystart = ( picturerect->ypos + 1 ) * pictureparam->linelength;
  ystop  = ( picturerect->ypos + picturerect->ysize - 1) * pictureparam->linelength;

  ysm1 = ystart - pictureparam->linelength;
  ysp1 = ystart + pictureparam->linelength;
  for(ys = ystart;
		ys < ystop;
		ys += pictureparam->linelength )
  {
	 for(x = xstart;
		  x < xstop;
		  x ++ )
    {
		grad = (short)( ( (long)picture[ ys+x+1 ] - (long)picture[ ys+x-1 ] ) >> 1 );

		gradient[ ys+x ].value.xgrad = (short)(
		  max( 0, min( pictureparam->dynamik-1, grad ) ) );

		grad = (short)( ( (long)picture[ ysp1+x ] - (long)picture[ ysm1+x ] ) >> 1 );
      
		gradient[ ys+x ].value.ygrad = (short)(
		  max( 0, min( pictureparam->dynamik-1, grad ) ) );
	 }

	 ysm1 += pictureparam->linelength;
	 ysp1 += pictureparam->linelength;
  }

  return( CCDLIB_Successful );
}


/**************************************************************************
*                                                                         *
*   CCDLIB_XYGradientFromPic : Routine zum Berechnen des volstaen-        *
*               digen Gradientenvektors aus einem Grauwertbild mit        *
*               Hilfe des rauschmindernden Sobel-Operators                *
*                                                                         *
*   Parameter:  *picturerect : Koordinate der oberen linken Ecke des      *
*                              Arbeitsbereiches im Bild                   *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*                                                                         *
*               ACHTUNG: maskdiameter darf nur ungerade Werte annehmen!   *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*               *gradient    :  32-Bit-Pointer auf die Gradientenwerte    *
*               Ein Gradientenwert ist eine Union aus einem 32-Bit-Wert   *
*               und dem X- und Y-Anteil des Vektors als short             *
*                                                                         *
*   Version:    1.0 (05.06.1994)                                          *
*   Written by: Georg Dittie, 1994                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_XYGradientFromPic
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  CCDLIB_GradientUnion  * gradient
)
{
  long x, xm1, xp1, ys, ysm1, ysp1,
       xstart, xstop, ystart, ystop;
  
  short grad;
  long  leftvalue, rightvalue, upvalue, downvalue;
  
  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL || gradient == NULL )
	 return( CCDLIB_NoPicturePointerError );

  xstart = picturerect->xpos + 1;
  xstop  = picturerect->xpos + picturerect->xsize - 1;
  ystart = ( picturerect->ypos + 1 ) * pictureparam->linelength;
  ystop  = ( picturerect->ypos + picturerect->ysize - 1) * pictureparam->linelength;

  ysm1 = ystart - pictureparam->linelength;
  ysp1 = ystart + pictureparam->linelength;
  for(ys = ystart;
		ys < ystop;
		ys += pictureparam->linelength )
  {
	 xm1 = xstart - 1;
    xp1 = xstart + 1;
	 for(x = xstart;
		  x < xstop;
		  x ++ )
    {
		leftvalue = ( (long)picture[ ysm1+xm1 ]
						 +(long)picture[ ys+xm1 ]
						 +(long)picture[ ysp1+xm1 ] ) / 3;

		rightvalue = ( (long)picture[ ysm1+xp1 ]
		 				  +(long)picture[ ys+xp1 ]
						  +(long)picture[ ysp1+xp1 ] ) / 3;

		grad = (short)( ( rightvalue - leftvalue ) >> 1 );
      
		gradient[ ys+x ].value.xgrad = (short)(
		  max( 0, min( pictureparam->dynamik-1, grad ) ) );

		upvalue = ( (long)picture[ ysm1+xp1 ]
		           +(long)picture[ ysm1+x ]
					  +(long)picture[ ysm1+xm1 ] ) / 3;

		downvalue = ( (long)picture[ ysp1+xp1 ]                       
		 				 +(long)picture[ ysp1+x ]
						 +(long)picture[ ysp1+xm1 ] ) / 3;

		grad = (short)( ( upvalue - downvalue ) >> 1 );
      
		gradient[ ys+x ].value.ygrad = (short)(
		  max( 0, min( pictureparam->dynamik-1, grad ) ) );

		xm1++;
      xp1++;
	 }

	 ysm1 += pictureparam->linelength;
	 ysp1 += pictureparam->linelength;
  }

  return( CCDLIB_Successful );
}
                                       

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcAbsGradientFromPic : Routine zum Berechnen der Gradien-    *
*               tenamplitude ( Betragswert ) Aus Geschwindigkeitsgruen-   *
*               den kann optional eine auf 4% genaue Naeherungsmethode    *
*               benutzt werden.                                           *
*                                                                         *
*   Parameter:  *picturerect : Koordinate der oberen linken Ecke des      *
*                              Arbeitsbereiches im Bild                   *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*               *gradient    : 32-Bit-Pointer auf die Gradientenwerte     *
*                                                                         *
*               Ein Gradientenwert ist eine Union aus einem 32-Bit-Wert   *
*               und dem X- und Y-Anteil des Vektors als short             *
*               CCDLIB_GradientOperationEnum : Methodenauswahl            *
*                                                                         *
*               ACHTUNG: maskdiameter darf nur ungerade Werte annehmen!   *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (05.06.1994)                                          *
*   Written by: Georg Dittie, 1994                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcAbsGradientFromPic
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  CCDLIB_GradientUnion * gradient,
  enum CCDLIB_GradientOperationEnum operationtype
)
{
  long avalue, bvalue, value;
  double avaluef, bvaluef, valuef;
  long afactor, bfactor, halfdynamikbits;
  
  long x, ys, xstart, xstop, ystart, ystop;

  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL || gradient == NULL )
	 return( CCDLIB_NoPicturePointerError );

  afactor = (long)floor( (double)pictureparam->dynamik * 0.960 + 0.5 );
  bfactor = (long)floor( (double)pictureparam->dynamik * 0.398 + 0.5 );
  halfdynamikbits = (long)floor( log( pictureparam->dynamik ) / log( 2. ) + 0.5 ) - 1;

  xstart = picturerect->xpos + 1;
  xstop  = picturerect->xpos + picturerect->xsize - 1;
  ystart = ( picturerect->ypos + 1 ) * pictureparam->linelength;
  ystop  = ( picturerect->ypos + picturerect->ysize - 1) * pictureparam->linelength;

  for(ys = ystart;
		ys < ystop;
		ys += pictureparam->linelength )
  {
	 switch( operationtype )
    {
    case CCDLIB_AproximationConst:
	   for(x = xstart;
		    x < xstop;
		    x ++ )
      {
		  avalue = (long)abs( gradient[ ys+x ].value.xgrad );
		  bvalue = (long)abs( gradient[ ys+x ].value.ygrad );
      
        // Naeherung geometrische Addition aus Bronstein Semendjajew
      
        if( avalue > bvalue )
        {
          value = ( afactor*avalue + bfactor*bvalue ) >> halfdynamikbits;
		  }
        else
        {
          value = ( afactor*bvalue + bfactor*avalue ) >> halfdynamikbits;
        }

		  picture[ x+ys ] = (short)(
			 max( 0, min( (long)pictureparam->dynamik-1, value ) ) );
		}
		break;
	 case CCDLIB_ExactConst:
		for(x = xstart;
		    x < xstop;
		    x ++ )
      {
		  avaluef = (double)( gradient[ ys+x ].value.xgrad );
		  bvaluef = (double)( gradient[ ys+x ].value.ygrad );
      
		  valuef = (short)floor( sqrt( avaluef*avaluef + bvaluef*bvaluef ) + 0.5 );
		  picture[ x+ys ] = ( max( 0, min( pictureparam->dynamik-1, valuef ) ) );
		}
		break;
	 default:
		return( CCDLIB_WrongParameterError );
	 }
  }

  return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcSobel : Routine zum direkten Berechnen des Sobeloperators  *
*   ohne Umweg ber den Gradienten aus einem Schwarzweibild              *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *sourcepic   : 32-Bit-Pointer auf die Bildquelle          *
*               *destpic     : 32-Bit-Pointer auf das Bildziel            *
*               offset       : +/- oder mit Offset bei dynamik/2          *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (23.08.2002)                                          *
*   Written by: Georg Dittie, 2002                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcSobel
(
   short *sourcepic,
   short *destpic,
   CCDLIB_PictureparamStruct *pictureparam
)
{
   long x, y, ys, xstop, ysstop, ysstep,
        ysm1, ysp1, index1, index2;
   double sval, hval;

   short *linebuf[2];

   if( sourcepic == NULL || destpic == NULL )
	   return( CCDLIB_NoPicturePointerError );

   ysstep = pictureparam->linelength;

   linebuf[0] = (short*)malloc( ysstep*2*sizeof(short) );
   if( linebuf == NULL  )
      return( CCDLIB_OutOfMemoryError );
   linebuf[1] = linebuf[0]+ysstep;
   memset( linebuf[0], 0, ysstep*2*sizeof(short) );

   xstop  = pictureparam->linelength;
   ysstop  = (pictureparam->columnheight-1)*ysstep;

   for( ys = ysstep, y = 1; ys < ysstop; ys += ysstep, y++ )
   {
      index1 = y%2;
      index2 = (y-1)%2;
      ysm1 = ys-ysstep;
      ysp1 = ys+ysstep;
      for( x = 0; x < xstop; x++ )
      {
         sval = ((double)sourcepic[ysp1+x] - (double)sourcepic[ysm1+x])/2.;
         hval = ((double)sourcepic[ys+x+1] - (double)sourcepic[ys+x-1])/2.;

         linebuf[index1][x] = (short)( sqrt( sval*sval + hval*hval ));
      }
      for( x = 0; x < xstop; x++ )
         destpic[ ysm1+x ] = linebuf[index2][x];
   }
   ys -= ysstep;
   index2 = (y-1)%2;
   for( x = 0; x < xstop; x++ )
       destpic[ ys+x ] = linebuf[index2][x];

   free( linebuf[0] );
   return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcPhase : Routine zum direkten Berechnen des Sobeloperators  *
*   ohne Umweg ber den Gradienten aus einem Schwarzweibild              *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *sourcepic   : 32-Bit-Pointer auf die Bildquelle          *
*               *destpic     : 32-Bit-Pointer auf das Bildziel            *
*               offset       : +/- oder mit Offset bei dynamik/2          *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (24.08.2002)                                          *
*   Written by: Georg Dittie, 2002                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcPhase
(
   short *sourcepic,
   short *destpic,
   CCDLIB_PictureparamStruct *pictureparam
)
{
   long x, y, ys, xstop, ysstop, ysstep,
        ysm1, ysp1, index1, index2;
   double sval, hval;

   short *linebuf[2];

   if( sourcepic == NULL || destpic == NULL )
	   return( CCDLIB_NoPicturePointerError );

   ysstep = pictureparam->linelength;

   linebuf[0] = (short*)malloc( ysstep*2*sizeof(short) );
   if( linebuf == NULL  )
      return( CCDLIB_OutOfMemoryError );
   linebuf[1] = linebuf[0]+ysstep;
   memset( linebuf[0], 0, ysstep*2*sizeof(short) );

   xstop  = pictureparam->linelength;
   ysstop  = (pictureparam->columnheight-1)*ysstep;

   for( ys = ysstep, y = 1; ys < ysstop; ys += ysstep, y++ )
   {
      index1 = y%2;
      index2 = (y-1)%2;
      ysm1 = ys-ysstep;
      ysp1 = ys+ysstep;
      for( x = 0; x < xstop; x++ )
      {
         sval = ((double)sourcepic[ysp1+x] - (double)sourcepic[ysm1+x])/2.;
         hval = ((double)sourcepic[ys+x+1] - (double)sourcepic[ys+x-1])/2.;

         if( sval != 0. || hval != 0 )
            linebuf[index1][x] = (short)( atan2( sval, hval )*32767./PI );
         else
            linebuf[index1][x] = 0.;
      }
      for( x = 0; x < xstop; x++ )
         destpic[ ysm1+x ] = linebuf[index2][x];
   }
   ys -= ysstep;
   index2 = (y-1)%2;
   for( x = 0; x < xstop; x++ )
       destpic[ ys+x ] = linebuf[index2][x];

   free( linebuf[0] );
   return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_BinarizePicture : Routine zum Binarisieren eines Grauwert-     *
*               bildes. Dabei koennen die beiden zuzuordnenden Grau-  	  *
*               werte gewaehlt werden.                                    *
*                                                                         *
*   Parameter:  *picturerect : Koordinate der oberen linken Ecke des      *
*                              Arbeitsbereiches im Bild                   *
*               *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*               treshold     : Binarisierungsschwelle                     *
*               logreyval    : Grauwert fuer Werte unter der Schwelle     *
*               higreyval    : Grauwert fuer Werte ueber der Schwelle     *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Das Resultatbild steht an derselben Stelle wie das Ausgangsbild.      *
*   ACHTUNG: Das Ausgangsbild wird dabei ueberschrieben !                 *
*                                                                         *
*   Version:    1.0 (05.06.1994)                                          *
*   Written by: Georg Dittie, 1994                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_BinarizePicture
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	short treshold,
	short logreyval,
	short higreyval
)
{
  long x, ys, xstart, ystart, xstop, ystop;

  short value;

  if( CCDLIB_CheckRectangleValues( picturerect, pictureparam ) == FALSE )
	 return( CCDLIB_WrongRectangleValuesError );

  if( picture == NULL )
	 return( CCDLIB_NoPicturePointerError );

  xstart = picturerect->xpos;
  xstop  = picturerect->xpos + picturerect->xsize;
  ystart = ( picturerect->ypos ) * pictureparam->linelength;
  ystop  = ( picturerect->ypos + picturerect->ysize ) * pictureparam->linelength;

  for(ys = ystart;
		ys < ystop;
		ys += pictureparam->linelength)
  {
	 for(x = xstart;
		  x < xstop;
		  x ++)
    {
		value = picture[ x+ys ];

		picture[ x+ys ] = ( value > treshold ? higreyval : logreyval );
    }
  }

  return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_MaximizeSaturation : Routine for maximizing the saturation     *
*               of standard RGB images.                                   *
*                                                                         *
*   Parameter:  *pictureparam: Systemparameter des zu bearbeitenden       *
*                              Bildes                                     *
*               *sourcepic   : 32-Bit-Pointer auf die Ausgangsilddaten    *
*               *destpic     : 32-Bit-Pointer auf die Zielbilddaten       *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (3. 09. 2000)                                         *
*   Written by: Georg Dittie, 2000                                        *
*                                                                         *
**************************************************************************/


enum CCDLIB_ErrorEnum CCDLIB_MaximizeSaturation
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   double exponent
)
{
	long x, ys, xstop, ystop, ystep;
   double rgb[3], hsi[3], erg[3], **fwd_matrix, **bck_matrix;
   double smax, sval, huefactor, divfactor;

   if( pictureparam->colorchannel != CCDLIB_RGBChannelColor )
      return( CCDLIB_WrongParameterError );

   CCDLIB_Matrixalloc( &fwd_matrix, 3 );
   CCDLIB_Matrixalloc( &bck_matrix, 3 );

   fwd_matrix[0][0] =  2./sqrt(6.);
   fwd_matrix[1][0] = -1./sqrt(6.);
   fwd_matrix[2][0] = -1./sqrt(6.);
   fwd_matrix[0][1] =  0.;
   fwd_matrix[1][1] =  1./sqrt(2.);
   fwd_matrix[2][1] = -1./sqrt(2.);
   fwd_matrix[0][2] =  1./sqrt(3.);
   fwd_matrix[1][2] =  1./sqrt(3.);
   fwd_matrix[2][2] =  1./sqrt(3.);

   bck_matrix[0][0] =  2./sqrt(6.);
   bck_matrix[1][0] =  0.;
   bck_matrix[2][0] =  1./sqrt(3.);
   bck_matrix[0][1] = -1./sqrt(6.);
   bck_matrix[1][1] =  1./sqrt(2.);
   bck_matrix[2][1] =  1./sqrt(3.);
   bck_matrix[0][2] = -1./sqrt(6.);
   bck_matrix[1][2] = -1./sqrt(2.);
   bck_matrix[2][2] =  1./sqrt(3.);

   xstop = pictureparam->linelength*3;
   ystop = pictureparam->linelength*pictureparam->columnheight*3;
   ystep = pictureparam->linelength*3;

   smax = 0;
	for(ys = 0;
		 ys < ystop;
		 ys += ystep )
	{
		for(x = 0;
			 x < xstop;
			 x += 3 )
		{
			rgb[ 0 ] = sourcepic[ x+ys+2 ];
			rgb[ 1 ] = sourcepic[ x+ys+1 ];
			rgb[ 2 ] = sourcepic[ x+ys ];

         CCDLIB_Matrixvectormul( erg, fwd_matrix, rgb, 3 );

         sval = erg[0]*erg[0] + erg[1]*erg[1];
         if( sval > smax ) smax = sval;
		}
	}
   if( smax == 0 )
   {
      CCDLIB_Matrixfree( fwd_matrix );
      CCDLIB_Matrixfree( bck_matrix );
      return( CCDLIB_Successful );
   }
   divfactor = 1./sqrt( smax );
   huefactor = 32767.*divfactor;

	for(ys = 0;
		 ys < ystop;
		 ys += ystep )
	{
		for(x = 0;
			 x < xstop;
			 x += 3 )
		{
			rgb[ 0 ] = sourcepic[ x+ys+2 ];
			rgb[ 1 ] = sourcepic[ x+ys+1 ];
			rgb[ 2 ] = sourcepic[ x+ys ];

         CCDLIB_Matrixvectormul( erg, fwd_matrix, rgb, 3 );

         if( erg[1] != 0. )
            hsi[0] = atan2( erg[0], erg[1] );
         else if( erg[0] > 0. )
            hsi[0] = PID2;
         else if( erg[0] < 0. )
            hsi[0] = -PID2;
         else
            hsi[0] = 0.;
         if( exponent != 1. )
            hsi[1] = 32767.*pow( sqrt(erg[0]*erg[0] + erg[1]*erg[1])*divfactor, exponent );
         else
            hsi[1] = huefactor*sqrt(erg[0]*erg[0] + erg[1]*erg[1]);
         hsi[2] = erg[2] * sqrt(3.);

         erg[0] = hsi[1] * sin(hsi[0]);
         erg[1] = hsi[1] * cos(hsi[0]);
         erg[2] = hsi[2] / sqrt(3.);

         CCDLIB_Matrixvectormul( rgb, bck_matrix, erg, 3 );

			destpic[ x+ys ]   = (short)(max(min( rgb[ 2 ], 32767. ), 0.));
			destpic[ x+ys+1 ] = (short)(max(min( rgb[ 1 ], 32767. ), 0.));
			destpic[ x+ys+2 ] = (short)(max(min( rgb[ 0 ], 32767. ), 0.));
		}
	}

   CCDLIB_Matrixfree( fwd_matrix );
   CCDLIB_Matrixfree( bck_matrix );

   return( CCDLIB_Successful );
}

/*** Help fuctions ***/

BOOL CCDLIB_CheckRectangleValues
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam
)
{
  if( picturerect->xpos < 0 ) return( FALSE );
  if( picturerect->ypos < 0 ) return( FALSE );
  if( picturerect->xpos >= pictureparam->linelength ) return( FALSE );
  if( picturerect->ypos >= pictureparam->columnheight ) return( FALSE );

  if( picturerect->xsize < 0 ) return( FALSE );
  if( picturerect->ysize < 0 ) return( FALSE );
  if( picturerect->xpos + picturerect->xsize > pictureparam->linelength ) return( FALSE );
  if( picturerect->ypos + picturerect->ysize > pictureparam->columnheight ) return( FALSE );

  return( TRUE );
}

/*** Funktionen zur Matrizen- und Vectorrechnung ***/

BOOL CCDLIB_Matrixalloc(
	double ***matrix,
	int size )
{
	int i,j;

   (*matrix) = (double **)malloc( size*sizeof( double* ) );
	if( (*matrix) == NULL )
		return( FALSE );

	(*matrix)[ 0 ] = (double *)malloc( size*size*sizeof( double ) );
   if( (*matrix)[ 0 ] == NULL )
	{
		free( *matrix );
		return( FALSE );
   }

   for( i = 1; i < size; i++ )
		(*matrix)[ i ] = (*matrix)[ 0 ] + size*i;

	for( i = 0; i < size; i++ )
      for( j = 0; j < size; j++ )
			(*matrix)[ i ][ j ] = 0.;

   return( TRUE );
}

void CCDLIB_Matrixfree(
   double **matrix )
{
   free( matrix[ 0 ] );
   free( matrix );
}

void CCDLIB_Matrixcopy(
   double **destmat,
   double **srcmat,
	int    size )
{
   int x, y;

	for( y=0; y<size; y++ )
		for( x=0; x<size; x++ )
			destmat[x][y] = srcmat[x][y];
}

void CCDLIB_Matrixmul(
	double **result,
	double **mat1,
	double **mat2,
	int    size )
{
	int x, y, k;
	double sum, A, B;

	for( y=0; y<size; y++ )
	{
		for( x=0; x<size; x++ )
		{
			sum = 0.;
			for( k=0; k<size; k++ )
			{
				A = mat1[k][y];
				B = mat2[x][k];
				if( A != 0. || B != 0. )
					sum += A * B;
			}
			result[x][y] = sum;
		}
	}
}

/* Sieht idiotisch aus, aber der Compiler macht prozessoroptimalen
	Code draus ! Geschwindigkeitssteigerung etwa 50 % */

void CCDLIB_FastMatVecMul(
	double *result,
	double **matrix,
	double *vector )
{
	double sum, M, V0, V1, V2, V3, *mx;

	mx = matrix[0];
	V0 = vector[0];
	V1 = vector[1];
	V2 = vector[2];
	V3 = vector[3];

	M = mx[0];
	sum = M * V0;
	M = mx[4];
	if( M != 0. )
		if( V1 != 0. )
			sum += M * V1;
	M = mx[8];
	if( M != 0. )
		if( V2 != 0. )
			sum += M * V2;
	M = mx[12];
	if( M != 0. )
		if( V3 != 0. )
			sum += M * V3;
	result[0] = sum;

	M = mx[1];
	sum = M * V0;
	M = mx[5];
	if( M != 0. )
		if( V1 != 0. )
			sum += M * V1;
	M = mx[9];
	if( M != 0. )
		if( V2 != 0. )
			sum += M * V2;
	M = mx[13];
	if( M != 0. )
		if( V3 != 0. )
			sum += M * V3;
	result[1] = sum;

	M = mx[2];
	sum = M * V0;
	M = mx[6];
	if( M != 0. )
		if( V1 != 0. )
			sum += M * V1;
	M = mx[10];
	if( M != 0. )
		if( V2 != 0. )
			sum += M * V2;
	M = mx[14];
	if( M != 0. )
		if( V3 != 0. )
			sum += M * V3;
	result[2] = sum;

	M = mx[3];
	sum = M * V0;
	M = mx[7];
	if( M != 0. )
		if( V1 != 0. )
			sum += M * V1;
	M = mx[11];
	if( M != 0. )
		if( V2 != 0. )
			sum += M * V2;
	M = mx[15];
	if( M != 0. )
		if( V3 != 0. )
			sum += M * V3;
	result[3] = sum;
}


void CCDLIB_Matrixvectormul(
	double *result,
	double **matrix,
	double *vector,
	int size )
{
	int x, y;
	double sum, M, V;

	for( y=0; y<size; y++ )
	{
		sum = 0.;
		for( x=0; x<size; x++ )
		{
			M = matrix[x][y];
			V = vector[x];
			if( M != 0. && V != 0. )
				sum += M * V;
		}
		result[y] = sum;
	}
}


/*** Passende Mae berechnen ***/

void CCDLIB_GetMatchingScale(
	CCDLIB_RectangleStruct * matchingrect,
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam )
{
	if( (double)picturerect->xsize/(double)pictureparam->linelength <
		 (double)picturerect->ysize/(double)pictureparam->columnheight )
	{
		matchingrect->xpos = 0;
		matchingrect->ypos = 0;
		matchingrect->xsize = (int)floor(
				(double)picturerect->xsize
	   	 / (double)picturerect->ysize
			 * (double)pictureparam->columnheight);
		matchingrect->ysize = pictureparam->columnheight;
	}
	else
	{
		matchingrect->xpos = 0;
		matchingrect->ypos = 0;
		matchingrect->xsize = pictureparam->linelength;
		matchingrect->ysize = (int)floor(
			  (double)picturerect->ysize
			/ (double)picturerect->xsize
			* (double)pictureparam->linelength);
	}
}

/*** Ortstransformationen ***/

BOOL CCDLIB_GetProjectionPoint(
	double *xp,
	double *yp,
	double xobj,
	double yobj,
	double zobj,
	double **projmatrix,
	long   linelength,
	long   columnheight,
	long   srclength,
	long   srcheight,
	double limitwidth )
{
	double vector[ 4 ],
			 result[ 4 ] = { 0., 0., 0., 0. };
	double xmid, ymid, xsrc, ysrc, xlimit, ylimit;
	BOOL   visible;

	xmid = (double)(linelength >> 1);
	ymid = (double)(columnheight >> 1);
	xsrc = (double)(srclength >> 1);
	ysrc = (double)(srcheight >> 1);

	vector[ 0 ] = xobj - xmid;
	vector[ 1 ] = ymid - yobj;
	vector[ 2 ] = zobj;
	vector[ 3 ] = 1.;

	CCDLIB_FastMatVecMul( result, (double**)projmatrix, vector );
   //CCDLIB_Matrixvectormul( result, (double**)projmatrix, vector, 4 );

	if( result[ 3 ] == 0. )
		return( FALSE );
	*xp = result[ 0 ] / result[ 3 ];
	*yp = result[ 1 ] / result[ 3 ];

	xlimit = xsrc-limitwidth;
	ylimit = ysrc-limitwidth;

	if( *xp < -xlimit || *xp >= xlimit ||
		 *yp < -ylimit || *yp >= ylimit )
		visible = FALSE;
	else
		visible = TRUE;

	*xp = *xp + xsrc;
	*yp = ysrc - *yp;

	return( visible );
}

/*** Berechnung der normierten Koordinatenmatrix fuer CalcRotScale ***/

enum CCDLIB_ErrorEnum CCDLIB_CalcRotScalematrix(
	double **projmatrix,
	double rotation,
	double magnification,
	double xpixelsize,
	double ypixelsize )
{
	double **Ps = NULL, **P = NULL;

	if( magnification <= 0. || xpixelsize == 0. || ypixelsize == 0. )
		return( CCDLIB_WrongParameterError );

	if( projmatrix == NULL )
		return( CCDLIB_NoPointerError );

	if( !CCDLIB_Matrixalloc( &Ps, 4 ) )
	{
		return( CCDLIB_OutOfMemoryError );
	}

	if( !CCDLIB_Matrixalloc( &P, 4 ) )
	{
		CCDLIB_Matrixfree( Ps );
		return( CCDLIB_OutOfMemoryError );
	}

	Ps[ 0 ][ 0 ] = cos( -rotation );
	Ps[ 1 ][ 0 ] = sin( -rotation );
	Ps[ 0 ][ 1 ] = -sin( -rotation );
	Ps[ 1 ][ 1 ] = cos( -rotation );
	Ps[ 2 ][ 2 ] = 1.;
	Ps[ 3 ][ 3 ] = 1.;

	P[ 0 ][ 0 ] = xpixelsize/ypixelsize;
	P[ 1 ][ 1 ] = 1.;
	P[ 2 ][ 3 ] = 1.;
	P[ 3 ][ 3 ] = magnification;

	CCDLIB_Matrixmul( projmatrix, P, Ps, 4 );

	CCDLIB_Matrixfree( Ps );
	CCDLIB_Matrixfree( P );

	return( CCDLIB_Successful );
}

/**************************************************************************
*                                                                         *
*   CCDLIB_CalcRotScale : Routine zur Entzerrung, Skalierung und          *
*               Rotation eines beliebigen Bildausschnitts                 *
*                                                                         *
*   Parameter:  *picturerect : Zu entzerrender Ausschnitt im Zielbild     *
*               *pictureparam: Systemparameter des Quellbilds             *
*               *picture     : 32-Bit-Pointer auf die Bilddaten           *
*               rotation     : Positionswinkel des Ausschnitts            *
*               magnification: Vergroesserung des Ausschnitts             *
*                                                                         *
*   Ergebnis:   CCDLIB_ErrorEnum : Detaillierte Fehler/Erfolgsmeldung     *
*                                                                         *
*               CCDLIB_Successful:            Alles OK !                  *
*               CCDLIB_WrongParameterError:   Unsinnige Werte bekommen    *
*               CCDLIB_WrongRectangleValuesE: Unsinnige Bildmasse         *
*               CCDLIB_NoPointerError:        Nicht initialisierte Ptr.   *
*               CCDLIB_NoPicturePointerError: Keine Bilddaten bekommen    *
*               CCDLIB_OutOfMemoryError:      Zu wenig Speicher           *
*                                                                         *
*   Version:    1.0 (Heiligabend 1997)                                    *
*   Written by: Georg Dittie, 1997                                        *
*                                                                         *
**************************************************************************/

enum CCDLIB_ErrorEnum CCDLIB_CalcRotScale(
	CCDLIB_PictureparamStruct * sourcepicpar,
	CCDLIB_PictureparamStruct * destpicpar,
	short *sourcepic,
	short *destpic,
	long turnxpos,
   long turnypos,
	double rotation,
	double magnification,
	enum CCDLIB_EnlargeMethodEnum method )
{
	double **projmatrix;

	double xp, yp;
	double xm00, xm01, xm02, xm03,
			 xm10, xm11, xm12, xm13,
			 xm20, xm21, xm22, xm23,
			 xm30, xm31, xm32, xm33;
	long v00, v01, v10, v11, xfrac, yfrac;
	double *sintab0, *sintab1, *sintab2, *sintab3, sinsum, nenner, angle, limitwidth;
	int x, xq, ys, yq, xqstart, yqstart, xqlimit, yqlimit,
		 xstart, xstop, ysstart, ysstop, ystep, yqstep,
		 xq0, xq1, xq2, xq3, yq0, yq1, yq2, yq3;


	if( sourcepic == NULL || destpic == NULL )
		return( CCDLIB_NoPicturePointerError );

	if( sourcepic == destpic )
		return( CCDLIB_WrongParameterError );

	if( !CCDLIB_Matrixalloc( &projmatrix, 4 ) )
	  return( CCDLIB_OutOfMemoryError );

	xqstart = turnxpos-destpicpar->linelength/2;
	xstart  = 0;
	xstop   = destpicpar->linelength;
	yqstart = turnypos-destpicpar->columnheight/2;
	ysstart = 0;
	ysstop  = destpicpar->linelength * destpicpar->columnheight;
	ystep   = destpicpar->linelength;
	yqstep  = sourcepicpar->linelength;
   xqlimit = sourcepicpar->linelength;
   yqlimit = sourcepicpar->linelength*sourcepicpar->columnheight;
	limitwidth = 2.;

	if( method == CCDLIB_Lanczos2Const )
	{
		limitwidth = 3.;

		sintab0 = (double *)malloc( 256*sizeof( double ));
		sintab1 = (double *)malloc( 256*sizeof( double ));
		sintab2 = (double *)malloc( 256*sizeof( double ));
		sintab3 = (double *)malloc( 256*sizeof( double ));

		if( sintab0 == NULL || sintab0 == NULL || sintab0 == NULL || sintab0 == NULL )
		{
			if( sintab0 != NULL ) free( sintab0 );
			if( sintab1 != NULL ) free( sintab1 );
			if( sintab2 != NULL ) free( sintab2 );
			if( sintab3 != NULL ) free( sintab3 );
			return( CCDLIB_OutOfMemoryError );
		}

		for( xfrac = 0; xfrac < 256; xfrac++ )
		{
			nenner = PID2*( (double)xfrac/256. - 2. );
			angle  = PI*(double)xfrac/256. - PI2;
			if( nenner != 0. )
				sintab0[ xfrac ] = sin(nenner)/nenner*sin(angle)/angle;
			else
				sintab0[ xfrac ] = 1.;

			nenner = PID2*( (double)xfrac/256. - 1. );
			angle  = PI*(double)xfrac/256. - PI;
			if( nenner != 0. )
				sintab1[ xfrac ] = sin(nenner)/nenner*sin(angle)/angle;
			else
				sintab1[ xfrac ] = 1.;

			nenner = PID2*( (double)xfrac/256. );
			angle  = PI*(double)xfrac/256.;
			if( nenner != 0. )
				sintab2[ xfrac ] = sin(nenner)/nenner*sin(angle)/angle;
			else
				sintab2[ xfrac ] = 1.;

			nenner = PID2*( (double)xfrac/256. + 1. );
			angle  = PI*(double)xfrac/256. + PI;
			if( nenner != 0. )
				sintab3[ xfrac ] = sin(nenner)/nenner*sin(angle)/angle;
			else
				sintab3[ xfrac ] = 1.;

			sinsum = sintab0[ xfrac ]+sintab1[ xfrac ]+sintab2[ xfrac ]+sintab3[ xfrac ];
			sinsum = sinsum;

			sintab0[ xfrac ] /= sinsum;
			sintab1[ xfrac ] /= sinsum;
			sintab2[ xfrac ] /= sinsum;
			sintab3[ xfrac ] /= sinsum;
		}
	}

	if( CCDLIB_CalcRotScalematrix(
		 (double **)projmatrix,
		 rotation,
		 magnification,
		 sourcepicpar->xpixelsize,
		 sourcepicpar->ypixelsize ) != CCDLIB_Successful )
		return( CCDLIB_WrongParameterError );

	if( sourcepicpar->colorchannel != CCDLIB_RGBChannelColor )
	{
		for( ys = ysstart, yq = yqstart; ys < ysstop; ys += ystep, yq++ )
		{
			for( x = xstart, xq = xqstart; x < xstop; x++, xq++ )
			{
				if( CCDLIB_GetProjectionPoint(
						&xp, &yp, (double)xq, (double)yq, 0., (double**)projmatrix,
                  destpicpar->linelength, destpicpar->columnheight,
						sourcepicpar->linelength, sourcepicpar->columnheight, limitwidth ) == FALSE )
				{
					destpic[ x+ys ] = destpicpar->blankvalue.word;
				}
				else
				{
					switch( method )
					{
					case CCDLIB_GetSampleConst:               /* Verkleinerung durch Undersamplen */
					case CCDLIB_NearestNeighborConst:         /* Naechster Nachbar-Kloetzelgrafik */
						xq0 = (int)floor( xp+0.5 );
						yq0 = (int)floor( yp+0.5 ) * yqstep;

						destpic[ x+ys ] = sourcepic[ xq0+yq0 ];
						break;
					case CCDLIB_MeanOfVicinityConst:          /* Verkleinerung durch Mitteln */
					case CCDLIB_LinearIntpolConst:            /* Lineare Interpolation */
						xq0 = (int)floor( xp );
						xq1 = xq0+1;
						yq0 = (int)floor( yp ) * yqstep;
						yq1 = yq0 + yqstep;


						xfrac = (long)( frac( xp )*256. );
						yfrac = (long)( frac( yp )*256. );

						v00 = (long)sourcepic[ xq0+yq0 ];
						v10 = (long)sourcepic[ xq1+yq0 ];
						v01 = (long)sourcepic[ xq0+yq1 ];
						v11 = (long)sourcepic[ xq1+yq1 ];

						destpic[ x+ys ] = (short)(
							( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
							(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );
						break;
					case CCDLIB_Lanczos2Const:
						xq1 = (int)floor( xp );
						xq0 = --xq1;
						xq2 = ++xq0;
						xq3 = ++xq1;
						yq1 = (int)floor( yp ) * yqstep;
						yq0 = yq1 - yqstep;
						yq2 = yq1 + yqstep;
						yq3 = yq2 + yqstep;

						xfrac = (long)( frac( xp )*256. );
						yfrac = (long)( frac( yp )*256. );

						xm00 = (double)sourcepic[ yq0+xq0 ];
						xm10 = (double)sourcepic[ yq0+xq1 ];
						xm20 = (double)sourcepic[ yq0+xq2 ];
						xm30 = (double)sourcepic[ yq0+xq3 ];
						xm01 = (double)sourcepic[ yq1+xq0 ];
						xm11 = (double)sourcepic[ yq1+xq1 ];
						xm21 = (double)sourcepic[ yq1+xq2 ];
						xm31 = (double)sourcepic[ yq1+xq3 ];
						xm02 = (double)sourcepic[ yq2+xq0 ];
						xm12 = (double)sourcepic[ yq2+xq1 ];
						xm22 = (double)sourcepic[ yq2+xq2 ];
						xm32 = (double)sourcepic[ yq2+xq3 ];
						xm03 = (double)sourcepic[ yq3+xq0 ];
						xm13 = (double)sourcepic[ yq3+xq1 ];
						xm23 = (double)sourcepic[ yq3+xq2 ];
						xm33 = (double)sourcepic[ yq3+xq3 ];

						destpic[ ys+x ] = (short)floor(
							( xm00 * sintab0[ xfrac ] * sintab0[ yfrac ]
							+ xm10 * sintab1[ xfrac ] * sintab0[ yfrac ]
							+ xm20 * sintab2[ xfrac ] * sintab0[ yfrac ]
							+ xm30 * sintab3[ xfrac ] * sintab0[ yfrac ]
							+ xm01 * sintab0[ xfrac ] * sintab1[ yfrac ]
							+ xm11 * sintab1[ xfrac ] * sintab1[ yfrac ]
							+ xm21 * sintab2[ xfrac ] * sintab1[ yfrac ]
							+ xm31 * sintab3[ xfrac ] * sintab1[ yfrac ]
							+ xm02 * sintab0[ xfrac ] * sintab2[ yfrac ]
							+ xm12 * sintab1[ xfrac ] * sintab2[ yfrac ]
							+ xm22 * sintab2[ xfrac ] * sintab2[ yfrac ]
							+ xm32 * sintab3[ xfrac ] * sintab2[ yfrac ]
							+ xm03 * sintab0[ xfrac ] * sintab3[ yfrac ]
							+ xm13 * sintab1[ xfrac ] * sintab3[ yfrac ]
							+ xm23 * sintab2[ xfrac ] * sintab3[ yfrac ]
							+ xm33 * sintab3[ xfrac ] * sintab3[ yfrac ] ) );
						break;
					default:
						return( CCDLIB_WrongParameterError );
					}
				}
			}
		}
	}
	else
	{
		ystep   *= 3;
		yqstep  *= 3;
		xstart  *= 3;
		xstop   *= 3;
		ysstart *= 3;
		ysstop  *= 3;
      xqlimit *= 3;
      yqlimit *= 3;
		for( ys = ysstart, yq = yqstart; ys < ysstop; ys += ystep, yq++ )
		{
			for( x = xstart, xq = xqstart; x < xstop; x+=3, xq++ )
			{
				if( CCDLIB_GetProjectionPoint(
						&xp, &yp, (double)xq, (double)yq, 0., (double**)projmatrix,
                  destpicpar->linelength, destpicpar->columnheight,
						sourcepicpar->linelength, sourcepicpar->columnheight, limitwidth ) == FALSE )
				{
					destpic[ x+ys+2 ] = destpicpar->blankvalue.word;
					destpic[ x+ys+1 ] = destpicpar->gblankvalue.word;
					destpic[ x+ys   ] = destpicpar->bblankvalue.word;
				}
				else
				{
					switch( method )
					{
					case CCDLIB_GetSampleConst:               /* Verkleinerung durch Undersamplen */
					case CCDLIB_NearestNeighborConst:         /* Naechster Nachbar-Kloetzelgrafik */
						xq0 = (int)floor( xp+0.5 ) * 3;
						yq0 = (int)floor( yp+0.5 ) * yqstep;
                  if( xq0<0 || xq0>=xqlimit || yq0<0 || yq0>=yqlimit ) break;

						destpic[ x+ys   ] = sourcepic[ xq0+yq0 ];
						destpic[ x+ys+1 ] = sourcepic[ xq0+yq0+1 ];
						destpic[ x+ys+2 ] = sourcepic[ xq0+yq0+2 ];
						break;
					case CCDLIB_MeanOfVicinityConst:          /* Verkleinerung durch Mitteln */
					case CCDLIB_LinearIntpolConst:            /* Lineare Interpolation */
						xq0 = (int)floor( xp ) * 3;
						xq1 = xq0 + 3;
						yq0 = (int)floor( yp ) * yqstep;
						yq1 = yq0 + yqstep;
                  if( xq0<0 || xq1>=xqlimit || yq0<0 || yq1>=yqlimit ) break;

						xfrac = (long)( frac( xp )*256. );
						yfrac = (long)( frac( yp )*256. );

						v00 = (long)sourcepic[ xq0+yq0 ];
						v10 = (long)sourcepic[ xq1+yq0 ];
						v01 = (long)sourcepic[ xq0+yq1 ];
						v11 = (long)sourcepic[ xq1+yq1 ];

						destpic[ x+ys ] = (short)(
							( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
							(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );

						v00 = (long)sourcepic[ xq0+yq0+1 ];
						v10 = (long)sourcepic[ xq1+yq0+1 ];
						v01 = (long)sourcepic[ xq0+yq1+1 ];
						v11 = (long)sourcepic[ xq1+yq1+1 ];

						destpic[ x+ys+1 ] = (short)(
							( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
							(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );

						v00 = (long)sourcepic[ xq0+yq0+2 ];
						v10 = (long)sourcepic[ xq1+yq0+2 ];
						v01 = (long)sourcepic[ xq0+yq1+2 ];
						v11 = (long)sourcepic[ xq1+yq1+2 ];

						destpic[ x+ys+2 ] = (short)(
							( (v00<<16) + (v10-v00)*(xfrac<<8) + (((v01-v00)<<8) +
							(v11-v01-v10+v00)*xfrac)*yfrac ) >> 16 );
						break;
					case CCDLIB_Lanczos2Const:
						xq1 = (int)floor( xp ) * 3;
						xq0 = xq1-3;
						xq2 = xq0+3;
						xq3 = xq1+6;
						yq1 = (int)floor( yp ) * yqstep;
						yq0 = yq1 - yqstep;
						yq2 = yq1 + yqstep;
						yq3 = yq2 + yqstep;
                  if( xq0<0 || xq3>=xqlimit || yq0<0 || yq3>=yqlimit ) break;

						xfrac = (long)( frac( xp )*256. );
						yfrac = 255-(long)( frac( yp )*256. );

						xm00 = (double)sourcepic[ yq0+xq0 ];
						xm10 = (double)sourcepic[ yq0+xq1 ];
						xm20 = (double)sourcepic[ yq0+xq2 ];
						xm30 = (double)sourcepic[ yq0+xq3 ];
						xm01 = (double)sourcepic[ yq1+xq0 ];
						xm11 = (double)sourcepic[ yq1+xq1 ];
						xm21 = (double)sourcepic[ yq1+xq2 ];
						xm31 = (double)sourcepic[ yq1+xq3 ];
						xm02 = (double)sourcepic[ yq2+xq0 ];
						xm12 = (double)sourcepic[ yq2+xq1 ];
						xm22 = (double)sourcepic[ yq2+xq2 ];
						xm32 = (double)sourcepic[ yq2+xq3 ];
						xm03 = (double)sourcepic[ yq3+xq0 ];
						xm13 = (double)sourcepic[ yq3+xq1 ];
						xm23 = (double)sourcepic[ yq3+xq2 ];
						xm33 = (double)sourcepic[ yq3+xq3 ];

						destpic[ ys+x ] = (short)floor(
							( xm00 * sintab0[ xfrac ] * sintab0[ yfrac ]
							+ xm10 * sintab1[ xfrac ] * sintab0[ yfrac ]
							+ xm20 * sintab2[ xfrac ] * sintab0[ yfrac ]
							+ xm30 * sintab3[ xfrac ] * sintab0[ yfrac ]
							+ xm01 * sintab0[ xfrac ] * sintab1[ yfrac ]
							+ xm11 * sintab1[ xfrac ] * sintab1[ yfrac ]
							+ xm21 * sintab2[ xfrac ] * sintab1[ yfrac ]
							+ xm31 * sintab3[ xfrac ] * sintab1[ yfrac ]
							+ xm02 * sintab0[ xfrac ] * sintab2[ yfrac ]
							+ xm12 * sintab1[ xfrac ] * sintab2[ yfrac ]
							+ xm22 * sintab2[ xfrac ] * sintab2[ yfrac ]
							+ xm32 * sintab3[ xfrac ] * sintab2[ yfrac ]
							+ xm03 * sintab0[ xfrac ] * sintab3[ yfrac ]
							+ xm13 * sintab1[ xfrac ] * sintab3[ yfrac ]
							+ xm23 * sintab2[ xfrac ] * sintab3[ yfrac ]
							+ xm33 * sintab3[ xfrac ] * sintab3[ yfrac ] ) );

						xq0++; xq1++; xq2++; xq3++;

						xm00 = (double)sourcepic[ yq0+xq0 ];
						xm10 = (double)sourcepic[ yq0+xq1 ];
						xm20 = (double)sourcepic[ yq0+xq2 ];
						xm30 = (double)sourcepic[ yq0+xq3 ];
						xm01 = (double)sourcepic[ yq1+xq0 ];
						xm11 = (double)sourcepic[ yq1+xq1 ];
						xm21 = (double)sourcepic[ yq1+xq2 ];
						xm31 = (double)sourcepic[ yq1+xq3 ];
						xm02 = (double)sourcepic[ yq2+xq0 ];
						xm12 = (double)sourcepic[ yq2+xq1 ];
						xm22 = (double)sourcepic[ yq2+xq2 ];
						xm32 = (double)sourcepic[ yq2+xq3 ];
						xm03 = (double)sourcepic[ yq3+xq0 ];
						xm13 = (double)sourcepic[ yq3+xq1 ];
						xm23 = (double)sourcepic[ yq3+xq2 ];
						xm33 = (double)sourcepic[ yq3+xq3 ];

						destpic[ ys+x+1 ] = (short)floor(
							( xm00 * sintab0[ xfrac ] * sintab0[ yfrac ]
							+ xm10 * sintab1[ xfrac ] * sintab0[ yfrac ]
							+ xm20 * sintab2[ xfrac ] * sintab0[ yfrac ]
							+ xm30 * sintab3[ xfrac ] * sintab0[ yfrac ]
							+ xm01 * sintab0[ xfrac ] * sintab1[ yfrac ]
							+ xm11 * sintab1[ xfrac ] * sintab1[ yfrac ]
							+ xm21 * sintab2[ xfrac ] * sintab1[ yfrac ]
							+ xm31 * sintab3[ xfrac ] * sintab1[ yfrac ]
							+ xm02 * sintab0[ xfrac ] * sintab2[ yfrac ]
							+ xm12 * sintab1[ xfrac ] * sintab2[ yfrac ]
							+ xm22 * sintab2[ xfrac ] * sintab2[ yfrac ]
							+ xm32 * sintab3[ xfrac ] * sintab2[ yfrac ]
							+ xm03 * sintab0[ xfrac ] * sintab3[ yfrac ]
							+ xm13 * sintab1[ xfrac ] * sintab3[ yfrac ]
							+ xm23 * sintab2[ xfrac ] * sintab3[ yfrac ]
							+ xm33 * sintab3[ xfrac ] * sintab3[ yfrac ] ) );

						xq0++; xq1++; xq2++; xq3++;

						xm00 = (double)sourcepic[ yq0+xq0 ];
						xm10 = (double)sourcepic[ yq0+xq1 ];
						xm20 = (double)sourcepic[ yq0+xq2 ];
						xm30 = (double)sourcepic[ yq0+xq3 ];
						xm01 = (double)sourcepic[ yq1+xq0 ];
						xm11 = (double)sourcepic[ yq1+xq1 ];
						xm21 = (double)sourcepic[ yq1+xq2 ];
						xm31 = (double)sourcepic[ yq1+xq3 ];
						xm02 = (double)sourcepic[ yq2+xq0 ];
						xm12 = (double)sourcepic[ yq2+xq1 ];
						xm22 = (double)sourcepic[ yq2+xq2 ];
						xm32 = (double)sourcepic[ yq2+xq3 ];
						xm03 = (double)sourcepic[ yq3+xq0 ];
						xm13 = (double)sourcepic[ yq3+xq1 ];
						xm23 = (double)sourcepic[ yq3+xq2 ];
						xm33 = (double)sourcepic[ yq3+xq3 ];

						destpic[ ys+x+2 ] = (short)floor(
							( xm00 * sintab0[ xfrac ] * sintab0[ yfrac ]
							+ xm10 * sintab1[ xfrac ] * sintab0[ yfrac ]
							+ xm20 * sintab2[ xfrac ] * sintab0[ yfrac ]
							+ xm30 * sintab3[ xfrac ] * sintab0[ yfrac ]
							+ xm01 * sintab0[ xfrac ] * sintab1[ yfrac ]
							+ xm11 * sintab1[ xfrac ] * sintab1[ yfrac ]
							+ xm21 * sintab2[ xfrac ] * sintab1[ yfrac ]
							+ xm31 * sintab3[ xfrac ] * sintab1[ yfrac ]
							+ xm02 * sintab0[ xfrac ] * sintab2[ yfrac ]
							+ xm12 * sintab1[ xfrac ] * sintab2[ yfrac ]
							+ xm22 * sintab2[ xfrac ] * sintab2[ yfrac ]
							+ xm32 * sintab3[ xfrac ] * sintab2[ yfrac ]
							+ xm03 * sintab0[ xfrac ] * sintab3[ yfrac ]
							+ xm13 * sintab1[ xfrac ] * sintab3[ yfrac ]
							+ xm23 * sintab2[ xfrac ] * sintab3[ yfrac ]
							+ xm33 * sintab3[ xfrac ] * sintab3[ yfrac ] ) );
						break;
					default:
						return( CCDLIB_WrongParameterError );
					}
				}
			}
		}
	}

	CCDLIB_Matrixfree( projmatrix );
	if( method == CCDLIB_Lanczos2Const )
	{
		free( sintab0 );
		free( sintab1 );
		free( sintab2 );
		free( sintab3 );
	}

	return( CCDLIB_Successful );
}


/************************************************************
*                                                           *
*     regression() berechnet ein Ausgleichspolynom bis zum  *
*     10. Grad mittels der Methode der kleinsten Fehler-    *
*     quadrate                                              *
*                                                           *
*     xwerte,ywerte = Zu aproximierende Werte (double *)    *
*     n     = Anzahl der Werte im Feld (long)               *
*     grad  = Grad des Polynoms (int, maximal 10)           *
*     koeff = Koeffizienten des Polynoms (double *)         *
*                                                           *
*     Version 1.0 vom 05.10.1989   (c) 1989                 *
*                                                           *
*     Quelle: Formelsammlung zur numerischen Mathematik     *
*     & Murray Spiegel, Statistik, Mc Graw-Hill             *
*                                                           *
************************************************************/

BOOL CCDLIB_Regression
(
   int grad,
   long n,
	double *xwerte,
   double *ywerte,
   double *koeff
)
{
   long i,j,k, *perm; int signdet;
   double **matrix, *b;

   if(grad>10) grad = 10; /* maximaler Polynomgrad 10 */

   if( CCDLIB_Matrixalloc( &matrix, grad+1 ) == FALSE )
      return( FALSE );

   b      = (double *)malloc((grad+1)*sizeof(double));
   perm   = (long *)malloc((grad+1)*sizeof(long));
   if( b==NULL || perm==NULL )
   {
      if( b != NULL ) free( b );
      if( perm != NULL ) free( perm );
      return(FALSE);
   }

   for(i=0;i<=grad;i++)
   {
      for(j=0;j<=grad;j++)
      {
         matrix[j][i] = 0.;
         for(k=0;k<n;k++)
         {
            matrix[j][i] += (CCDLIB_Ipow( xwerte[k], (i+j) ));
         }
      }
      b[i] = 0.;
      for(k=0;k<n;k++)
      {
         b[i] += ( ywerte[k] * CCDLIB_Ipow( xwerte[k], i ));
      }
   }

   if( 0!=CCDLIB_GaussSolve(
            0, grad+1, matrix, matrix, perm, b, koeff, &signdet))
      return(FALSE);

   CCDLIB_Matrixfree( matrix );
   free(b);
   free(perm);

   return(TRUE);
}

/************************************************************
*                                                           *
*     2D-Regression() berechnet eine polynomiale Aus-       *
*     gleichsflaeche an eine Flaeche aus Wertetripeln bis   *
*     zum 2. Grad mittels der Methode der kleinsten Fehler- *
*     quadrate                                              *
*                                                           *
*     xwerte, ywerte, zwerte = Zu aproximierende Werte      *
*     n     = Anzahl der Werte im Feld (long)               *
*     grad  = Grad des Polynoms (int, maximal 2)            *
*     koeff = Koeffizienten des Polynoms (double *)         *
*                                                           *
*     Version 1.0 vom 30. 8. 2000   (c) 1989 - 2000         *
*                                                           *
*     Quelle: Formelsammlung zur numerischen Mathematik     *
*     & Murray Spiegel, Statistik, Mc Graw-Hill             *
*                                                           *
************************************************************/

BOOL CCDLIB_2DRegression
(
   int grad,
   long n,
	double *xwerte,
   double *ywerte,
   double *zwerte,
   double *koeff
)
{
   long i, j, k, gradp, grad1,
        ix, iy, jx, jy, *perm;
   int signdet;
   double **matrix, *b, *saveptr;

   if(grad>4) grad = 4; /* maximaler Polynomgrad 2 */
   if(grad<1) grad = 1; /* minimaler Polynomgrad 1 */

   grad1 = grad+1;
   gradp = grad1*grad1;

   if( CCDLIB_Matrixalloc( &matrix, gradp ) == FALSE )
      return( FALSE );

   b      = (double *)malloc((gradp)*sizeof(double));
   perm   = (long *)malloc((gradp)*sizeof(long));
   if( b == NULL || perm == NULL )
   {
      if( b != NULL ) free( b );
      if( perm != NULL ) free( perm );
      return(FALSE);
   }
   saveptr = matrix[0];

   /* Erzeugt Polynom der Form  C + X + Y + X^2 + Y^2 + ... = Z */

   for( i=0; i<gradp; i++ )
   {
      ix = i%grad1;
      iy = i/grad1;
      for( j=0; j<gradp; j++ )
      {
         matrix[j][i] = 0.;
         jx = j%grad1;
         jy = j/grad1;
         for( k=0; k<n; k++ )
         {
            matrix[j][i] +=
               ( CCDLIB_Ipow( xwerte[k], (ix+jx) ) *
                 CCDLIB_Ipow( ywerte[k], (iy+jy) ) );
         }
      }
      b[i] = 0.;
      for( k=0; k<n; k++ )
      {
         b[i] += ( zwerte[k] *
            CCDLIB_Ipow( xwerte[k], ix )*CCDLIB_Ipow( ywerte[k], iy ) );
      }
   }

   CCDLIB_GaussSolve( 0, gradp, matrix, matrix, perm, b, koeff, &signdet);

   matrix[0] = saveptr;
   CCDLIB_Matrixfree( matrix );
   free(b);
   free(perm);

   return(TRUE);
}

double CCDLIB_Eval2DPolynom
(
   int grad,
	double x,
   double y,
   double *koeff
)
{
   double erg, xpow, ypow;
   long i, j, k;

   erg = 0;
   k   = 0;
   for( j=0, ypow=1; j<=grad; j++, ypow*=y )
      for( i=0, xpow=1.; i<=grad; i++, xpow*=x, k++ )
         erg += xpow * ypow * koeff[ k ];

   return( erg );
}

/**************************************************
*    Potenzieren mit Integerzahl                  *
**************************************************/

double CCDLIB_Ipow
(
   double arg,
   long exp
)
{
   double erg = 1.; long i;

   if( exp==0 ) return( 1. );

   if( exp>0 )
      for( i=0; i<exp; i++ )
         erg *= arg;
   else
      for( i=0; i<exp; i++ )
         erg /= arg;

   return( erg );
}

/***************************************************************
*                                                              *
*     Lsung linearer Gleichungssysteme mittels Gaussmethode   *
*                                                              *
*     Version 1.0 vom 5.10.1989 by Dittelsoft & Engeln-Mllges *
*                                                              *
*     int cas => 0 Zerlegungsmarix und Berechnung der Lsung   *
*                1 Nur Matrix zerlegen                         *
*                2 Nur Gleichungen lsen                       *
*                                                              *
*     long n  => Dimension der Matrix                          *
*     double *matrix[n] => Matrix als Vektor aus Zeigern       *
*     double *lumat[n]  => Zerlegungsmatrix                    *
*     (wenn lumat = matrix geht matrix verloren)               *
*     long perm[n] => Vektor der vertauschten Zeilen           *
*     double b[n] => rechte Seite des Gleichungssystems        *
*     int *signdet => Vorzeichen der Determinante              *
*                                                              *
*     double x[n] => Lsungsvektor des Systems                 *
*                                                              *
*     Fehlermeldungen : 0 = Alles OK                           *
*                       1 = n<2 oder falsche Parameter         *
*                       2 = out of memory                      *
*                       3 = Matrix ist singulr                *
*                       4 = Matrix ist zu schlecht kond.       *
*                                                              *
***************************************************************/

int CCDLIB_GaussSolve
(
   int cas,
   long n,
   double **matrix,
   double **lumat,
   long *perm,
   double *b,
   double *x,
   int *signdet )
{
   int res;

   if( n<2 ) return(1);

   switch(cas)
   {
   case 1 :
      return(CCDLIB_Gaudec( n, matrix, lumat, perm, signdet ));
   case 2 :
      return(CCDLIB_Gausol( n, lumat, perm, b, x));
   default :
      res = CCDLIB_Gaudec( n, matrix, lumat, perm, signdet );
      if(res==0) return( CCDLIB_Gausol( n, lumat, perm, b, x ) );
      else       return(res);
   }
}

/***************************************************************
*                                                              *
*     gaudec zerlegt eine Matrix in eine untere und eine obe-  *
*     re Dreiecksmatrix. Parameter wie bei gauss()             *
*     version 1.0 vom 05.10.1989                               *
*                                                              *
***************************************************************/

int CCDLIB_Gaudec
(
   long n,
   double **matrix,
   double **lumat,
   long *perm,
   int *signdet
)
{
   long j0,j,k,m; double piv, temp, *temp1, *d, zmax;

   d = (double *)malloc(n*sizeof(double));
   if(d==NULL) return(2);

   if(lumat != matrix)
   {
      for( k=0; k<n; k++ )
      {
         for( j=0; j<n; j++ )
            lumat[k][j] = matrix[k][j];
      }
   }

   for( k=0; k<n; k++ )
   {
      perm[k] = k;
      for( zmax=0., j=0; j<n; j++ )
      {
         if( zmax < fabs(lumat[k][j]) );
            zmax = fabs(lumat[k][j]);
      }
      if(zmax == 0.) return(3);
      d[k] = zmax;
   }
   *signdet = 1;

   for( k=0; k<n-1; k++ )
   {
      piv = fabs(lumat[k][k])/d[k];
      j0 = k;
      for( j=k+1; j<n; j++ )
      {
         temp = fabs(lumat[j][k])/d[j];
         if( piv<temp )
         {
            piv = temp;
            j0 = j;
         }
      }
      if( piv < 1e-15 )
      {
         *signdet = 0;
         return(4);
      }

      if( j0!=k )
      {
         *signdet = -*signdet;
         m = perm[j0]; perm[j0] = perm[k]; perm[k] = m;
         temp = d[j0]; d[j0] = d[k]; d[k] = temp;
         temp1 = lumat[j0]; lumat[j0] = lumat[k]; lumat[k] = temp1;
      }

      for( j=k+1; j<n; j++ )
      {
         if(lumat[j][k]!=0.)
         {
            lumat[j][k] /= lumat[k][k];
            for( temp=lumat[j][k],m=k+1; m<n; m++ )
               lumat[j][m] -= temp*lumat[k][m];
         }
      }
   }
   if( fabs( lumat[n-1][n-1] ) < 1e-15 )
   {
      *signdet = 0;
      return(4);
   }

   free(d);
   return(0);
}

/***************************************************************
*                                                              *
*     gausol berechnet matrix * x = b Version 1.0 05.10.1989   *
*                                                              *
***************************************************************/

int CCDLIB_Gausol
(
   long n,
   double **lumat,
   long *perm,
   double *b,
   double *x
)
{
   long j, k;
   double sum;

   x[0] = b[perm[0]];
   for( k=1; k<n; k++ )
   {
      for( x[k]=b[perm[k]], j=0; j<k; j++ )
         x[k] -= (lumat[k][j]*x[j]);
   }

   x[n-1] /= lumat[n-1][n-1];
   for( k=n-2; k>=0; k-- )
   {
      for( sum=0., j=k+1; j<n; j++ )
         sum += ( lumat[k][j]*x[j] );

      x[k] = ( x[k]-sum )/lumat[k][k];
   }
   return(0);
}



/*** Routine zur Erzeugung eines leeren Bildes ***/

CCDLIB_PictureparamStruct CCDLIB_Getnullpicture( void )
{
	CCDLIB_PictureparamStruct nullpicturestruct =
	{
		0L,                   		/* Breite des gesamten Bildpuffers in Pixel */
		0L,                 			/* Hoehe des gesamten Bildpuffers in Pixel */
		0x8000L,                   /* Umfang der moeglichen Graustufen */
		16,                  		/* Bits pro Pixel */
		1.0,                  		/* Verhaeltnis von Pixelbreite/Pixelhoehe (X/Y) */
		1.0,                  		/* Pixelbreite */
		1.0,                  		/* Pixelhoehe */
		CCDLIB_ShortPixels,      	/* Daten-Type eines Pixels */
		CCDLIB_BlackWhiteColorType,/* Zugehoerigkeit zu einem SW / Farbbild */
		CCDLIB_EmptyColor,      	/* Evtl. Farbkanal */
		CCDLIB_CartesianType,		/* Projektionsart der Bilddaten */
		0,                         /* Fuellwert fuer unsichtbare Pixel */
		0,                         /* Fuellwert fuer unsichtbare Pixel */
		0,                         /* Fuellwert fuer unsichtbare Pixel */
		1,                  			/* Grauskala */
		0,                  			/* Offset fuer Grau-Nullpunkt i.e. 0 */
		"Counts",              		/* Grauwertskaleneinheit */
		0.,               			/* Nullpunktkoordinate in Pixeln, bez. auf oben links */
		0.,               			/* Nullpunktkoordinate in Pixeln,,bez. auf oben links */
		1.,                  		/* X-Achsenskaleneinheit pro Pixel */
		1.,                  		/* Y-Achsenskaleneinheit pro Pixel */
		0.,                 			/* X-Achsenskaleneinheit beim Nullpunkt */
		0.,                 			/* Y-Achsenskaleneinheit beim Nullpunkt */
		"Pixel",              		/* Skaleneinheit der X-Achse */
		"Pixel",              		/* Skaleneinheit der Y-Achse */
		1,                  			/* Bildnummer des Bildes in einer Serie (1-n) */
		1                 			/* Laenge einer Aufnahmeserie */
	};

	return( nullpicturestruct );
}


