#ifndef __CCDLIB

#define __CCDLIB

/*** Defines ***/

#define CCDLIB_ConvolutionWidthConst 65

/*** Logic ***/

#ifndef BOOL
#define BOOL int
#endif

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*** math ***/

#define PI   3.14159265358979323846
#define PI2  PI*2.
#define PID2 PI/2.
#define PID3 PI/3.
#define PID4 PI/4.
#define PID6 PI/6.
#define PID12 PI/12.
#define PID36 PI/36.
#define PID23 PI/1.5
#define PID43 PI/0.75
#define PI3D2 PI*1.5

#define PI_GRAD PI/180.
#define PI_MIN  PI/10800.
#define PI_SEC  PI/648000.

#define PI_270GRAD PI3D2
#define PI_180GRAD PI
#define PI_90GRAD PID2
#define PI_15GRAD PID12
#define PI_5GRAD PID36


#define RAD (3.14159265358979323846/180.)
#define DEG (180./3.14159265358979323846)

#define fint(x) ((x)<0?(ceil(x)):(floor(x)))
#define frac(x) ((x)<0?((x)-ceil(x)):((x)-floor(x)))
#define sgn(x)  ((x)<0?(-1):(1))
#define fdiv(x,y) (fint((x)/(y))*(y))
#define deg(x)  ((x)*180./M_PI)

/*** string const ***/

#define CCDLIB_MAXFILENAMELEN 1024
#define CCDLIB_MAXPATHLEN     1024
#define CCDLIB_FITSSTRLEN     72
#define CCDLIB_MINIMALRIM     20

/***********************************************************
* definition of specific argument enums                    *
***********************************************************/

enum CCDLIB_NoiseTypeEnum
{
  CCDLIB_NoisePinkFlatConst,				/* Gaussverteiltes rosa Rauschen */
  CCDLIB_NoiseWhiteFlatConst,          /* Nur Weisses Rauschen */
  CCDLIB_NoiseDenseWhiteFlatConst,     /* Nur Weisses Rauschen grosser Amplitude */
  CCDLIB_NoiseWhiteConst,              /* Addiere weisses Rauschen */
  CCDLIB_NoiseDenseWhiteConst,         /* Addiere intensives weisses Rauschen */
  CCDLIB_NoiseBrightConst,             /* Fuege weisse Pixel hinzu */
  CCDLIB_NoiseDenseBrightConst,        /* Fuege viele weisse Pixel hinzu */
  CCDLIB_NoiseDarkConst,               /* Fuege schwarze Pixel hinzu */
  CCDLIB_NoiseDenseDarkConst,          /* Fuege viele schwarze Pixel hinzu */
  CCDLIB_NoiseMixedConst,              /* Fuege weisse und schwarze Pixel hinzu */
  CCDLIB_NoiseDenseMixedConst          /* Fuege viele weisse und schwarze Pixel hinzu */
};

enum CCDLIB_NoiseFilterEnum
{
  CCDLIB_Noise3x3MedianConst,          /* 3x3 Median */
  CCDLIB_Noise5x3MedianConst,          /* 5x3 Median */
  CCDLIB_Noise5x5MedianConst,          /* 5x5 Median */
  CCDLIB_Noise7x5MedianConst,          /* 7x5 Median */
  CCDLIB_Noise3x3MittelConst,          /* 3x3 Mittel */
  CCDLIB_Noise5x5MittelConst,          /* 5x5 Mittel */
  CCDLIB_Noise3x3SigmaConst,           /* 3x3 Sigma */
  CCDLIB_Noise5x5SigmaConst,           /* 3x3 Sigma */
  CCDLIB_NoiseZeilenMedianConst,       /* Median Zeilenkorrektur */
  CCDLIB_NoiseZeilenMittelConst,       /* Mittel Zeilenkorektur */
  CCDLIB_NoiseNoFilterConst				/* Keine Rauschfilterung */
};

enum CCDLIB_HistogramrangeEnum
{
  CCDLIB_HistoExaktConst,          		/* Exakt */
  CCDLIB_Histo99_8Const,           		/* 99.8 % */
  CCDLIB_Histo99_6Const,           		/* 99.6 % */
  CCDLIB_Histo99Const,          			/* 99 % */
  CCDLIB_Histo98Const,          			/* 98 % */
  CCDLIB_Histo96Const,          			/* 96 % */
  CCDLIB_Histo90Const,           		/* 90 % */
  CCDLIB_Histo80Const,           		/* 80 % */
};

enum CCDLIB_EnlargeMethodEnum
{
  CCDLIB_GetSampleConst,               /* Verkleinerung durch Undersamplen */
  CCDLIB_MeanOfVicinityConst,          /* Verkleinerung durch Mitteln */
  CCDLIB_NearestNeighborConst,         /* Naechster Nachbar-Kloetzelgrafik */
  CCDLIB_LinearIntpolConst,            /* Lineare Interpolation */
  CCDLIB_Lanczos2Const
};

enum CCDLIB_MoveMethodEnum
{
  CCDLIB_MoveUnchangedConst,         /* 1zu1 Transfer */
  CCDLIB_MoveCutoffConst,            /* Ausschnitt */
  CCDLIB_MoveShrink2Const,
  CCDLIB_MoveShrink4Const
};

enum CCDLIB_FlipMethodEnum
{
	CCDLIB_FlipTopdownConst,    /* flip top-bottom */
	CCDLIB_FlipMirrorConst,     /* flip left-right */
	CCDLIB_FlipErectConst,      /* 180 degree turn */
	CCDLIB_FlipLeftturnConst,   /* turn 90 degree left */
	CCDLIB_FlipRightturnConst   /* turn 90 degree right */
};


enum CCDLIB_MaskTypEnum
{
  CCDLIB_HiPassConst,
  CCDLIB_LoPassConst,
  CCDLIB_BandPassConst,
  CCDLIB_OwnPSFConst,
  CCDLIB_EffectFilterConst
};

enum CCDLIB_LowpassTypEnum
{
  CCDLIB_LPMedianConst,
  CCDLIB_LPMeanvalueConst,
  CCDLIB_LPLinemedianConst,
  CCDLIB_LPLinemeanConst,
  CCDLIB_LPGaussConst,
  CCDLIB_LPRectangleConst,
  CCDLIB_LPTriangleConst,
  CCDLIB_LPGaussRectConst,
  CCDLIB_LPFreeformConst
};

enum CCDLIB_HighpassTypEnum
{
  CCDLIB_HPGaussConst,
  CCDLIB_HPRectangleConst,
  CCDLIB_HPTriangleConst,
  CCDLIB_HPGaussRectConst,
  CCDLIB_HPFreeformConst
};

enum CCDLIB_HighpassCharEnum
{
  CCDLIB_HighCriticalConst,
  CCDLIB_HighBesselConst,
  CCDLIB_HighButterworthConst,
  CCDLIB_HighSquareConst
};

enum CCDLIB_EffektFiltTypEnum
{
  CCDLIB_NGradientConst,
  CCDLIB_NEGradientConst,
  CCDLIB_EGradientConst,
  CCDLIB_SEGradientConst,
  CCDLIB_SGradientConst,
  CCDLIB_SWGradientConst,
  CCDLIB_WGradientConst,
  CCDLIB_NWGradientConst,
  CCDLIB_SobelConst,
  CCDLIB_PhasefigureConst,
  CCDLIB_HoughTransConst
};

enum CCDLIB_BandpassAmpEnum
{
  CCDLIB_AmpGaussConst,
  CCDLIB_AmpRectangleConst,
  CCDLIB_AmpTriangleConst,
  CCDLIB_AmpGaussRectConst,
  CCDLIB_AmpFreeformConst
};

enum CCDLIB_BandpassFiltEnum
{
  CCDLIB_FiltGaussConst,
  CCDLIB_FiltRectangleConst,
  CCDLIB_FiltTriangleConst,
  CCDLIB_FiltGaussRectConst,
  CCDLIB_FiltFreeformConst,
  CCDLIB_NoBandpassConst
};

enum CCDLIB_BandpassCharEnum
{
  CCDLIB_BandCriticalConst,
  CCDLIB_BandBesselConst,
  CCDLIB_BandButterworthConst,
  CCDLIB_BandSquareConst
};

enum CCDLIB_OperationTypeEnum
{
  CCDLIB_SharpRectangleConst,          /* Entfaltung mit Rechteck */
  CCDLIB_SharpGaussConst,              /* Entfaltung mit Gauss */
  CCDLIB_SharpTriangleConst,           /* Entfaltung mit Kegel */
  CCDLIB_SharpGaussRectConst,          /* Entfaltung mit ueberlagertem Rechteck/Gauss */
  CCDLIB_MeanValueConst,               /* Mittelwertbildung in kreisfoermiger Umgebung */
  CCDLIB_MeanGaussConst,               /* Gaussmittelung in kreisfoermiger Umgebung */
  CCDLIB_MaskedMedianConst,            /* Echter Median in kreisfoermiger Umgebung */
  CCDLIB_LorentzMaskConst,
  CCDLIB_OpNGradientConst,
  CCDLIB_OpNEGradientConst,
  CCDLIB_OpEGradientConst,
  CCDLIB_OpSEGradientConst,
  CCDLIB_OpSGradientConst,
  CCDLIB_OpSWGradientConst,
  CCDLIB_OpWGradientConst,
  CCDLIB_OpNWGradientConst
};

enum CCDLIB_GradientOperationEnum
{
  CCDLIB_ExactConst,                   /* Echter langsamer Pythagoras */
  CCDLIB_AproximationConst             /* Schnelle Approximation mit max 4% Fehler */
};

enum CCDLIB_HistogramModifierEnum
{
  CCDLIB_LinearConst,                  /* Linear modification */
  CCDLIB_ExponentConst,                /* Exponential correction */
  CCDLIB_LogarithmusConst,             /* Logarithmic correction */
  CCDLIB_HistogramConst,               /* Histogram controlled correction */
  CCDLIB_HistoBoostConst,              /* Threshold controlled histogram */
  CCDLIB_GammaKorrekturConst,          /* Gamma correction */
  CCDLIB_S_SchlagConst                 /* S-curve */
};

enum CCDLIB_GainEnum
{
  CCDLIB_GainLinearConst,              /* Linear modification */
  CCDLIB_GainLogConst,                 /* Logarithmic correction */
  CCDLIB_Gain2LogConst,                /* Bilogarithmic correction */
  CCDLIB_GainGammaConst,          		/* Gamma correction */
};

enum CCDLIB_BackgroundHandlingEnum
{
  CCDLIB_ManualConst,                  /* Manuelle Korrektur */
  CCDLIB_AutomaticConst,               /* Automatische Korrektur */
  CCDLIB_NoCorrektionConst             /* Keine Korrektur */
};

enum CCDLIB_DarkCurrentEnum
{
  CCDLIB_DarkFileConst,                /* Korrektur per Datei */
  CCDLIB_ConstantConst,                /* Korrektur per Konstante */
  CCDLIB_PolynomConst                  /* Korrektur per Polynom */
};

enum CCDLIB_ShiftTypeEnum
{
	CCDLIB_RotConstAngleConst,   			/* Rotation um einen konst. Win. */
	CCDLIB_RotConstTangentConst, 			/* Rot. mit konst. Tangente */
	CCDLIB_ShiftConstRadConst   			/* Shift mit konst. Radius */
};

enum CCDLIB_PeakHeightCriterionEnum
{
  CCDLIB_9Div10OfRangeConst,
  CCDLIB_4Div5OfRangeConst,
  CCDLIB_3Div4OfRangeConst,
  CCDLIB_2Div3OfRangeConst,
  CCDLIB_HalfOfRangeConst,
  CCDLIB_1Div3OfRangeConst,
  CCDLIB_1Div4OfRangeConst,
  CCDLIB_1Div5OfRangeConst,
  CCDLIB_1Div10OfRangeConst
};

enum CCDLIB_PeakWidthCriterionEnum
{
  CCDLIB_NeighborValueConst,
  CCDLIB_ExtremlyNarrowConst,
  CCDLIB_VeryNarrowConst,
  CCDLIB_NarrowConst,
  CCDLIB_NormalConst,
  CCDLIB_BroadConst,
  CCDLIB_VeryBroadConst,
  CCDLIB_ExtremlyBroadConst,
  CCDLIB_HalfSpectrumConst
};


enum CCDLIB_PixelTypeEnum
{
  CCDLIB_UnsignedCharPixels  =   8,    /* FITS-Code fuer unsigned char */
  CCDLIB_ShortPixels         =  16,    /* FITS-Code fuer short */
  CCDLIB_UnsignedShortPixels =  17,    /* Code fuer unsigned short (kein FITS) */
  CCDLIB_LongPixels          =  32,    /* FITS-Code fuer long */
  CCDLIB_FloatPixels         = -32,    /* FITS-Code fuer 4 Byte floats */
  CCDLIB_DoublePixels        = -64     /* FITS-Code fuer 8 Byte doubles */
};

enum CCDLIB_ColorTypeEnum
{
  CCDLIB_BlackWhiteColorType,          /* Schwarz-Weiss-Bild */
  CCDLIB_RGBColorType,                 /* Bild im RGB-Format */
  CCDLIB_HSIColorType,                 /* Bild im Hue/Saturation/Intensity-Format */
  CCDLIB_YUVColorType,                 /* Bild im Luminanz (Y)/Chrominanz (U,V) Format */
  CCDLIB_NumvaluesNoColorType          /* Daten sind keine Bilddaten */
};

enum CCDLIB_ColorEnum
{
  CCDLIB_BlackWhiteColor,              /* Schwarz-Weiss */
  CCDLIB_RGBChannelColor,              /* RGB-Farbbild */
  CCDLIB_RedColor,                     /* Rotanteil */
  CCDLIB_GreenColor,                   /* Gruenanteil */
  CCDLIB_BlueColor,                    /* Blauanteil */
  CCDLIB_HueColor,                     /* Farbtoenung */
  CCDLIB_SaturationColor,              /* Farbsaettigung */
  CCDLIB_IntensityColor,               /* Intensitaet */
  CCDLIB_LuminanceColor,               /* Luminanzsignal */
  CCDLIB_U_ChrominanceColor,           /* Chrominanz U-Vektor */
  CCDLIB_V_ChrominanceColor,           /* Chrominanz V-Vektor */
  CCDLIB_PhaseColor,                   /* Phasenbild */
  CCDLIB_NumvaluesNoColor,             /* Daten sind keine Bilddaten */
  CCDLIB_EmptyColor                    /* Bild hat keinen Inhalt */
};

enum CCDLIB_ProjectionEnum
{
  CCDLIB_ConicalType,                  /* Kegelprojektion */
  CCDLIB_GnomonicType,                 /* Gnomonische Projektion */
  CCDLIB_PolarCoordType,               /* Polarkoordinaten */
  CCDLIB_StereographicType,            /* Stereografische Projektion */
  CCDLIB_OrthogonalType,               /* Flaechentreue Projektion */
  CCDLIB_WernerType,                   /* Werner-Projektion */
  CCDLIB_WinkelType,                   /* Winkel-Projektion */
  CCDLIB_MollweideType,                /* Weltkarte nach Mollweide */
  CCDLIB_ZylinderType,                 /* Einfache Plattkarte */
  CCDLIB_MercatorType,                 /* Mercator-Projektion */
  CCDLIB_PolarAzimutalType,            /* Polarkarten in Azimutprojektion */
  CCDLIB_PolarStereoType,              /* Polarkarten in Stereografischer Projektion */
  CCDLIB_CartesianType,                /* Kartesische Koordinaten (Normalfall) */
};

enum CCDLIB_SeeingEnum
{
  CCDLIB_ExcellentSeeing,              /* Seeing 1 */
  CCDLIB_FairSeeing,                   /* Seeing 2 */
  CCDLIB_ModerateSeeing,               /* Seeing 3 */
  CCDLIB_PoorSeeing,                   /* Seeing 4 */
  CCDLIB_BadSeeing,                    /* Seeing 5 */
  CCDLIB_UselessToObserve,             /* Seeing 6 */
};

enum CCDLIB_PictureTypeEnum
{
  CCDLIB_SingleRawPicture,             /* Einzelnes Rohbild */
  CCDLIB_SeriesRawPicture,             /* Rohbild in einer Serie */
  CCDLIB_ProcessedPicture,             /* Verarbeitetes Bild */
  CCDLIB_FalseColorPicture,            /* Falschfarbenbild */
  CCDLIB_WorldmapPicture,              /* Karte */
  CCDLIB_AssembledPicture,             /* Mosaikbild */
  CCDLIB_NoPicture                     /* Inhalt keine Bilddaten */
};

enum CCDLIB_AequidensitenEnum
{
	CCDLIB_None,                        /* Keine Aequidensiten */
	CCDLIB_BlackOnWhite,                /* Schwarze Linien auf weissem Grund */
	CCDLIB_WhiteOnBlack,                /* Weisse Linien auf schwarzem Grund */
	CCDLIB_WhiteOnDarkened,             /* Weisse Linien auf gedunkeltem Bild */
	CCDLIB_Invers,                      /* Linien sind zum Bild invers */
	CCDLIB_ColorOnPicture               /* Farbige Linien auf dem Bild */
};

enum CCDLIB_FitMethodEnum
{
	CCDLIB_Rigid,                       /* Starre Ueberlagerung */
	CCDLIB_BrightestPoint,              /* Hellster Punkt in der Bildmitte */
	CCDLIB_CenteredDisk,                /* Zentrierung auf helle Scheibe */
   CCDLIB_FitPattern,                  /* Mit einzulernendem Passmuster */
   CCDLIB_NeighbourFit,                /* Passmustersuche in Umgebung */
	CCDLIB_DerotFit,                    /* Zwei Passmuster mit Derotation */
	CCDLIB_CrossCorrelation,            /* ganzflchige Kreuzkorrelation */
	CCDLIB_DerotCrossCorr,              /* Kreuzkorrelation mit Derotation */
	CCDLIB_Coordinates,                 /* manuelle Koordinatenangabe */
	CCDLIB_Manual                       /* Ueberlagern von Hand */
};

enum CCDLIB_ColorSeparationEnum
{
	CCDLIB_SeparRGBConst,               /* Zerlegung in drei Farbkanle */
	CCDLIB_SeparRedConst,               /* Rotauszug */
	CCDLIB_SeparGreenConst,             /* Grnauszug */
	CCDLIB_SeparBlueConst,              /* Blauauszug */
	CCDLIB_SeparLuminanceConst          /* Luminanz nach IEC */
};

enum CCDLIB_ColorCombineEnum
{
	CCDLIB_CombineLRGBConst,            /* Kombiniere drei Farbkanle plus SW-Bild */
	CCDLIB_CombineRGBConst,             /* Kombiniere drei Farbkanle */
	CCDLIB_BWAnaglyphConst,             /* schwarzweies Anaglyphenbild */
	CCDLIB_ColorAnaglyphConst           /* Farbiges Anaglyphenbild */
};

enum CCDLIB_RetrievePSFEnum
{
	CCDLIB_PSFStarConst,                /* PSF aus Sternbild */
	CCDLIB_PSFDiskConst,                /* PSF aus runder Kante */
	CCDLIB_PSFLimbConst                 /* PSF aus gerader Kante */
};

enum CCDLIB_FrametypeEnum
{
	CCDLIB_FullframeConst,              /* Vollbild */
	CCDLIB_OddPicConst,                 /* Ungerade Zeilen */
	CCDLIB_EvenPicConst                 /* Gerade Zeilen */
};

enum CCDLIB_FITSColorEnum
{
	CCDLIB_RGBConst,                    /* RGB-Folge */
	CCDLIB_BGRConst,                    /* Folge BGR */
   CCDLIB_SeparateConst                /* In Einzelbildern */
};

enum CCDLIB_FITSFrameEnum
{
	CCDLIB_ColorTripleConst,             /* Farb-Tripel */
	CCDLIB_ColorFramesConst              /* Farb-Frames */
};

enum CCDLIB_FITSBzeroEnum
{
   CCDLIB_FITSNozeroConst,              /* no offset */
	CCDLIB_FITSCharConst,                /* 128 */
	CCDLIB_FITSShortConst,               /* 32k */
	CCDLIB_FITSLongConst                 /* 2G */
};

enum CCDLIB_OrientationEnum
{
   CCDLIB_Horizontal,
   CCDLIB_Vertical
};

enum CCDLIB_EqualizeEnum
{
   CCDLIB_NoChangeConst,
   CCDLIB_SubBackConst
};

enum CCDLIB_AddImagetypeEnum
{
	CCDLIB_AddAverage,
   CCDLIB_AddAccumulate,
   CCDLIB_AddExposure
};


/***********************************************************
* definition of specific data types                        *
***********************************************************/

typedef struct
{
  short day;                           /* Tag 1 - 28/31 */
  short month;                         /* Monat 1 - 12 */
  short year;                          /* Jahr 4 stellig */
  short hour;                          /* Stunde 0 - 23 */
  short min;                           /* Minute 0 - 59 */
  short sec;                           /* Sekunde 0 - 59 */
  short msec;                          /* Millisekunden 0 - 999 */
  short timezone;                      /* Zeitzone -12 bis 12 bez. UT ( 0 = UT ) */
  char  name[12];                      /* Tagesname */
  double juldat;                       /* Vollstaendiges julianisches Datum auf 15 Stellen */
}
CCDLIB_TimeStruct;


// gradient values for pictures
typedef union
{
  long ramword;                        /* 32-Bit Wert im Ram */
  struct
  {
	 short xgrad;                       /* Horizontaler Vektoranteil */
	 short ygrad;                       /* Vertikaler Vektoranteil */
  } value;
}
CCDLIB_GradientUnion;

typedef struct
{
  long xpos;                           /* X-Koordinate linke obere Ecke */
  long ypos;                           /* Y-Koordinate linke obere Ecke */
  long xsize;                          /* Breite des Rechtecks in Pixel */
  long ysize;                          /* Hoehe des Rechtecks in Pixel */
}
CCDLIB_RectangleStruct;

typedef struct
{
  long xpos;                           /* X-Koordinate linke obere Ecke */
  long ypos;                           /* Y-Koordinate linke obere Ecke */
}
CCDLIB_PositionStruct;

typedef struct
{
  long xsize;                          /* Breite des Rechtecks in Pixel */
  long ysize;                          /* Hoehe des Rechtecks in Pixel */
}
CCDLIB_RectsizeStruct;


/**********************************************************
* definition of physical image description                *
***********************************************************/

typedef struct
{
  long   linelength;                   /* Breite des gesamten Bildpuffers in Pixel */
  long   columnheight;                 /* Hoehe des gesamten Bildpuffers in Pixel */
  long   dynamik;                      /* Umfang der moeglichen Graustufen */
  short  bitsperpixel;                 /* Bits pro Pixel */
  double aspectratio;                  /* Verhaeltnis von Pixelbreite/Pixelhoehe (X/Y) */
  double xpixelsize;                   /* Breite eines Pixel in Mikrometer */
  double ypixelsize;                   /* Breite eines Pixel in Mikrometer */
  enum CCDLIB_PixelTypeEnum pixeltype;      /* Daten-Type eines Pixels */
  enum CCDLIB_ColorTypeEnum colortype;      /* Zugehoerigkeit zu einem SW / Farbbild */
  enum CCDLIB_ColorEnum     colorchannel;   /* Evtl. Farbkanal */
  enum CCDLIB_ProjecionEnum projectionstyle;/* Projektionsart der Bilddaten */
  union
  {
	 unsigned char  ubyte;
	 short          word;
	 unsigned short uword;
	 long           dword;
	 float          sfloat;
	 double         lfloat;
  } blankvalue;                        /* Grau/Rotwert fuer nicht besetzte Bildpunkte */
  union
  {
	 unsigned char  ubyte;
	 short          word;
	 unsigned short uword;
	 long           dword;
	 float          sfloat;
	 double         lfloat;
  } gblankvalue;                       /* Gruenwert fuer nicht besetzte Bildpunkte */
  union
  {
	 unsigned char  ubyte;
	 short          word;
	 unsigned short uword;
	 long           dword;
	 float          sfloat;
	 double         lfloat;
  } bblankvalue;                       /* Blauwert fuer nicht besetzte Bildpunkte */
  union
  {
	 unsigned char  ubyte;
	 short          word;
	 unsigned short uword;
	 long           dword;
	 float          sfloat;
	 double         lfloat;
  } greyscale;                        /* Grauwertscala in Units pro Graustufe */
  union
  {
	 unsigned char  ubyte;
	 short          word;
	 unsigned short uword;
	 long           dword;
	 float          sfloat;
	 double         lfloat;
  } greyoffset;                        /* Offset fuer Grau-Nullpunkt i.e. 0 */
  char   pixelunit[ 24 ];              /* Grauwertskaleneinheit */
  double xoriginofscale;               /* Nullpunktkoordinate in Pixeln, bez. auf oben links */
  double yoriginofscale;               /* Nullpunktkoordinate in Pixeln,,bez. auf oben links */
  double xscaledelta;                  /* X-Achsenskaleneinheit pro Pixel */
  double yscaledelta;                  /* Y-Achsenskaleneinheit pro Pixel */
  double xvalatorigin;                 /* X-Achsenskaleneinheit beim Nullpunkt */
  double yvalatorigin;                 /* Y-Achsenskaleneinheit beim Nullpunkt */
  char   xaxisunit[ 24 ];              /* Skaleneinheit der X-Achse */
  char   yaxisunit[ 24 ];              /* Skaleneinheit der Y-Achse */
  long   serialcount;                  /* Bildnummer des Bildes in einer Serie (1-n) */
  long   serieslength;                 /* Laenge einer Aufnahmeserie */
}
CCDLIB_PictureparamStruct;

/***********************************************************
* definition of image contents description                 *
***********************************************************/

typedef struct
{
  CCDLIB_TimeStruct observationtime;   /* Zeit und Datum der Aufnahme */
  CCDLIB_TimeStruct filetime;          /* Zeit und Datum der Dateierstellung */
  enum CCDLIB_PictureTypeEnum picturetype;  /* Art des Bildinhalts */
  long serialcount;                    /* Bildnummer des Bildes in einer Serie (1-n) */
  long serieslength;                   /* Laenge einer Aufnahmeserie */
  char observer[ CCDLIB_FITSSTRLEN ];                 /* Beobachter */
  char instrument[ CCDLIB_FITSSTRLEN ];               /* Aufnahmeinstrument */
  char telescope[ CCDLIB_FITSSTRLEN ];                /* Aufnahmeoptik */
  char object[ CCDLIB_FITSSTRLEN ];                   /* Beobachtetes Objekt */
  char dataorigin[ CCDLIB_FITSSTRLEN ];               /* System, dass die Daten erzeugt */
  char weathercomment[ CCDLIB_FITSSTRLEN ];           /* Kommentar zum Wetter */
  char seeingcomment[ CCDLIB_FITSSTRLEN ];            /* Kommentar zum Seeing */
  enum CCDLIB_SeeingEnum seeing;       /* Seeing Klassifikation */
  double integrationtime;              /* Belichtungszeit in Sekunden */
  long integrationcount;               /* Anzahl ueberlagerter Bilder (i.e. 1) */
  char picturename[ CCDLIB_MAXFILENAMELEN ];
  char filepfad[ CCDLIB_MAXPATHLEN ];
}
CCDLIB_PictureDataStruct;
/**********************************************************
* error handling                                          *
***********************************************************/

/* error symbols */
enum CCDLIB_ErrorEnum
{
	 CCDLIB_Successful,                  /* Operation war erfolgreich */
	 CCDLIB_WrongParameterError,         /* Unsinniger Parameter wurde uebergeben */
	 CCDLIB_WrongRectangleValuesError,   /* Unmoeglicher Bildausschnitt */
	 CCDLIB_NoPointerError,              /* Unbekannter Pointer auf Datenfeld */
	 CCDLIB_NoPicturePointerError,       /* Unbekannter Pointer auf Bilddaten */
	 CCDLIB_WrongPixelTypeError,         /* Nicht erlaubter Datentyp der Pixel */
	 CCDLIB_OutOfMemoryError,            /* Nicht genug Speicherplatz */
	 CCDLIB_FileOpenError,               /* Datei ist nicht zu oeffnen */
	 CCDLIB_WrongFileFormatError,        /* Datei hat ein unbekanntes Format */
	 CCDLIB_NoFITSFileError,             /* Kein SIMPLE FITS Format */
	 CCDLIB_NoOESFileError,              /* Kein OES Astro Format */
	 CCDLIB_NoBMPFileError,              /* Kein Windoof BMP Format */
	 CCDLIB_NoFLMFileError,              /* Kein Screen Maschine FLM Format */
	 CCDLIB_NoST4FileError,              /* Kein SBIG ST4 Format */
	 CCDLIB_NoST6FileError,              /* Kein SBIG ST6 Format */
	 CCDLIB_NoStarliteXPressFileError,   /* Kein Starlite XPress Format */
	 CCDLIB_NoFITSExtensionsError,       /* Keine FITS XTension unterstuetzt */
	 CCDLIB_TooMuchDimensionsError,      /* Mehr als vier FITS Dimensionen */
	 CCDLIB_WriteOnFileError,            /* Schreiben auf Datei missglueckt */
	 CCDLIB_ReadFromFileError            /* Lesen von Datei missglueckt */
};


/**********************************************************
* function prototypes                                     *
**********************************************************/


enum CCDLIB_ErrorEnum CCDLIB_CopyRegion(
  short * destbuffer,
  short * sourcebuffer,
  CCDLIB_PictureparamStruct * picpar,
  CCDLIB_RectangleStruct * copyrect );

void CCDLIB_InitRectangle(
	CCDLIB_RectangleStruct * picturerect,
	long xpos,
	long ypos,
	long xsize,
	long ysize );

enum CCDLIB_ErrorEnum CCDLIB_FlipMirror
(
	enum CCDLIB_FlipMethodEnum flipmethod,
	CCDLIB_PictureparamStruct * sourcepicparam,
	CCDLIB_PictureparamStruct * destpicparam,
	short * sourcepic,
	short * destpic );

enum CCDLIB_ErrorEnum CCDLIB_AddPictures
(
	CCDLIB_RectangleStruct * sourcepicrect,
	CCDLIB_PositionStruct  * destpicpos,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_PictureparamStruct * addparam,
	long ** addpuffer,
	long ** addcount,
	long piccount,
	BOOL constarea,
   enum CCDLIB_AddImagetypeEnum imagetype );

enum CCDLIB_ErrorEnum CCDLIB_MoveAddfieldToPicture
(
	CCDLIB_PictureparamStruct * addpicparam,
	CCDLIB_PictureparamStruct * destpicparam,
	short * destpic,
	long * addpuffer,
	long * addcount,
   enum CCDLIB_MoveMethodEnum movemethod,
   CCDLIB_RectangleStruct *cutrect );

enum CCDLIB_ErrorEnum CCDLIB_QuickEnlarge
(
	CCDLIB_PictureparamStruct * picparam,
	short * sourcepic,
	short * destpic,
	long enlargefac );

enum CCDLIB_ErrorEnum CCDLIB_GetPictureCorr
(
   double *correlation,
	short * picture,
	short * reference,
	CCDLIB_PictureparamStruct * pictureparam,
   BOOL piclowpass,
	CCDLIB_PositionStruct *picshift,
   CCDLIB_RectangleStruct *evalrect );

enum CCDLIB_ErrorEnum CCDLIB_ExtractOddEven
(
	short *oddpic,
	short *evenpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct * pictureparam );

enum CCDLIB_ErrorEnum CCDLIB_ExtractInterlace
(
	short *extractpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct * pictureparam,
   enum CCDLIB_FrametypeEnum frametype );

enum CCDLIB_ErrorEnum CCDLIB_GetCenterOfDisk(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_PositionStruct *centercoord,
   double *matchvalue
);

long CCDLIB_GetMeanSample(
	long xcoord,
	long ycoord,
	long picwidth,
	long picheight,
	enum CCDLIB_ColorChannelEnum colorchannel,
	long samplesize,
	short *picture );

enum CCDLIB_ErrorEnum CCDLIB_GetCenterOfBrightness
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
   double threshold,
	short * picture,
	CCDLIB_PositionStruct  *centercoord
);

enum CCDLIB_ErrorEnum CCDLIB_GetMinDifference
(
   CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture1,
	short * picture2,
	CCDLIB_PositionStruct *centercoord
);

enum CCDLIB_ErrorEnum CCDLIB_2DCrossCorrelation
(
   CCDLIB_RectsizeStruct  * fitsize,
   long subpixelfak,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
   short * fitpuffer,
	CCDLIB_PositionStruct  *centercoord,
   double * correlation
);

enum CCDLIB_ErrorEnum CCDLIB_2DQuickCorrelation
(
	CCDLIB_RectangleStruct *fitsquare,
   long searchradius,
   long subpixelfak,
	CCDLIB_PictureparamStruct *pictureparam,
	short *picture,
	short *fitpuffer,
	CCDLIB_PositionStruct *centercoord,
	double *correlation
);

void CCDLIB_QuickLowPassFilter(
   short *tempbuf,
   short *picbuffer,
   CCDLIB_PictureparamStruct *picparam,
   CCDLIB_RectangleStruct *checkrect,
   long width
);

BOOL CCDLIB_GetCenterMajority(
   short *buffer,
   CCDLIB_PictureparamStruct *picparam,
   long fwhmlevel,
   long samplecoord,
   long samplenum,
   long sampledist,
   enum CCDLIB_OrientationEnum sampledir );
   
enum CCDLIB_ErrorEnum CCDLIB_SetNoiseInPicture
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  enum CCDLIB_NoiseTypeEnum noisetype
);

short CCDLIB_GaussDevRandom
(
  short mean,
  short dynamik
);

enum CCDLIB_ErrorEnum CCDLIB_RemoveStars(
	short *destpic,
	short *sourcepic,
	CCDLIB_PictureparamStruct *picparam,
	double masklevel );

enum CCDLIB_ErrorEnum CCDLIB_RGBCorrection
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   long bluexshift,
   long blueyshift,
   long bluesizedif,
   long redxshift,
   long redyshift,
   long redsizedif );

enum CCDLIB_ErrorEnum CCDLIB_ColorCorrection
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   long redcorr,
   long greencorr,
   long bluecorr );

enum CCDLIB_ErrorEnum CCDLIB_SeparatedMaskedFilterOp
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype,
   void (* workmsg)()
);

enum CCDLIB_ErrorEnum CCDLIB_GetSeparableMask(
	double *mask,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype
);

enum CCDLIB_ErrorEnum CCDLIB_MaskedFilterOperation
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
   BOOL adaptiv,
   long astigmatism,
   double astigvector,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype,
   void (* workmsg)()
);

short CCDLIB_GetMedianFromArray
(
  short *values_pd,
  long valuenumber_n
);

enum CCDLIB_ErrorEnum CCDLIB_GetNotSeparableMask(
	double **mask,
	long maskdiameter,
	long rectdiameter,
	long filtdiameter,
	double effiency,
   long astigmatism,
   double astigvector,
	enum CCDLIB_OperationTypeEnum operationtype,
	enum CCDLIB_BandpassFiltEnum filtertype
);

enum CCDLIB_ErrorEnum CCDLIB_LineFilterOperation
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	unsigned long maskdiameter,
	enum CCDLIB_OperationTypeEnum operationtype
);

enum CCDLIB_ErrorEnum CCDLIB_RemoveHotpixel
(
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	CCDLIB_PositionStruct *hotpixelpos,
   long hotpixelnum
);

enum CCDLIB_ErrorEnum CCDLIB_NoisefilterFunction
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	double sigmalimit,
	enum CCDLIB_NoiseFilterEnum filtertype,
	int pseudoflag
);

enum CCDLIB_ErrorEnum CCDLIB_GetSeparableNoiseFilterMask
(
	long filterwidth,
	long filterheight,
	enum CCDLIB_NoiseFilterEnum filtertype,
	double *conv1Dh,
	double *conv1Dv
);

enum CCDLIB_ErrorEnum CCDLIB_GetNotSeparableNoiseFilterMask
(
	long filterwidth,
	long filterheight,
	enum CCDLIB_NoiseFilterEnum filtertype,
	double **conv2D
);

short CCDLIB_RemoveGreaterThanSigmaFromArray
(
	short *sortarray,
	short greyvalue,
	long arraylength,
	double sigmalimit
);

enum CCDLIB_ErrorEnum CCDLIB_CountBrightsteps
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	long * greycount,
	long * redcount,
	long * greencount,
	long * bluecount,
   long * nullcount,
   long * negcount,
   long * greymax
);

enum CCDLIB_ErrorEnum CCDLIB_RetrieveHistogram
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  long * histogramfield
);

enum CCDLIB_ErrorEnum CCDLIB_GetMaximumInHistogram
(
  long * histogramfield,
  long dynamik,
  long * lowestposition,
  long * highestposition,
  long * maximumposition,
  long * maximalvalue,
  double * maximumpart,
  double * entropy
);

enum CCDLIB_ErrorEnum CCDLIB_EvaluateHistogram
(
  long * histogramfield,
  long dynamik,
  enum CCDLIB_HistogramrangeEnum interval,
  long hotpixelnum,
  long deadpixelnum,
  long * lowestposition,
  long * highestposition,
  long * lowposition,
  long * highposition,
  long * maximumposition,
  long * maximalvalue,
  double * maximumpart,
  double * entropy
);

enum CCDLIB_ErrorEnum CCDLIB_GetLocalStatistics
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  double * mean,
  double * sigma
);

enum CCDLIB_ErrorEnum CCDLIB_GetNthMaximumInHistogram
(
  long * histogramfield,
  long dynamik,
  short maximumnumber,
  enum CCDLIB_PeakHeightCriterionEnum peakheightcriterion,
  enum CCDLIB_PeakWidthCriterionEnum peakwidthcriterion,
  long * maximumposition,
  long * maximalvalue,
  double * maximumpart
);                             

enum CCDLIB_ErrorEnum CCDLIB_ExtractLuminance
(
  short * sourcepicture,
  CCDLIB_PictureparamStruct * pictureparam,
  short * destpicture
);

enum CCDLIB_ErrorEnum CCDLIB_SeparateColors
(
  short * sourcepicture,
  CCDLIB_PictureparamStruct * pictureparam,
  enum CCDLIB_ColorSeparationEnum separmethod,
  short * channel1,
  short * channel2,
  short * channel3
);

enum CCDLIB_ErrorEnum CCDLIB_CombineColors
(
   short * destpic,
   CCDLIB_PictureparamStruct *pictureparam,
   enum CCDLIB_ColorCombineEnum combinemethod,
   short * redpic,
   short * greenpic,
   short * bluepic
);

enum CCDLIB_ErrorEnum CCDLIB_DarkframeCorrection
(
  short * destpicture,
  short * sourcepicture,
  short * darkframe,
  CCDLIB_PictureparamStruct * pictureparam
);

enum CCDLIB_ErrorEnum CCDLIB_FlatfieldCorrection
(
  short * destpicture,
  short * sourcepicture,
  short * flatfield,
  CCDLIB_PictureparamStruct * pictureparam
);

enum CCDLIB_ErrorEnum CCDLIB_SubtractBackground
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic
);

enum CCDLIB_ErrorEnum CCDLIB_Equalize
(
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
   enum CCDLIB_GainEnum gaintype,
   double gamma,
   enum CCDLIB_EqualizeEnum subtracttype
);

enum CCDLIB_ErrorEnum CCDLIB_QuickEqualize
(
   short *picbuffer,
   CCDLIB_PictureparamStruct *pictureparam
);

enum CCDLIB_ErrorEnum CCDLIB_ModifyHistogram
(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	short * histogramLUT );

enum CCDLIB_ErrorEnum CCDLIB_CalcHistogramLUT
(
	short * histogramLUT,
	long dynamik,
	enum CCDLIB_HistogramModifierEnum modifier,
	double gamma,
	long   wendepunkt,
	double highgamma,
	double lowgamma,
   long boostlevel,
	BOOL equalize,
   double equfactor,
	enum CCDLIB_HistogramrangeEnum interval,
   long backgroundlevel,
	long * histogramfield );

enum CCDLIB_ErrorEnum CCDLIB_SubtractShiftDifference
(
	CCDLIB_PictureparamStruct * pictureparam,
	short * sourcepic,
	short * destpic,
	long xcenter,
	long ycenter,
	double shiftval,
	double subintensity,
	enum CCDLIB_ShiftTypeEnum shifttype
);

enum CCDLIB_ErrorEnum CCDLIB_CalcXYGradientFromPic
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	CCDLIB_GradientUnion * gradient
);

enum CCDLIB_ErrorEnum CCDLIB_XYGradientFromPic
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  CCDLIB_GradientUnion * gradient
);

enum CCDLIB_ErrorEnum CCDLIB_CalcAbsGradientFromPic
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  CCDLIB_GradientUnion * gradient,
  enum CCDLIB_GradientOperationEnum operationtype
);

enum CCDLIB_ErrorEnum CCDLIB_CalcSobel
(
   short *sourcepic,
   short *destpic,
   CCDLIB_PictureparamStruct *pictureparam
);

enum CCDLIB_ErrorEnum CCDLIB_CalcPhase
(
   short *sourcepic,
   short *destpic,
   CCDLIB_PictureparamStruct *pictureparam
);

enum CCDLIB_ErrorEnum CCDLIB_BinarizePicture
(
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam,
  short * picture,
  short treshold,
  short logreyval,
  short higreyval
);

enum CCDLIB_ErrorEnum CCDLIB_MaximizeSaturation
(
	short * sourcepic,
	short * destpic,
	CCDLIB_PictureparamStruct * pictureparam,
   double exponent
);

BOOL CCDLIB_CheckRectangleValues
( 
  CCDLIB_RectangleStruct * picturerect,
  CCDLIB_PictureparamStruct * pictureparam
);

/* Rotieren und Skalieren */

enum CCDLIB_ErrorEnum CCDLIB_ScaleAndRotateRegion
(
	short * sourcepicture,
	CCDLIB_PictureparamStruct * sourcepicpar,
   double srcregionxpos,
   double srcregionypos,
   double srcregionxsize,
   double srcregionysize,
	double srcregturnxpos,
	double srcregturnypos,
	double srcregionrotangle,
	double xmagnification,
	double ymagnification,
	short * destpicture,
	CCDLIB_PictureparamStruct * destpicpar,
   long destregturnxpos,
	long destregturnypos,
	enum CCDLIB_EnlargeMethodEnum method
);

/* Perspektivtransformationen */

BOOL CCDLIB_GetProjectionPoint(
	double *xp,
	double *yp,
	double xobj,
	double yobj,
	double zobj,
	double **projmatrix,
	long   linelength,
	long   columnheight,
	long   srclength,
	long   srcheight,
	double limitwidth );

void CCDLIB_GetMatchingScale(
	CCDLIB_RectangleStruct * matchingrect,
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam );

enum CCDLIB_ErrorEnum CCDLIB_CalcRotScalematrix(
	double **projmatrix,
	double rotation,
	double magnification,
	double xpixelsize,
	double ypixelsize );

enum CCDLIB_ErrorEnum CCDLIB_CalcRotScale(
	CCDLIB_PictureparamStruct * sourcepicpar,
	CCDLIB_PictureparamStruct * destpicpar,
	short *sourcepic,
	short *destpic,
	long turnxpos,
	long turnypos,
	double rotation,
	double magnification,
	enum CCDLIB_EnlargeMethodEnum method );

/*** Hilfsfunktionen zur Matrixmathematik ***/

BOOL CCDLIB_Matrixalloc(
	double ***matrix,
	int size );

void CCDLIB_Matrixfree(
	double **matrix );

void CCDLIB_Matrixcopy(
	double **destmat,
	double **srcmat,
	int    size );

void CCDLIB_Matrixmul(
   double **result,
	double **mat1,
   double **mat2,
	int    size );

void CCDLIB_Matrixvectormul(
	double *result,
	double **matrix,
	double *vector,
	int size );

void CCDLIB_FastMatVecMul(
	double *result,
	double **matrix,
	double *vector );

CCDLIB_PictureparamStruct CCDLIB_Getnullpicture( void );

enum CCDLIB_ErrorEnum CCDLIB_CreateMask(
   double *histoentropy,
   double *contrastrange,
	char **mask,
   CCDLIB_RectangleStruct *checkrect,
	short *picbuffer,
	CCDLIB_PictureparamStruct *picparam,
	double masklevel );

BOOL CCDLIB_EvaluatePicQuality(
	double *sobelmax,
	double *sobelmean,
	double *noiselevel,
	double *sharpindex,
   double *entropy,
	double *contrastval,
   double *signaltonoise,
   CCDLIB_RectangleStruct *checkrect,
	short *picbuffer,
	CCDLIB_PictureparamStruct *picparam,
   CCDLIB_RectangleStruct *evalrect );

void CCDLIB_GetLineStatistics(
	long  *mean,
	double *sigma,
	long  *minimum,
	long  *maximum,
	short  *values,
	long   valnum );

enum CCDLIB_ErrorEnum CCDLIB_CalcArtificialBackground(
   short * background,
	short * picture,
	CCDLIB_PictureparamStruct * pictureparam,
	long polygrad );

enum CCDLIB_ErrorEnum CCDLIB_GetHistoMatrix(
	CCDLIB_RectangleStruct * picturerect,
	CCDLIB_PictureparamStruct * pictureparam,
	short * picture,
	double * xvalues,
	double * yvalues,
	double * zvalues );

BOOL CCDLIB_Regression(
   int grad,
   long n,
	double *xwerte,
   double *ywerte,
   double *koeff );

BOOL CCDLIB_2DRegression(
   int grad,
   long n,
	double *xwerte,
   double *ywerte,
   double *zwerte,
   double *koeff );

double CCDLIB_Eval2DPolynom(
   int grad,
	double x,
   double y,
   double *koeff );

double CCDLIB_Ipow(
   double arg,
   long exp );
   
int CCDLIB_GaussSolve(
   int cas,
   long n,
   double **matrix,
   double **lumat,
   long *perm,
   double *b,
   double *x,
   int *signdet );

int CCDLIB_Gaudec(
   long n,
   double **matrix,
   double **lumat,
   long *perm,
   int *signdet );

int CCDLIB_Gausol(
   long n,
   double **lumat,
   long *perm,
   double *b,
   double *x );

#endif
